# -*- coding: utf-8 -*-
import os
import os.path
import sys
from datetime import datetime

from . import blackbox_client
from . import yenv
from . import utils
import importlib
import json
from itertools import product
from environs import Env
import logging

env = Env()
try:
    env.read_env()
except IOError:
    pass

CURRENT_USER = os.environ.get('USER', os.environ.get('LOGNAME', 'root'))
if CURRENT_USER in ('root', 'www-data'):
    ENVIRONMENT = (os.getenv('ENVIRONMENT') or yenv.type).lower().replace('-', '_')
else:
    ENVIRONMENT = (os.getenv('ENVIRONMENT') or CURRENT_USER).lower().replace('-', '_')

CURRENT_DC = os.environ.get('DEPLOY_NODE_DC', os.environ.get('QLOUD_DATACENTER', 'vla')).lower()

# Вычислим в каком порядке будем загружать конфиги
FALLBACK_CONFIG = 'development'
CONFIGS_TO_TRY = [ENVIRONMENT]

module = ENVIRONMENT
# Далее, откусываем по одному модулю с конца, например если было
# testing.dir-6010, то останется только testing
while '.' in module:
    module = module.rsplit('.', 1)[0]
    CONFIGS_TO_TRY.append(module)

CONFIGS_TO_TRY.append(FALLBACK_CONFIG)

LOADED_CONFIG = None

for prefix, config in product(('src.', ''), CONFIGS_TO_TRY):
    try:
        environment_settings = importlib.import_module(
            prefix + 'intranet.yandex_directory.src.env_settings.' + config
        )
        LOADED_CONFIG = config
        break
    except ImportError:
        pass

if LOADED_CONFIG is None:
    raise RuntimeError('I\'ve tried these configs: {}, but was unabled to load any'.format(
        ', '.join(CONFIGS_TO_TRY)
    ))


def get_setting(name, env_type=None, subcast=None, default=None):
    default = getattr(environment_settings, name, default)

    if env_type:
        value = getattr(env, env_type)(name, default, subcast=subcast)
    else:
        value = env(name, default, subcast=subcast)

    if isinstance(value, str):
        if value.lower() == 'false':
            value = False
        elif value.lower() == 'true':
            value = True
        elif value.lower() == 'none':
            value = None

    return value


# Экспериментальный хак, чтобы больше соответствовать DRY
# Если получится, можно будет отказаться от дублирования параметров
# в settings/base.py
def def_setting(name, env_type=None, subcast=None, default=None):
    globals()[name] = get_setting(
        name,
        env_type=env_type,
        subcast=subcast,
        default=default,
    )


# окружения которые считаются production
# для них не светим подробности ошибки при 500
# и запрещаем recreate таблиц при миграции
PRODUCTION_ENVIRONMENTS = get_setting('PRODUCTION_ENVIRONMENTS', env_type='list')

INTERNAL = get_setting('INTERNAL')
VIEW_CLASSES_WITHOUT_VERSION = get_setting('VIEW_CLASSES_WITHOUT_VERSION', env_type='list')

BASE_PATH = os.path.dirname(os.path.normpath(__file__))
ENABLE_ADMIN_API = get_setting('ENABLE_ADMIN_API')

SITE_BASE_URI = get_setting('SITE_BASE_URI')
PASSPORT_URL_LANDING = get_setting('PASSPORT_URL_LANDING')
PASSPORT_API = get_setting('PASSPORT_API')
TEAM_PASSPORT_API = get_setting('TEAM_PASSPORT_API')

USE_DB = 'POSTGRES'

ROBOT_LOGIN = 'robot'
ROBOT_PASSWORD = 'robot_pass'

DATABASE_CONNECT_ARGS = {
    'sslmode': get_setting('DATABASES_SSL_MODE'),
    'connect_timeout': int(get_setting('DATABASES_CONNECT_TIMEOUT')),
}
DATABASE_POOL_SIZE = get_setting('DATABASE_POOL_SIZE', env_type='int') or 5
DATABASE_MAX_OVERFLOW = get_setting('DATABASE_MAX_OVERFLOW', env_type='int') or 10
DATABASE_POOL_RECYCLE = get_setting('DATABASE_POOL_RECYCLE', env_type='int') or 300

PACKAGE_NAME = get_setting('PACKAGE_NAME')

DATABASES = {
    'main': {},
    'meta': {}
}

MAIN_DB_SHARDS = get_setting('MAIN_DB_SHARDS')
MAIN_DB_DEFAULTS = get_setting('MAIN_DB_DEFAULTS')
META_DB_DEFAULTS = get_setting('META_DB_DEFAULTS')

# Настройки для баз нужно писать в формате MAIN_DB_{shard_num}_{role}_[HOST|PORT|...],
# где shard_num - номер шарда, role - поль базы, master или replica, например:
# MAIN_DB_1_MASTER_PORT и MAIN_DB_1_REPLICA_PORT


def build_db_config_for_database(db_params, alias, shard):
    for role in ['master', 'replica']:
        if shard not in DATABASES[alias]:
            DATABASES[alias][shard] = {}

        def _get(param_name):
            env_var = '{alias}_DB_{shard}_{role}_{name}'.format(
                alias=alias.upper(),
                shard=shard,
                role=role.upper(),
                name=param_name.upper(),
            )
            default = db_params.get(param_name)

            return get_setting(env_var, default=default)

        connect_args = DATABASE_CONNECT_ARGS.copy()
        if role == 'master':
            connect_args['target_session_attrs'] = 'read-write'

        DATABASES[alias][shard][role.lower()] = {
            'host': _get('host'),
            'port': int(_get('port')),
            'database': _get('name'),
            'user': _get('user'),
            'password': _get('password'),
            'connect_args': connect_args,
            'pool_size': DATABASE_POOL_SIZE,
            'pool_max_overflow': DATABASE_MAX_OVERFLOW,
            'pool_recycle': DATABASE_POOL_RECYCLE,
        }


build_db_config_for_database(
    META_DB_DEFAULTS,
    alias='meta',
    shard=1,
)

for idx, shard_params in enumerate(MAIN_DB_SHARDS, 1):
    db_params = MAIN_DB_DEFAULTS.copy()

    # Для совместимости со старым форматом конфига,
    # оставим возможность перечислять в MAIN_DB_SHARDS
    # только имена баз
    if isinstance(shard_params, str):
        db_params['name'] = shard_params
    else:
        # но если там словарик, то используем его,
        # чтобы переписать дефолтные настройки.
        db_params.update(shard_params)

    build_db_config_for_database(
        db_params,
        alias='main',
        shard=idx,
    )


MIGRATE_DATABASE_USER = get_setting('MIGRATE_DATABASE_USER')
MIGRATE_DBS = get_setting('MIGRATE_DBS')

VAR_DATA_BASE_DIR_PATH = get_setting('VAR_DATA_BASE_DIR_PATH')
VAR_DATA_DIR_NAME = get_setting('VAR_DATA_DIR_NAME')
VAR_DATA_PATH = os.path.join(VAR_DATA_BASE_DIR_PATH, VAR_DATA_DIR_NAME)


INTERNAL_SERVICES = get_setting('INTERNAL_SERVICES')


INTERNAL_SERVICES_BY_IDENTITY = dict(
    (item['identity'], item) for item in INTERNAL_SERVICES
)

# По умолчанию, для сервиса нет ни одного токена
for item in INTERNAL_SERVICES:
    if 'tokens' not in list(item.keys()):
        item['tokens'] = []


TOKENS_PATHS = ['/etc/yandex/tokens.json', './tokens.json']

for path in TOKENS_PATHS:
    if os.path.isfile(path):
        with open(path) as json_tokens_file:
            tokens = json.load(json_tokens_file)
            for identity, service_tokens in list(tokens.items()):
                if identity in INTERNAL_SERVICES_BY_IDENTITY:
                    INTERNAL_SERVICES_BY_IDENTITY[identity]['tokens'].extend(service_tokens)


INTERNAL_SERVICES_BY_TOKEN = dict()
for item in INTERNAL_SERVICES:
    for token in item['tokens']:
        INTERNAL_SERVICES_BY_TOKEN[token] = item

# web-chib
USER_DATA_FOR_TEST = {
    'uid': 66865142,
}


# https://tech.yandex.ru/oauth/doc/dg/tasks/get-oauth-token-docpage/
OAUTH_DATA_FOR_TESTS = {
    'access_token': '8cfd3a479de64ca691c0624c1e85f6c7',
    'client': {
        'name': 'Events',
        'id': 'fb8070440d284159991d0799f8b093a2'
    },
    'scope': 'lenta:all yaru:read',
    'uid': USER_DATA_FOR_TEST['uid']
}


BLACKBOX = {
    'url': get_setting('BLACKBOX_URL'),
    'type': 'blackbox',
    'dbfields': [
        blackbox_client.FIELD_LOGIN,
        blackbox_client.FIELD_FIO,
        blackbox_client.FIELD_NICKNAME,
        blackbox_client.FIELD_SEX,
        blackbox_client.FIELD_EMAIL,
        blackbox_client.FIELD_LANGUAGE,
        ('account_info.birth_date.uid', 'birth_date'),
        ('userinfo.firstname.uid', 'first_name'),
        ('userinfo.lastname.uid', 'last_name'),
    ],

}
TEAM_BLACKBOX = {
    'url': get_setting('TEAM_BLACKBOX_URL'),
    'type': 'team_blackbox',
}

WMI_URL = get_setting('WMI_URL')

CONDUCTOR_CLIENT_CLASS = get_setting('CONDUCTOR_CLIENT_CLASS')

PAGINATION = {
    'page_query_param': 'page',
    'pages_query_param': 'pages',
    'total_query_param': 'total',
    'per_page_query_param': 'per_page',
    'per_page': int(get_setting('PER_PAGE_DEFAULT')),
    'max_per_page': int(get_setting('PER_PAGE_MAX')),
    'max_per_page_for_admin': int(get_setting('PER_PAGE_MAX_FOR_ADMIN')),
}

CONDUCTOR_URL = get_setting('CONDUCTOR_URL')

LOG_LEVEL = get_setting('LOG_LEVEL')
DB_LOG_LEVEL = get_setting('DB_LOG_LEVEL')

LOG_STREAM = sys.stdout

if ENVIRONMENT == 'autotests' and not os.environ.get('ARCADIA'):
    LOG_STREAM = open(
        os.path.join(
            os.path.dirname(
                os.path.dirname(
                    os.path.dirname(__file__))),
            'autotests.log'
        ),
        'a'
    )
else:
    # если проект запущен в терминале, руками, то лог будем писать в yandex-directory.log
    if sys.stdout.isatty():
        LOG_STREAM = open('yandex-directory.log', 'a')

if os.environ.get('DEPLOY_STAGE_ID', os.environ.get('QLOUD_ENVIRONMENT')) is not None:
    LOG_FORMATTER = 'json'
    LOGGERS = {
        'dir_access': {
            'level': logging.DEBUG,
            'handlers': ['dir_access', 'stream'],
        },
        'dir_default': {
            'level': logging.DEBUG,
            'handlers': ['dir_default', 'stream'],
        },
        'dir_error': {
            'level': logging.DEBUG,
            'handlers': ['dir_error', 'stream'],
        },
        'dir_requests': {
            'level': logging.DEBUG,
            'handlers': ['dir_requests'],
        },
    }
    LOG_FORMATTERS = {
        'dir_access_tskv': {
            '()': 'intranet.yandex_directory.src.yandex_directory.directory_logging.logger.TSKVFormatter',
            'fmt': "tskv\ttskv_format=ydisk-directory-access-log\thost=%(host_name)s\tname=%(name)s\tappname=%(appname)s\tunixtime=%(created)d\ttimestamp=%(asctime)s\ttimezone=+0300\tycrid=%(ycrid)s\trequest_id=%(request_id)s\tpid=%(process)s\tmodule=%(module)s\theaders=%(headers)s\tmethod=%(method)s\tuid=%(uid)s\torg_id=%(org_id)s\turi=%(uri)s\tproto=%(proto)s\tstatus=%(status)s\trequest_time=%(request_time).3f"
        },
        'dir_default_tskv': {
            '()': 'intranet.yandex_directory.src.yandex_directory.directory_logging.logger.TSKVFormatter',
            'fmt': "tskv\ttskv_format=ydisk-directory-default-log\thost=%(host_name)s\tname=%(name)s\tappname=%(appname)s\tunixtime=%(created)d\ttimestamp=%(asctime)s\ttimezone=+0300\tycrid=%(ycrid)s\trequest_id=%(request_id)s\tpid=%(process)s\tmodule=%(module)s\tlevel=%(levelname)s\tmessage=%(message)s\tcontext=%(context)s"
        },
        'dir_error_tskv': {
            '()': 'intranet.yandex_directory.src.yandex_directory.directory_logging.logger.TSKVFormatter',
            'fmt': "tskv\ttskv_format=ydisk-directory-error-log\thost=%(host_name)s\tname=%(name)s\tappname=%(appname)s\tunixtime=%(created)d\ttimestamp=%(asctime)s\ttimezone=+0300\tycrid=%(ycrid)s\trequest_id=%(request_id)s\tpid=%(process)s\tmodule=%(module)s\tlevel=%(levelname)s\tmessage=%(message)s\tcontext=%(context)s"
        },
        'dir_requests_tskv': {
            '()': 'intranet.yandex_directory.src.yandex_directory.directory_logging.logger.TSKVFormatter',
            'fmt': "tskv\ttskv_format=ydisk-directory-requests-log\thost=%(host_name)s\tname=%(name)s\tappname=%(appname)s\tunixtime=%(created)d\ttimestamp=%(asctime)s\ttimezone=+0300\tycrid=%(ycrid)s\trequest_id=%(request_id)s\tpid=%(process)s\tmodule=%(module)s\tmessage=%(message)s\trequest_time=%(request_time).3f"
        },
    }
    LOG_HANDLERS = {
        'dir_access': {
            'class': 'logging.handlers.WatchedFileHandler',
            'formatter': 'dir_access_tskv',
            'filename': '/var/log/yandex/directory/access-tskv.log',
            'level': 'DEBUG',
        },
        'dir_default': {
            'class': 'logging.handlers.WatchedFileHandler',
            'formatter': 'dir_default_tskv',
            'level': 'DEBUG',
            'filename': '/var/log/yandex/directory/default-tskv.log',
        },
        'dir_error': {
            'class': 'logging.handlers.WatchedFileHandler',
            'formatter': 'dir_error_tskv',
            'level': 'DEBUG',
            'filename': '/var/log/yandex/directory/error-tskv.log',
        },
        'dir_requests': {
            'class': 'logging.handlers.WatchedFileHandler',
            'formatter': 'dir_requests_tskv',
            'level': 'DEBUG',
            'filename': '/var/log/yandex/directory/requests-tskv.log',
        },
    }
else:
    LOG_FORMATTER = 'default'
    LOGGERS = {
        'dir_access': {
            'level': logging.DEBUG,
            'handlers': ['stream'],
        },
        'dir_default': {
            'level': logging.DEBUG,
            'handlers': ['stream'],
        },
        'dir_error': {
            'level': logging.DEBUG,
            'handlers': ['stream'],
        },
        'dir_requests': {
            'level': logging.DEBUG,
            'handlers': [],
        },
    }
    LOG_FORMATTERS = {}
    LOG_HANDLERS = {}

LOGGING = {
    'version': 1,
    'disable_existing_loggers': True,

    'formatters': {
        'default': {
            '()': 'ylog.context.ContextFormatter',
            'datefmt': '%Y-%m-%d %H:%M:%S',
            'fmt': '%(asctime)s %(threadName)-10s "%(name)-10s" %(levelname)-8s %(message)s ',
        },
        'json': {
            '()': 'ylog.format.QloudJsonFormatter',
        },
        **LOG_FORMATTERS,
    },
    'loggers': {
        'tools_structured_logs': {
            'level': LOG_LEVEL,
            'propagate': False,
            'handlers': ['stream'],
        },
        'tvm2.thread_tvm2': {
            'level': logging.DEBUG,
            'propagate': False,
            'handlers': [],
        },
        **LOGGERS,
    },
    'handlers': {
        'stream': {
            'class': 'logging.StreamHandler',
            'formatter': LOG_FORMATTER,
            'stream': LOG_STREAM,
            'level': LOG_LEVEL,
        },
        'null': {
            'level': 'DEBUG',
            'class': 'logging.NullHandler',
        },
        'golovan_level_counter': {
            'class': 'intranet.yandex_directory.src.yandex_directory.core.monitoring.handlers.GolovanErrorsCountHandler',
            'level': 'WARNING',
        },
        **LOG_HANDLERS,
    },
    'root': {
        'level': LOG_LEVEL,
        'formatter': LOG_FORMATTER,
        'handlers': [
            'golovan_level_counter',
        ]
    },
}

try:
    PORT = int(get_setting('PORT'))
except (AttributeError, ValueError, TypeError):
    PORT = None

X_DEBUG = get_setting('X_DEBUG')
LOG_SQL_QUERIES = get_setting('LOG_SQL_QUERIES', default=True)

DEFAULT_NOTIFICATION_TIMEOUT = 5

SUBSCRIPTIONS = get_setting('SUBSCRIPTIONS')

# http://flask.pocoo.org/docs/0.10/api/#flask.Flask.testing
TESTING = get_setting('TESTING')
YANDEX_ALL_CA = get_setting('YANDEX_ALL_CA')

STAFF_API_URL = get_setting('STAFF_API_URL')


TEMPLATES_DIR = get_setting('TEMPLATES_DIR')

PORTAL_TLDS = get_setting('PORTAL_TLDS').split(',')
DEFAULT_TLD = get_setting('DEFAULT_TLD')


YANDEX_TEAM_ORG_DOMAINS = get_setting('YANDEX_TEAM_ORG_DOMAINS')

DISK_BASE_URL = get_setting('DISK_BASE_URL')
DISK_NO_EMAIL = get_setting('DISK_NO_EMAIL')

EVENTS_LOG_SIZE_DAYS = int(get_setting('EVENTS_LOG_SIZE_DAYS'))

# домен верхнего уровня для создаваемых организаций
# в итоге  <my-company-label>.domain.part.ru
DOMAIN_PART = get_setting('DOMAIN_PART')


MAIL_SEND_FUNCTION = get_setting('MAIL_SEND_FUNCTION')

# размер задержки перед отправкой письма (если неообходима задежка)
# ждем когда почтовый ящик доиниализируется
MAIL_SEND_DELAY = int(get_setting('MAIL_SEND_DELAY'))

# Изменять класс аггрегатора нужно потому, что в продакшене нам нужен бекэнд кэша uwsgi,
# который использует общую память с другими воркерами на этой машине.
# А для разработки мы uwsgi не запускаем, поэтому используем кэш в памяти.
GOLOVAN_STATS_AGGREGATOR_CLASS = get_setting('GOLOVAN_STATS_AGGREGATOR_CLASS')
# таймаут на выполнение функций, собирающих метрики для голована
GOLOVAN_STATS_AGGREGATOR_PERIOD = get_setting('GOLOVAN_STATS_AGGREGATOR_PERIOD', subcast=float)
GOLOVAN_STATS_AGGREGATOR_FUNCTION_TIMEOUT = get_setting('GOLOVAN_STATS_AGGREGATOR_FUNCTION_TIMEOUT', subcast=float)

SYNC_WITH_STAFF_ENABLED = int(get_setting('SYNC_WITH_STAFF_ENABLED'))

# отправляем в голован количество статусов миграций, которые висят дольше определенного времени
MONITOR_BLOCKED_DOMAINS_LONGER_THAN_MIN = int(get_setting('MONITOR_BLOCKED_DOMAINS_LONGER_THAN_MIN'))

# время жизни кода приглашения в часах
INVITES_HOURS_LIFETIME = int(get_setting('INVITES_HOURS_LIFETIME'))

# Ограничение на число активных инвайтов для одной организации
def_setting('MAX_INVITES_COUNT', default=100, env_type='int')
# Ограничение на число инвайтов когда либо созданных одной организацией
def_setting('MAX_TOTAL_INVITES_COUNT', default=1000, env_type='int')

# записываем в лог долгие запросы к базе (bool)
LOG_SLOW_SQL_QUERIES = get_setting('LOG_SLOW_SQL_QUERIES')

# раз в столько секунд, мы пингуем и переустанавливаем "больные" коннекты к базе
PING_DB_CONNECTIONS_INTERVAL = int(get_setting('PING_DB_CONNECTIONS_INTERVAL'))

# TTL для кэша подписок на события
WEBHOOKS_CACHE_TTL = int(get_setting('WEBHOOKS_CACHE_TTL'))
# Размер кэша подписок на события
WEBHOOKS_CACHE_SIZE = int(get_setting('WEBHOOKS_CACHE_SIZE'))

# возможнные языки в организации
ORGANIZATION_LANGUAGES = get_setting('ORGANIZATION_LANGUAGES').split(",")

PORTAL_URL_TEMPLATE = get_setting('PORTAL_URL_TEMPLATE')
PORTAL_URL = PORTAL_URL_TEMPLATE.format(tld='ru')

OAUTH_BASE_URL = get_setting('OAUTH_BASE_URL')

# поддерживаемые версии API
SUPPORTED_API_VERSIONS = get_setting('SUPPORTED_API_VERSIONS', env_type='list', subcast=int)
SUPPORTED_API_VERSIONS.sort()

TVM_URL = get_setting('TVM_URL')

# настройки для TVM 2.0
TVM_CLIENT_ID = get_setting('TVM_CLIENT_ID')
TVM_SECRET = get_setting('TVM_SECRET')
TEAM_TVM_CLIENT_ID = get_setting('TEAM_TVM_CLIENT_ID')
TEAM_TVM_SECRET = get_setting('TEAM_TVM_SECRET')
BLACKBOX_TVM_CLIENT_ID = get_setting('BLACKBOX_TVM_CLIENT_ID')
TEAM_BLACKBOX_TVM_CLIENT_ID = get_setting('TEAM_BLACKBOX_TVM_CLIENT_ID')
DISK_TVM_CLIENT_ID = get_setting('DISK_TVM_CLIENT_ID')
BIGML_TVM_CLIENT_ID = get_setting('BIGML_TVM_CLIENT_ID')
FORMS_TVM_CLIENT_ID = get_setting('FORMS_TVM_CLIENT_ID')
ZORA_TVM_CLIENT_ID = get_setting('ZORA_TVM_CLIENT_ID')
METRIKA_TVM_CLIENT_ID = get_setting('METRIKA_TVM_CLIENT_ID')
DOMENATOR_TVM_CLIENT_ID = get_setting('DOMENATOR_TVM_CLIENT_ID')
DIRECT_TVM_CLIENT_ID = get_setting('DIRECT_TVM_CLIENT_ID')
ALICE_B2B_TVM_CLIENT_ID = get_setting('ALICE_B2B_TVM_CLIENT_ID')
DIRECTORY_TVM_CLIENT_ID = get_setting('DIRECTORY_TVM_CLIENT_ID')


SERVICE_OAUTH_DATA_FILE = get_setting('SERVICE_OAUTH_DATA_FILE')

# закрывать сервис от балансера отдачей 503 ошибок после
# определенного количества ошибок (BACKPRESSURE_CLOSE_AFTER_ERRORS_COUNT) в жизненно-важных подсистемах в смоук-тестах
# за определенное количество попыток (BACKPRESSURE_CHECK_WINDOW_SIZE)
BACKPRESSURE_CLOSE_AFTER_ERRORS_COUNT = int(get_setting('BACKPRESSURE_CLOSE_AFTER_ERRORS_COUNT'))
BACKPRESSURE_CHECK_WINDOW_SIZE = int(get_setting('BACKPRESSURE_CHECK_WINDOW_SIZE'))

BILLING_API_URL = get_setting('BILLING_API_URL')
BILLING_TVM2_CLIENT_ID = get_setting('BILLING_TVM2_CLIENT_ID')
BILLING_API_TOKEN = get_setting('BILLING_API_TOKEN')
BILLING_MANAGER_UID = get_setting('BILLING_MANAGER_UID')
BILLING_YT_TABLES_PATH = get_setting('BILLING_YT_TABLES_PATH')
BILLING_YT_TOKEN = get_setting('BILLING_YT_TOKEN')
BILLING_YT_CLUSTERS = get_setting('BILLING_YT_CLUSTERS', env_type='list')
BILLING_SAVING_ENABLED = get_setting('BILLING_SAVING_ENABLED', env_type='bool')
BILLING_CLIENT_CLASS = get_setting('BILLING_CLIENT_CLASS')
BILLING_PRODUCT_ID_RUB = get_setting('BILLING_PRODUCT_ID_RUB')
BILLING_INVOICE_API_URL = get_setting('BILLING_INVOICE_API_URL')
BILLING_INVOICE_API_TVM2_CLIENT_ID = get_setting('BILLING_INVOICE_API_TVM2_CLIENT_ID')
BILLING_CONNECT_ONE_PAID_PRODUCT_PRICE = get_setting('BILLING_CONNECT_ONE_PAID_PRODUCT_PRICE')
BILLING_CONNECT_TWO_PAID_PRODUCT_PRICE = get_setting('BILLING_CONNECT_TWO_PAID_PRODUCT_PRICE')
BILLING_CONNECT_PRODUCT_ID = get_setting('BILLING_CONNECT_PRODUCT_ID')
BILLING_CONNECT_ONE_PRODUCT_ID = get_setting('BILLING_CONNECT_ONE_PRODUCT_ID')
BILLING_CONNECT_TWO_PRODUCT_ID = get_setting('BILLING_CONNECT_TWO_PRODUCT_ID')
BILLING_TRACKER_TEN_PRODUCT_ID = get_setting('BILLING_TRACKER_TEN_PRODUCT_ID')
BILLING_TRACKER_HUNDRED_PRODUCT_ID = get_setting('BILLING_TRACKER_HUNDRED_PRODUCT_ID')
BILLING_TRACKER_FIVE_HUNDRED_PRODUCT_ID = get_setting('BILLING_TRACKER_FIVE_HUNDRED_PRODUCT_ID')
BILLING_TRACKER_TWO_THOUSAND_PRODUCT_ID = get_setting('BILLING_TRACKER_TWO_THOUSAND_PRODUCT_ID')
BILLING_CLIENT_UID_FOR_TESTING = get_setting('BILLING_CLIENT_UID_FOR_TESTING')
BILLING_TIMEOUT = get_setting('BILLING_TIMEOUT', env_type='float', default=3)

BILLING_PARTNER_DISK_PRODUCT_ID = get_setting('BILLING_PARTNER_DISK_PRODUCT_ID')
# product_id для бесплатного доступа к сервису
PRODUCT_ID_FREE = 'free'

TRACKER_PRODUCT_ID_1 = 511591
TRACKER_PRODUCT_ID_100 = 511592
TRACKER_PRODUCT_ID_250 = 511593

BILLING_PRODUCT_IDS_FOR_TRACKER = utils.RangeDict({
    range(1, 101): TRACKER_PRODUCT_ID_1,
    range(101, 251): TRACKER_PRODUCT_ID_100,
    range(251, 9999999999): TRACKER_PRODUCT_ID_250,
})

BILLING_PRODUCT_IDS_FOR_SERVICES = {
    'connect': {
        1: BILLING_CONNECT_ONE_PRODUCT_ID,
        2: BILLING_CONNECT_TWO_PRODUCT_ID,
        3: BILLING_CONNECT_PRODUCT_ID,
    },
    'tracker': {
        5: PRODUCT_ID_FREE,
        # Временно сделали трекер бесплатным DIR-8844
        # нужно убрать этот коммит целиком к концу сентября
        10: PRODUCT_ID_FREE,  # BILLING_TRACKER_TEN_PRODUCT_ID,
        100: PRODUCT_ID_FREE,  # BILLING_TRACKER_HUNDRED_PRODUCT_ID,
        500: BILLING_TRACKER_FIVE_HUNDRED_PRODUCT_ID,
        2000: BILLING_TRACKER_TWO_THOUSAND_PRODUCT_ID,
    },
    'disk': {
        50000000: BILLING_PARTNER_DISK_PRODUCT_ID,
    },
}
TRACKER_FREE_LICENSES = 5
BILLING_PAYMENT_TERM = int(get_setting('BILLING_PAYMENT_TERM'))

# отправляем платным сервисам письма за определенное количество дней до окончания триала
# числа внутри - количество дней для рассылки письма
MAIL_IDS_BEFORE_TRIAL_END = {
    'tracker': {
        7: get_setting('TRACKER_7_DAYS_BEFORE_TRIAL_END_MAIL_ID'),
        3: get_setting('TRACKER_3_DAYS_BEFORE_TRIAL_END_MAIL_ID'),
        1: get_setting('TRACKER_1_DAYS_BEFORE_TRIAL_END_MAIL_ID'),
    }
}

# настройки для складывания данных в yt
YT_CLUSTER = get_setting('YT_CLUSTER')
YT_TOKEN = get_setting('YT_TOKEN')

YT_ANALYTICS_TABLES_PATH = get_setting('YT_ANALYTICS_TABLES_PATH')
YT_ANALYTICS_SECOND_TABLES_PATH = get_setting('YT_ANALYTICS_SECOND_TABLES_PATH')  # временно для тестирования очереди
YT_ANALYTICS_SAVING_ENABLED = get_setting('YT_ANALYTICS_SAVING_ENABLED')

# общие настройки походов в YT
# работают и для биллинговых YT клиентов и для оставльных
YT_CONNECT_TIMEOUT = get_setting('YT_CONNECT_TIMEOUT')  # миллисекунды
YT_REQUEST_TIMEOUT = get_setting('YT_REQUEST_TIMEOUT')  # миллисекунды

# частота опроса задач воркерров при разгребании очереди
TASK_POLLING_INTERVAL = float(get_setting('TASK_POLLING_INTERVAL'))
# максимальная задержка для опроса воркеров при разгребании очереди
TASK_POLLING_MAX_INTERVAL = float(get_setting('TASK_POLLING_MAX_INTERVAL'))

# парметры для рассылятора https://sender.yandex-team.ru/
SENDER_BASE_URL = get_setting('SENDER_BASE_URL')
SENDER_ACCOUNT_SLUG = get_setting('SENDER_ACCOUNT_SLUG')
SENDER_USER = get_setting('SENDER_USER')
SENDER_CAMPAIGN_SLUG = {
    'WELCOME_EMAIL': get_setting('SENDER_CAMPAIGN_SLUG_WELCOME_EMAIL'),
    'AFTER_CREATE_ORGANIZATION_EMAIL': get_setting('SENDER_CAMPAIGN_SLUG_AFTER_CREATE_ORGANIZATION_EMAIL'),
    'TRACKER_ACTIVATED_EMAIL': get_setting('SENDER_CAMPAIGN_SLUG_TRACKER_ACTIVATED_EMAIL'),
    'TRACKER_SUBSCRIPTION_ACTIVATED_EMAIL': get_setting('SENDER_CAMPAIGN_SLUG_TRACKER_SUBSCRIPTION_ACTIVATED_EMAIL'),
    'TRACKER_DISABLE_NOTIFICATION_1': get_setting('SENDER_CAMPAIGN_SLUG_TRACKER_DISABLE_NOTIFICATION_1'),
    'TRACKER_DISABLE_NOTIFICATION_2': get_setting('SENDER_CAMPAIGN_SLUG_TRACKER_DISABLE_NOTIFICATION_2'),
    'TRACKER_DISABLE_NOTIFICATION_3': get_setting('SENDER_CAMPAIGN_SLUG_TRACKER_DISABLE_NOTIFICATION_3'),
    'PAID_MODE_ENABLE_EMAIL': get_setting('SENDER_CAMPAIGN_SLUG_PAID_MODE_ENABLE_EMAIL'),
    'CHANGE_PASSWORD_EMAIL': get_setting('SENDER_CAMPAIGN_SLUG_CHANGE_PASSWORD_EMAIL'),
    'DISABLE_DOMAIN_EMAIL': get_setting('SENDER_CAMPAIGN_SLUG_DISABLE_DOMAIN_EMAIL'),
    'INVITE_USER': get_setting('SENDER_CAMPAIGN_SLUG_INVITE_USER'),
    'ACCESS_RESTORE_TRY_RESTORE': get_setting('SENDER_CAMPAIGN_SLUG_ACCESS_RESTORE_TRY_RESTORE'),
    'ACCESS_RESTORE_SOME_RESTORE': get_setting('SENDER_CAMPAIGN_SLUG_ACCESS_RESTORE_SOME_RESTORE'),
    'ACCESS_RESTORE_SUCCESS_RESTORE': get_setting('SENDER_CAMPAIGN_SLUG_ACCESS_RESTORE_SUCCESS_RESTORE'),
    'CONNECT_DEBT_EMAILS': {
        14:  get_setting('SENDER_CAMPAIGN_SLUG_14_DAYS_DEBT_EMAIL'),
        21:  get_setting('SENDER_CAMPAIGN_SLUG_21_DAYS_DEBT_EMAIL'),
        28:  get_setting('SENDER_CAMPAIGN_SLUG_28_DAYS_DEBT_EMAIL'),
        29:  get_setting('SENDER_CAMPAIGN_SLUG_29_DAYS_DEBT_EMAIL'),
        30:  get_setting('SENDER_CAMPAIGN_SLUG_30_DAYS_EMAIL'),
        31:  get_setting('SENDER_CAMPAIGN_SLUG_31_DAYS_EMAIL'),
    },
    'REQUEST_ACCESS_TO_RESOURCE': get_setting('SENDER_CAMPAIGN_SLUG_REQUEST_ACCESS_TO_RESOURCE'),
    'CONFIRM_ACCESS_TO_RESOURCE': get_setting('SENDER_CAMPAIGN_SLUG_CONFIRM_ACCESS_TO_RESOURCE'),
    'DENY_ACCESS_TO_RESOURCE': get_setting('SENDER_CAMPAIGN_SLUG_DENY_ACCESS_TO_RESOURCE'),
    'METRIKA_RESOURCE_UNBIND': get_setting('SENDER_CAMPAIGN_METRIKA_RESOURCE_UNBIND')
}


CLOUD_NOTIFY_SLUG = {
    'TRACKER_ACTIVATED_EMAIL': 'tracker.access.welcome-admin',
    'TRACKER_SUBSCRIPTION_ACTIVATED_EMAIL': 'tracker.access.welcome-admin',
    'TRACKER_DISABLE_NOTIFICATION_1': 'tracker.org.suspend-1-month',
    'TRACKER_DISABLE_NOTIFICATION_2': 'tracker.org.suspend-7-days',
    'TRACKER_DISABLE_NOTIFICATION_3': 'tracker.org.suspend',
    'expiration_1d': 'tracker.trial.expire-soon-1-day',
    'expiration_7d': 'tracker.trial.expire-soon-7-days',
    'expiration_3d': '__disabled__',
}

# маппинг кампании в сендере на шаблон в cloud notify
SENDER_TO_CLOUD_NOTIFY_MAPPING = {
    sender_slug: CLOUD_NOTIFY_SLUG[key]
    for key, sender_slug in SENDER_CAMPAIGN_SLUG.items()
    if key in CLOUD_NOTIFY_SLUG
}
SENDER_TO_CLOUD_NOTIFY_MAPPING.update({
    sender_slug: CLOUD_NOTIFY_SLUG[f'expiration_{day}d']
    for day, sender_slug in MAIL_IDS_BEFORE_TRIAL_END['tracker'].items()
})

YASMS_SENDER = get_setting('YASMS_SENDER')
YASMS_BASE_URL = get_setting('YASMS_BASE_URL')
SMS_SEND_FUNCTION = get_setting('SMS_SEND_FUNCTION')
SMS_NOTIFY_TRACKER_HOLD_TIME = int(get_setting('SMS_NOTIFY_TRACKER_HOLD_TIME'))

# По умолчанию - 10G на пользователя
PER_HUMAN_DISK_LIMIT = int(get_setting('PER_HUMAN_DISK_LIMIT'))
# Лимит места для организаций с <= 10 пользователей
TEN_HUMAN_DISK_LIMIT = int(get_setting('TEN_HUMAN_DISK_LIMIT'))

# список id организаций доступных для yamb-test
YAMB_TEST_ORGANIZATIONS = list(map(int, get_setting('YAMB_TEST_ORGANIZATIONS').split(','))) if \
    get_setting('YAMB_TEST_ORGANIZATIONS') else []

# Данные для аватарницы, в которой лежат лого организаций.
MDS_READ_API = get_setting('MDS_READ_API')
MDS_WRITE_API = get_setting('MDS_WRITE_API')
MDS_NAMESPACE = get_setting('MDS_NAMESPACE')

# Даные для работы с s3 api mds
MDS_S3_API_URL = get_setting('MDS_S3_API_URL')

ROBOT_ACCOUNT_NICKNAME_PREFIX = get_setting('ROBOT_ACCOUNT_NICKNAME_PREFIX')

# для ссылок на staff
STAFF_YANDEX_TEAM_URL = get_setting('STAFF_YANDEX_TEAM_URL')
STAFF_URL = get_setting('STAFF_URL')
STAFF_TOKEN = get_setting('STAFF_TOKEN')

PLAYGROUND_OAUTH_CONFIG = get_setting('PLAYGROUND_OAUTH_CONFIG')

WEBMASTER_URL = get_setting('WEBMASTER_URL')
WEBMASTER_CLIENT_ID = get_setting('WEBMASTER_CLIENT_ID')

LOCK_DNS_DELEGATION_DURATION = get_setting('LOCK_DNS_DELEGATION_DURATION')

# Настройка для включения кэширования
CACHE_ENABLED = get_setting('CACHE_ENABLED', env_type='bool')
REDIS_HOSTS = get_setting('REDIS_HOSTS')
REDIS_DB_NAME = get_setting('REDIS_DB_NAME')
REDIS_PASSWORD = get_setting('REDIS_PASSWORD')
REDIS_SOCKET_TIMEOUT = get_setting('REDIS_SOCKET_TIMEOUT')

READ_ONLY_MODE = get_setting('READ_ONLY_MODE', env_type='bool')

# url сборщика почты
YARM_URL = get_setting('YARM_URL')
YANDEX_MAIL_SERVERS = get_setting('YANDEX_MAIL_SERVERS', env_type='list')

STEP_API = get_setting('STEP_API')
STEP_TOKEN = get_setting('STEP_TOKEN')

# включение на компоненте подсчета общих для окружения показателей
COMMON_METRIC_CALC_ENABLED = get_setting('COMMON_METRIC_CALC_ENABLED', env_type='bool')

# сколько ждем зарастания дыр событий при оповещении подписчиков (сек)
EVENT_HOLE_WAIT_TTL = get_setting('EVENT_HOLE_WAIT_TTL', env_type='int')

# url IDM
IDM_API = get_setting('IDM_API')
# OAuth токен для api IDM
IDM_TOKEN = get_setting('IDM_TOKEN')
# ttl кэша для IDM
IDM_CACHE_TTL = get_setting('IDM_CACHE_TTL',  env_type='int')
# slug админки
CONNECT_ADMIN_SLUG = get_setting('CONNECT_ADMIN_SLUG')

SCHEDULER_MODE = get_setting('SCHEDULER_MODE', env_type='bool')

SCHEDULER_HARAKIRI_INTERVAL_MIN = get_setting('SCHEDULER_HARAKIRI_INTERVAL', default=2*60*60, env_type='int')
SCHEDULER_HARAKIRI_INTERVAL_MAX = get_setting('SCHEDULER_HARAKIRI_INTERVAL', default=3*60*60, env_type='int')
SCHEDULER_HARAKIRI_INTERVAL_TIMEOUT = get_setting('SCHEDULER_HARAKIRI_INTERVAL', default=1*60*60, env_type='int')
SHORT_SCHEDULER_HARAKIRI_INTERVAL_MIN = get_setting('SHORT_SCHEDULER_HARAKIRI_INTERVAL', default=5*60, env_type='int')
SHORT_SCHEDULER_HARAKIRI_INTERVAL_MAX = get_setting('SHORT_SCHEDULER_HARAKIRI_INTERVAL', default=10*60, env_type='int')
SHORT_SCHEDULER_HARAKIRI_INTERVAL_TIMEOUT = get_setting('SHORT_SCHEDULER_HARAKIRI_INTERVAL', default=2*60, env_type='int')

DEFAULT_QUEUE_NAME = get_setting('DEFAULT_QUEUE_NAME')


# размер пачки для разгребания событий
NOTIFY_BATCH_SIZE = get_setting('NOTIFY_BATCH_SIZE',  env_type='int')
# в такое количество потоков мы будем разгребать очередь новых ревизий
NOTIFY_THREAD_POOL_SIZE = get_setting('NOTIFY_THREAD_POOL_SIZE',  env_type='int')

MIGRATE_DEPUTY_ADMINS = get_setting('MIGRATE_DEPUTY_ADMINS',  env_type='bool')


PROMOCODE_FOR_EDUCATION_ORG = get_setting('PROMOCODE_FOR_EDUCATION_ORG')

PROMOCODE_FOR_YANDEX_PROJECT_ORG = get_setting('PROMOCODE_FOR_YANDEX_PROJECT_ORG')

BIGML_API = get_setting('BIGML_API')

# url сервиса настроек
SETTINGS_WS_URL = get_setting('SETTINGS_WS_URL')
SETTINGS_WS_TOKEN = get_setting('SETTINGS_WS_TOKEN')

# DIR-5335 теперь для различных типов организации будут различные шарды
SHARDS_FOR_PORTAL = get_setting('SHARDS_FOR_PORTAL', env_type='list', subcast=int)

# DIR-5586 теперь при выборе шарда нам нужна константа, чтобы выбирать шард
WEIGHT_COEF_FOR_SHARDS = get_setting('WEIGHT_COEF_FOR_SHARDS', env_type='int')

SECRET_PLACEHOLDER = get_setting('SECRET_PLACEHOLDER')
SENSITIVE_KEYS = get_setting('SENSITIVE_KEYS')

# DIR-5821 шаблон ссылки приглашения в организацию
INVITE_LINK_TEMPLATE = PORTAL_URL_TEMPLATE + 'invites/?code={code}'

DEFAULT_INVITE_LIMIT = get_setting('DEFAULT_INVITE_LIMIT', env_type='int')

VIP_MANY_USERS_COUNT = get_setting('VIP_MANY_USERS_COUNT', env_type='int')

# когда происходят технические изменения в базе за которыми стоит
# директория - нужно использовать этот сервис.
DIRECTORY_SERVICE_NAME = 'directory'
SHARDS_WITH_WEIGHT_CACHE_TTL = get_setting('SHARDS_WITH_WEIGHT_CACHE_TTL',  env_type='int')

FORMS_HOST = get_setting('FORMS_HOST')
ZORA_HOST = get_setting('ZORA_HOST')
DOMENATOR_HOST = get_setting('DOMENATOR_HOST')
METRIKA_HOST = get_setting('METRIKA_HOST')
YANDEXSPRAV_HOST = get_setting('YANDEXSPRAV_HOST')
DIRECT_HOST = get_setting('DIRECT_HOST')
ALICE_B2B_HOST = get_setting('ALICE_B2B_HOST')
METRIKA_SERVICE_SLUG = 'metrika'

CLOUD_ORGANIZATION_TYPE = 'cloud'
CLOUD_PRESETS = ['cloud', 'cloud-tracker']

def_setting('ORG_IDS_INCONSISTENCIES_METRIC_NAME', default='org_ids_inconsistencies_axxx')
def_setting('ORG_IDS_INCONSISTENCIES_CACHE_KEY', default='org_ids_inconsitencies')
# Кэшируем на 20 минут, а подновлять будем раз в 15
def_setting('ORG_IDS_INCONSISTENCIES_CACHE_TTL', default=20 * 60, env_type='int')
# Время, в течении которого мы даём фоновым таскам отработать.
# В течении этого времени, спустя момент найма или увольнения,
# допустим рассинхрон org_id в Директории и Паспорте.
#
# Для начала, сделаем тут 1 минуту, чтобы посмотреть, много ли будет проблем.
# Но надо стремиться к тому, чтобы тут можно было установить 5 секунд.
def_setting('ORG_IDS_INCONSISTENCIES_DELAY', default=1 * 60, env_type='int')
# Размер окна (в секундах), в котором мы выбираем учётные записи для сверки.
def_setting('ORG_IDS_INCONSISTENCIES_WINDOW', default=15 * 60, env_type='int')
# Интервал с которым надо обновлять значение.
# Этот параметр должен быть меньше, чем ORG_IDS_INCONSISTENCIES_CACHE_TTL,
# чтобы на графике не образовывались "дыры".
# Если сделать интервал меньше  "окна", то график будет более плавным, но
# вырастет нагрузка на базу и blackbox.
def_setting('ORG_IDS_INCONSISTENCIES_FREQUENCY', default=15 * 60, env_type='int')

# Количество секунд, которое мы будем ждать по-умолчанию, после принятия пользователем инвайта
def_setting('DEFAULT_WAIT_BEFORE_REDIRECT', default=0, env_type='int')

# Домены с этим суффиксом будут подтверждаться автоматически
def_setting('AUTO_APPROVE_ZONE', default='.auto.connect-tk.tk')


MAX_LABEL_LENGTH = get_setting('MAX_LABEL_LENGTH', env_type='int', default=100)
MAX_LABEL_CLOUD_LENGTH = get_setting('MAX_LABEL_CLOUD_LENGTH', env_type='int', default=1000)

ORG_LIMITS = {
    'user_aliases_max_count': 0,
    'users_limit': 1000,
}

for name in ORG_LIMITS:
    value = get_setting(
        name='ORG_LIMIT_' + name.upper(),
        env_type='int',
    )

    if value is None:
        raise ValueError('No default value provided for ORG_LIMIT_%s' % name)
    else:
        ORG_LIMITS[name] = value

EMAIL_FOR_TESTS = 'connect-test-mail@yandex-team.ru'

# tld домены для Яндекс.РКУБ (Россия, Казахстан, Украина, Беларусь)
RKUB_TLDS = ['ru', 'kz', 'ua', 'by']
SUPPORT_TLDS = ['com'] + RKUB_TLDS

REGISTRAR_GEN_TOKEN_SECRET = get_setting('REGISTRAR_GEN_TOKEN_SECRET')
REGISTRAR_STORE_TOKEN_SECRET = get_setting('REGISTRAR_STORE_TOKEN_SECRET')
REGISTRAR_STORE_PASSWORD_SECRET = get_setting('REGISTRAR_STORE_PASSWORD_SECRET')

def_setting('GENDARME_URL', default='https://test.gendarme.mail.yandex.net')
def_setting('GENDARME_TVM_CLIENT_ID', default=2001650, env_type='int')

def_setting('FOURAS_HOST', default='https://test.fouras.mail.yandex.net')
def_setting('FOURAS_TVM_CLIENT_ID', default=2010950, env_type='int')

def_setting('SSO_CONFIG_HOST', default='https://federal-cfg-api-test.passport.yandex.net')
def_setting('SSO_CONFIG_TVM_CLIENT_ID', default=2032888, env_type='int')
def_setting('SSO_CONFIG_NAMESPACE', default='360')

def_setting('IDM_B2B_HOST', default='https://idm-api.test.yandex.ru')
def_setting('IDM_B2B_TVM_CLIENT_ID', default=2014382, env_type='int')

BIND_ACCESS_SKIP_CHECK = get_setting('BIND_ACCESS_SKIP_CHECK')
CHECK_RESOURCE_EXISTENCE_ENABLED = get_setting('CHECK_RESOURCE_EXISTENCE_ENABLED')
YANDEXSPRAV_TVM_CLIENT_ID = get_setting('YANDEXSPRAV_TVM_CLIENT_ID')

def_setting('COPY_ANALYTICS_TO_YT', default=False)

DIRECT_SERVICE_SLUG = 'direct'
ALICE_SERVICE_SLUG = 'alice_b2b'

STRUCTURED_LOGS = {
    'enable_tracking': True,
    'db_tracking': bool(get_setting('LOG_ALL_SQL_QUERIES', default=True)),
    'http_tracking': bool(get_setting('LOG_ALL_HTTP_REQUESTS', default=True)),
    'enable_stacktraces': False,
    'http_response_max_size': 0,
    'sql_warning_threshold': float(get_setting('SLOW_SQL_QUERIES_TIME', default=0.1)) * 1000,  # msec
    'context_providers': [
        'intranet.yandex_directory.src.yandex_directory.directory_logging.tools_structured_logs.library_integration_layer.providers.request',
        'intranet.yandex_directory.src.yandex_directory.directory_logging.tools_structured_logs.library_integration_layer.providers.auth',
        'intranet.yandex_directory.src.yandex_directory.directory_logging.tools_structured_logs.library_integration_layer.providers.b2b',
        'tools_structured_logs.logic.common_log_fields.field_structure.endpoint',
    ],
}

def_setting('PARTNER_DISK_HOST', default='https://cloud-api.dst.yandex.net:8443')
def_setting('PARTNER_DISK_TVM_CLIENT_ID', default=2000060, env_type='int')

def_setting('CRON_CHECK_BALANCE', default=True)

ALWAYS_ENABLED_SERVICES = ['calendar', 'mail', 'disk']
USER_ORGANIZATIONS_LIMIT = get_setting('USER_ORGANIZATIONS_LIMIT')

CLOUD_HOST = get_setting('CLOUD_HOST')
CLOUD_ORG_HOST = get_setting('CLOUD_ORG_HOST')
CLOUD_ORG_TIMEOUT = get_setting('CLOUD_ORG_TIMEOUT', env_type='float', default=3)
CLOUD_IAM_HOST = get_setting('CLOUD_IAM_HOST')
CLOUD_IAM_TIMEOUT = get_setting('CLOUD_IAM_TIMEOUT', env_type='float', default=3)
CLOUD_TS_HOST = get_setting('CLOUD_TS_HOST')
CLOUD_TS_TIMEOUT = get_setting('CLOUD_TS_TIMEOUT', env_type='float', default=1)
CLOUD_USE_LOCAL_SERT = get_setting('CLOUD_USE_LOCAL_SERT', env_type='bool', default=True)
CLOUD_DEFAULT_CA_PATH = '/etc/ssl/certs/ca-certificates.crt'
CLOUD_ACCESS_SERVICE_HOST = get_setting('CLOUD_ACCESS_SERVICE_HOST')
CLOUD_ACCESS_SERVICE_TIMEOUT = get_setting('CLOUD_ACCESS_SERVICE_TIMEOUT', env_type='float', default=1)
CLOUD_SESSION_SERVICE_HOST = get_setting('CLOUD_SESSION_SERVICE_HOST')
CLOUD_SESSION_SERVICE_TIMEOUT = get_setting('CLOUD_SESSION_SERVICE_TIMEOUT', env_type='float', default=1)
CLOUD_TRACKER_ID_PREFIX = get_setting('CLOUD_TRACKER_ID_PREFIX')

CLOUD_KEY = get_setting('CLOUD_KEY')
CLOUD_SERVICE_ACCOUNT_ID = get_setting('CLOUD_SERVICE_ACCOUNT_ID')
CLOUD_PRIVATE_KEY = get_setting('CLOUD_PRIVATE_KEY')
CLOUD_NOTIFY_URL = get_setting('CLOUD_NOTIFY_URL')

TRACKER_YT_ORGANIZATIONS_TABLE = get_setting('TRACKER_YT_ORGANIZATIONS_TABLE')

DOMENATOR_DOMAIN_TOKEN_PROXY = get_setting('DOMENATOR_DOMAIN_TOKEN_PROXY')
DOMENATOR_WHO_IS_DOMAIN_PROXY = get_setting('DOMENATOR_WHO_IS_DOMAIN_PROXY')
DOMENATOR_REGISTRAR_PROXY = get_setting('DOMENATOR_REGISTRAR_PROXY')
USE_DOMENATOR_ENABLED_FOR_DOMAINS = get_setting('USE_DOMENATOR_ENABLED_FOR_DOMAINS', default=False)

BLACK_DOMAINS_LIST = get_setting('BLACK_DOMAINS_LIST')
NEW_TRACKER_PRICING = get_setting('NEW_TRACKER_PRICING', default=False)

YA_TEAM_ORG_IDS = get_setting('YA_TEAM_ORG_IDS', default={})

LOGBROKER_ENDPOINT = f"{CURRENT_DC}.{get_setting('LOGBROKER_ENDPOINT')}"
LOGBROKER_PORT = get_setting('LOGBROKER_PORT')
LOGBROKER_TVM_ID = get_setting('LOGBROKER_TVM_ID')
LOGBROKER_TOPIC = get_setting('LOGBROKER_TOPIC')
LOGBROKER_CONSUMER_NAME = get_setting('LOGBROKER_CONSUMER_NAME')
PASSPORT_LOGBROKER_TOPIC = get_setting('PASSPORT_LOGBROKER_TOPIC')
PASSPORT_LOGBROKER_CONSUMER_NAME = get_setting('PASSPORT_LOGBROKER_CONSUMER_NAME')

LOGBROKER_READ_TIMEOUT = get_setting('LOGBROKER_READ_TIMEOUT', env_type='int', default=1)
LOGBROKER_READ_MAX_LAG = get_setting('LOGBROKER_READ_MAX_LAG', env_type='int', default=None)
LOGBROKER_READ_TOTAL_MESSAGES_IN_REQUEST = get_setting('LOGBROKER_READ_TOTAL_MESSAGES_IN_REQUEST', env_type='int', default=100)
LOGBROKER_READ_TOTAL_MESSAGES = get_setting('LOGBROKER_READ_TOTAL_MESSAGES', env_type='int', default=1)
LOGBROKER_READ_INTERVAL_MIN = get_setting('LOGBROKER_READ_INTERVAL_MIN', env_type='float', default=0.1)
LOGBROKER_READ_INTERVAL_MULTIPLIER = get_setting('LOGBROKER_READ_INTERVAL_MULTIPLIER', env_type='float', default=2)
LOGBROKER_READ_INTERVAL_MAX = get_setting('LOGBROKER_READ_INTERVAL_MAX', env_type='float', default=5)


TRIAL_SHUTDOWN_DATE_FOR_CLOUD_ORGANIZATIONS = datetime.fromisoformat(get_setting(
    'TRIAL_SHUTDOWN_DATE_FOR_CLOUD_ORGANIZATIONS',
    default='2021-04-01',
))

SCHEDULED_TASK_RESULT_LIFE_TIME = get_setting('SCHEDULED_TASK_RESULT_LIFE_TIME', default=7 * 24 * 60 * 60)
SCHEDULED_TASK_RESULT_MONITORING_LIFE_TIME = get_setting('SCHEDULED_TASK_RESULT_MONITORING_LIFE_TIME', default=1 * 24 * 60 * 60)

SYNC_CLOUD_ORGS_TASK_CHUNK_SIZE = get_setting('SYNC_CLOUD_ORGS_TASK_CHUNK_SIZE', default=1000)
SYNC_SSO_ORGS_TASK_CHUNK_SIZE = get_setting('SYNC_SSO_ORGS_TASK_CHUNK_SIZE', default=1000)
SYNC_SSO_ORG_TASK_MAX_OPERATIONS = get_setting('SYNC_SSO_ORG_TASK_MAX_OPERATIONS', default=100)

EVENT_NOTIFICATION_ENQUEUE_ON_END = get_setting('EVENT_NOTIFICATION_ENQUEUE_ON_END', env_type='bool', default=True)
EVENT_NOTIFICATION_TASK_CHUNKS_COUNT = get_setting('EVENT_NOTIFICATION_TASK_CHUNKS_COUNT', default=5)
EVENT_NOTIFICATION_TASK_EVENT_IN_CHUNK_COUNT = get_setting('EVENT_NOTIFICATION_TASK_EVENT_IN_CHUNK_COUNT', default=100)
