# -*- coding: utf-8 -*-
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
from intranet.yandex_directory.src.yandex_directory.core.utils.tasks import ChangeOrganizationOwnerTask
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_meta_connection,
)
from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import access_restore as access_restore_emails
from intranet.yandex_directory.src.yandex_directory.core.task_queue import Task
from intranet.yandex_directory.src.yandex_directory.core.models.access_restore import RestoreTypes
from intranet.yandex_directory.src.yandex_directory.core.models import (
    OrganizationAccessRestoreModel,
    OrganizationModel,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
    get_domains_from_db_or_domenator,
    DomainFilter,
)


class AfterAccessRestoreEmailTask(Task):

    singleton = True

    def do(self, restore_id, org_id):
        with get_meta_connection(for_write=True) as meta_connection:
            access_restore_emails.send_organization_new_owner(
                meta_connection,
                self.main_connection,
                restore_id,
            )
            access_restore_emails.send_take_organization(
                meta_connection,
                self.main_connection,
                restore_id,
            )


class AccessRestoreTask(Task):

    singleton = True
    need_rollback = True

    def do(self, restore_id, org_id):
        dependencies = self.get_dependencies(self.task_id)

        if dependencies:
            if self.has_failed_dependencies():
                self.rollback(restore_id, org_id)
            else:
                self.final_step(restore_id, org_id)
        else:
            self.first_step(restore_id, org_id)

    def first_step(self, restore_id, org_id):
        with get_meta_connection(for_write=True) as meta_connection:
            restore = OrganizationAccessRestoreModel(meta_connection).get(restore_id)
            if not restore:
                log.warning('Restore not exist')
                return

            OrganizationAccessRestoreModel(meta_connection).filter(
                id=restore_id
            ).update(
                restore_task_id=self.task_id
            )

            org_id = restore['org_id']
            new_admin_uid = restore['new_admin_uid']
            old_admin_uid = restore['old_admin_uid']
            domain = restore['domain']

            organization = OrganizationModel(self.main_connection).get(org_id)

            if not organization:
                log.warning('Organization not exist')
                return

            if organization['admin_uid'] != old_admin_uid:
                log.warning('Organization owner was changed')
                return

            domain_info = get_domains_from_db_or_domenator(
                meta_connection=meta_connection,
                main_connection=self.main_connection,
                domain_filter=DomainFilter(org_id=org_id, name=domain, master=True, owned=True),
                one=True,
            )

        if not domain_info:
            log.warning('No master domain in organization')
            return

        task = ChangeOrganizationOwnerTask(self.main_connection).delay(
            org_id=org_id,
            old_owner_uid=old_admin_uid,
            new_owner_uid=new_admin_uid,
        )
        self.wait_for(task)

    def final_step(self, restore_id, org_id):
        with get_meta_connection(for_write=True) as meta_connection:
            OrganizationAccessRestoreModel(meta_connection).\
                filter(id=restore_id).\
                update(state=RestoreTypes.success)
        try:
            AfterAccessRestoreEmailTask(self.main_connection).delay(org_id=org_id, restore_id=restore_id)
        except DuplicatedTask:
            pass

    def rollback(self, restore_id, org_id):
        with get_meta_connection(for_write=True) as meta_connection:
            OrganizationAccessRestoreModel(meta_connection). \
                filter(id=restore_id). \
                update(state=RestoreTypes.failed)
