# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory import webmaster
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_shard,
    get_main_connection,
    get_meta_connection,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import utcnow
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationMetaModel
from intranet.yandex_directory.src.yandex_directory.core.models.access_restore import (
    OrganizationAccessRestoreModel,
    RestoreTypes,
)
from intranet.yandex_directory.src.yandex_directory.access_restore.tasks import AccessRestoreTask
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask

from intranet.yandex_directory.src.yandex_directory import app


def init_access_restore_task():
    """
    Для подтвержденных доменов запускаем задачу восстановления доступа.
    Для протухших восстановлений обновим статус
    """

    with get_meta_connection(for_write=True) as meta_connection:
        # Обновим статус для протухших задач.
        # При проставлении протухания не фильтруем по environment,
        # потому что с текущим построителем запросов сложно сделать
        # подходящий UPDATE. Да и в данном случае это не критично.
        OrganizationAccessRestoreModel(meta_connection).filter(
            expires_at__lt=utcnow(),
            state=RestoreTypes.in_progress,
        ).update(
            state=RestoreTypes.expired
        )

        restores = OrganizationAccessRestoreModel(meta_connection).filter(
            state=RestoreTypes.in_progress,
        ).fields(
            'new_admin_uid', 'domain', 'id', 'org_id'
        ).all()

        # Обязательно надо обрабатывать только запросы по смене доступа
        # из текущего окружения. Иначе AccessRestore может запуститься дважды,
        # да и при тестировании будет странное поведение.
        same_environment = OrganizationMetaModel(meta_connection).orgs_from_same_environment(
            [item['org_id'] for item in restores]
        )

        for restore in restores:
            new_admin_uid = restore['new_admin_uid']
            domain = restore['domain']
            restore_id = restore['id']
            org_id = restore['org_id']

            if org_id not in same_environment:
                continue

            if not webmaster.is_verified(domain, new_admin_uid):
                continue

            # для подтвержденных доменов запустим восстановление доступа
            shard = get_shard(meta_connection, org_id)
            with get_main_connection(shard=shard, for_write=True) as main_connection:
                try:
                    AccessRestoreTask(main_connection).delay(
                        restore_id=restore_id,
                        org_id=org_id,
                    )
                except DuplicatedTask:
                    pass
