# coding: utf-8
from flask import (
    g,
    request,
)
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    scopes_required,
    no_permission_required,
    requires,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import (
    scope,
)
from intranet.yandex_directory.src.yandex_directory.common import schemas
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_shard,
    get_main_connection,
    mogrify,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import NewAdminAlreadyOwner, NewAdminFromOtherOrganization
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    json_error_not_found,
    json_error,
    format_datetime,
    get_user_id_from_passport_by_login,
    get_user_data_from_blackbox_by_uid,
    mask_email,
)
from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import access_restore as access_restore_emails
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    get_organization_admin_uid,
    is_outer_uid,
    only_fields,
)

from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.models import (
    DomainModel,
    UserModel,
    OrganizationAccessRestoreModel,
    UserMetaModel,
    DepartmentModel,
    GroupModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.department import ROOT_DEPARTMENT_ID
from intranet.yandex_directory.src.yandex_directory.core.models.group import GROUP_TYPE_GENERIC
from intranet.yandex_directory.src.yandex_directory.core.models.access_restore import RestoreTypes
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema,
)
from intranet.yandex_directory.src.yandex_directory.auth.middlewares import get_client_ip_from
from intranet.yandex_directory.src.yandex_directory.core.utils import get_login
from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
    get_domains_from_db_or_domenator,
    DomainFilter,
)

RESTORE_CREATE_SCHEMA = {
    'title': 'Create restore record',
    'type': 'object',
    'properties': {
        'domain': schemas.STRING,
        'control_answers': {
            'type': 'object',
            'properties': {
                'admins': schemas.LIST_OF_STRINGS,
                'maillists': schemas.LIST_OF_STRINGS,
                'users': schemas.LIST_OF_STRINGS,
                'no_users': schemas.BOOLEAN,
                'no_maillists': schemas.BOOLEAN,
                'forgot_admins': schemas.BOOLEAN,
                'enabled_services': schemas.LIST_OF_STRINGS,
                'paid': schemas.BOOLEAN,
            },
        }
    },
    'required': ['domain', 'control_answers'],
    'additionalProperties': True
}

MAX_LOGINS_TO_CHECK = 10
CORRECT_THRESHOLD = 0.75


def get_restore_list_response(restores):
    public_fields = ['domain', 'expires_at', 'id', 'state', 'created_at']
    response = [only_fields(r, *public_fields) for r in restores]
    for r in response:
        r['restore_id'] = r['id']
        r['created_at'] = format_datetime(r['created_at'])
        r['expires_at'] = format_datetime(r['expires_at'])
        del r['id']
    return response


class RestoreDetailView(View):
    @scopes_required([scope.access_restore])
    @no_permission_required
    @requires(org_id=False, user=True)
    def get(self, meta_connection, _, restore_id):
        """
        Статус заявки на восставновление доступа
        ---
        tags:
          - Восстановление доступа
        parameters:
          - in: path
            name: restore_id
            required: true
            type: string
            description: id записи на восстановление
        responses:
          200:
            description: Словарь с информацией.
          404:
            description: Задача с указанным id не найдена.
        """
        restore_info = OrganizationAccessRestoreModel(meta_connection) \
            .filter(id=restore_id, new_admin_uid=g.user.passport_uid).all()

        if restore_info:
            return json_response(get_restore_list_response(restore_info)[0])
        else:
            return json_error_not_found()


class RestoreListView(View):
    methods = ['get', 'post']

    @scopes_required([scope.access_restore])
    @no_permission_required
    @requires(org_id=False, user=True)
    def get(self, meta_connection, _):
        """
        Список непросроченных заявок на восстановление доступа, которые запустил пользователь
        ---
        tags:
          - Восстановление доступа
        responses:
          200:
            description: Список заявок
          404:
            description: Заявки для данного пользователя не найдены
        """
        restore_info = OrganizationAccessRestoreModel(meta_connection) \
            .get_all_valid_restores_for_uid(g.user.passport_uid)

        return json_response(get_restore_list_response(restore_info))

    @scopes_required([scope.access_restore])
    @no_permission_required
    @uses_schema(RESTORE_CREATE_SCHEMA)
    @requires(org_id=False, user=True)
    def post(self, meta_connection, _, data):
        """
        Создает заявку на восстановление доступа
        ---
        tags:
          - Восстановление доступа
        parameters:
          - in: body
            name: body
        responses:
          201:
            description: Заявка создана
          404:
            description: Указанного домена нет в Коннекте
          422:
            description: Новый админ уже владелец или из другой орагнизации
        """

        domain = data['domain']
        new_admin_uid = g.user.passport_uid

        # проверим, вдруг такая заявка уже есть, тогда вернем ее статус
        active_restore = OrganizationAccessRestoreModel(meta_connection)\
            .get_active_restore_for_uid_domain(new_admin_uid, domain)
        if active_restore:
            return json_response(
                data={
                    'restore_id': active_restore['id'],
                    'state': active_restore['state']
                },
                status_code=201,
            )

        domain_info = get_domains_from_db_or_domenator(
            meta_connection=meta_connection,
            domain_filter=DomainFilter(name=domain, owned=True, master=True),
            one=True
        )
        if not domain_info:
            return json_error_not_found()

        org_id = domain_info['org_id']
        new_admin_uid = g.user.passport_uid

        shard = get_shard(meta_connection, org_id)
        with get_main_connection(for_write=True, shard=shard) as main_connection:
            old_admin_uid = get_organization_admin_uid(main_connection, org_id)

            if new_admin_uid == old_admin_uid:
                raise NewAdminAlreadyOwner(uid=new_admin_uid)

            # новый владелец должен быть внешней учеткой или состоять в той же ораганизации
            if not (is_outer_uid(new_admin_uid)
                    or UserModel(main_connection).filter(org_id=org_id, id=new_admin_uid, is_robot=False).one()):
                raise NewAdminFromOtherOrganization(uid=new_admin_uid)

            control_answers = data['control_answers']
            if getattr(g, 'user', None) and g.user.ip:
                user_ip = g.user.ip
            else:
                user_ip = get_client_ip_from(request.headers)

            restore_record = OrganizationAccessRestoreModel(meta_connection).create(
                org_id=org_id,
                domain=domain,
                new_admin_uid=new_admin_uid,
                old_admin_uid=old_admin_uid,
                ip=user_ip,
                control_answers=control_answers,
            )

            access_restore_emails.send_some_try_restore(
                meta_connection,
                main_connection,
                restore_record['id']
            )

        if not control_answers_is_valid(meta_connection, control_answers, org_id, shard, domain):
            OrganizationAccessRestoreModel(meta_connection)\
                .filter(id=restore_record['id'])\
                .update(state=RestoreTypes.invalid_answers)

            return json_response(
                data={
                    'restore_id': restore_record['id'],
                    'state': 'invalid_answers',
                },
                status_code=201,
            )

        return json_response(
            data={
                'restore_id': restore_record['id'],
                'state': restore_record['state'],
            },
            status_code=201,
        )


class MaskOwnerLoginView(View):
    methods = ['get']

    @scopes_required([scope.access_restore])
    @no_permission_required
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _, domain):
        """
        Маскированный логин (email) текущего владельца организации.
        Алгоритм маскирования взяли из паспорта
        ---
        tags:
          - Восстановление доступа
        responses:
          200:
            description: Маскированный логин
          404:
            description: Домен или владелец не найден
        """
        domain_info = get_domains_from_db_or_domenator(
            meta_connection=meta_connection,
            domain_filter=DomainFilter(name=domain, owned=True, master=True),
            one=True,
        )
        if not domain_info:
            return json_error_not_found()

        org_id = domain_info['org_id']
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard) as main_connection:
            admin_uid = get_organization_admin_uid(main_connection, org_id)
            admin_info = get_user_data_from_blackbox_by_uid(admin_uid)
            if admin_info:
                admin_login = admin_info['default_email'] or admin_info['login']
                return json_response({'owner_login': mask_email(admin_login)})
            return json_error_not_found()


def control_answers_is_valid(meta_connection, answers, org_id, shard, domain):
    # Пока проверяем только админов и ящики
    # Из каждого массива берем первые 3 элемента (пока), чтобы не проверять бесконечно долго
    # Подробнее алгоритм здесь https://wiki.yandex-team.ru/ws/Vosstanovlenie-dostupa/

    inner_admins = []
    user_filter = {
        'org_id': org_id,
        'is_dismissed': False,
        'is_robot': False
    }

    def get_admins_score():
        def _check_inner_admin(main_conn, uid, login):
            if uid and UserModel(main_conn).filter(id=uid, **inner_admin_filter).one():
                inner_admins.append(get_login(login))
                return True

        admins = set(answers.get('admins', [])[:MAX_LOGINS_TO_CHECK])
        forgot_admins = answers.get('forgot_admins')

        admins_score = 0
        if forgot_admins:
            # считаем валидным ответом, пользователь не помнит/не знает админов
            admins_score = 2
        elif admins:
            outer_admins = 0
            inner_admin_filter = user_filter.copy()
            inner_admin_filter['role'] = ['admin', 'deputy_admin']

            with get_main_connection(shard=shard) as main_connection:
                for login in admins:
                    # сначала логин проверяем как есть
                    uid = get_user_id_from_passport_by_login(login)
                    if uid:
                        if UserMetaModel(meta_connection).filter(
                                org_id=org_id,
                                id=uid,
                                user_type=['outer_admin', 'deputy_admin']
                        ).one():
                            # нашли внешнего админа
                            outer_admins += 1
                            continue
                        elif _check_inner_admin(main_connection, uid, login):
                            continue
                    # случай, когда логин указан без доменной части
                    # если доменная часть была, но такого пользователя не существует,
                    # то эта ветка ничего не сломает, потому что домен не отрезаем
                    login_with_domain = '{}@{}'.format(login, domain)

                    # ищем по uid, а не просто по login, чтобы избежать мороки с алиасами учетки
                    uid = get_user_id_from_passport_by_login(login_with_domain)
                    _check_inner_admin(main_connection, uid, login)

                if outer_admins + len(inner_admins) >= CORRECT_THRESHOLD * len(admins):
                    admins_score = 3

        return admins_score

    def get_users_score():
        users = set(map(get_login, answers.get('users', [])[:MAX_LOGINS_TO_CHECK]))
        no_users = answers.get('no_users')

        users_score = 0
        with get_main_connection(shard=shard) as main_connection:
            total_users = UserModel(main_connection).filter(**user_filter).count()
            if no_users:
                if total_users == 0:
                    # если пользователей действительно нет - это валидный ответ
                    users_score = 1
                elif UserModel(main_connection).filter(role='user', **user_filter).count() == 0 and inner_admins:
                    # либо обычных пользователей у него нет, но есть внутренние админы/замы
                    # и он указал их в графе админы - это ок
                    users_score = 1
            elif users:
                if inner_admins:
                    # если внутренние админы были указаны в графе admins - не будем тут их учитывать
                    # чтобы пользователь не использовал один и тот же ящик дважды
                    users -= set(inner_admins)
                    if not users:
                        if UserModel(main_connection).filter(role='user', **user_filter).count() == 0:
                            # других пользователей нет - считаем ответ валидным
                            users_score = 1
                        return users_score

                users_count = len(users)

                # добавили требование на минимальное число ящиков, которое надо ввести
                if not check_min_users_count(total_users, users_count + len(inner_admins)):
                    return users_score

                if count_logins_from_db(main_connection, list(users), org_id, 'users') \
                        >= users_count * CORRECT_THRESHOLD:
                    users_score = 2

        return users_score

    def get_maillists_score():
        maillists = set(map(get_login, answers.get('maillists', [])[:MAX_LOGINS_TO_CHECK]))
        no_maillists = answers.get('no_maillists')

        maillists_score = 0
        maillist_filter = {
            'org_id': org_id,
            'removed': False,
        }
        with get_main_connection(shard=shard) as main_connection:
            if no_maillists:
                # если рассылок действительно нет - это валидный ответ
                if organization_has_no_maillists_except_all(main_connection, maillist_filter):
                    maillists_score = 1
            elif maillists:
                # сначала уберем из ответов корневую рассылку
                root_label = DepartmentModel(main_connection).get(
                    department_id=ROOT_DEPARTMENT_ID,
                    org_id=org_id,
                    fields=['label']
                )['label']
                if root_label in maillists:
                    maillists.remove(root_label)
                    if not maillists:
                        if organization_has_no_maillists_except_all(main_connection, maillist_filter):
                            # если в организации больше нет рассылок, считаем ответ валидным
                            maillists_score = 2
                        return maillists_score
                groups_count = count_logins_from_db(main_connection, list(maillists), org_id, 'groups')
                deps_count = count_logins_from_db(main_connection, list(maillists), org_id, 'departments')
                if (groups_count + deps_count) >= len(maillists) * CORRECT_THRESHOLD:
                    maillists_score = 2

        return maillists_score

    mid_score = get_admins_score() + get_users_score()
    if mid_score < 4:
        return False

    return mid_score + get_maillists_score() > 4


def count_logins_from_db(main_connection, logins, org_id, table):
    table_filter = {
        'users': 'AND NOT is_dismissed',
        'groups': "AND NOT removed AND type = 'generic'",
        'departments': 'AND NOT removed',
    }
    table_col = 'nickname' if table == 'users' else 'label'

    query = """
                SELECT count(*) FROM {table}
                WHERE org_id = %(org_id)s
                {table_filter}
                AND ({table_col} IN %(logins)s OR aliases::jsonb ?| %(json_logins)s::text[]);
        """.format(table=table, table_col=table_col, table_filter=table_filter[table])
    query = mogrify(
        main_connection,
        query,
        {
            'logins': tuple(logins),
            'org_id': org_id,
            'json_logins': logins,
        }
    )
    return main_connection.execute(query).fetchone()[0]


def check_min_users_count(total_org_users, input_users_count):
    # пороги на глаз, примерно как саппорты запрашивают
    users_count_threshold = {
        4: 1,
        9: 2,
        20: 3,
        40: 4,
    }
    threshold_for_big = 5

    for total_count, min_count in sorted(users_count_threshold.items()):
        if total_org_users < total_count:
            return input_users_count >= min_count
    return input_users_count >= threshold_for_big


def organization_has_no_maillists_except_all(main, filter_data):
    return GroupModel(main).filter(type=GROUP_TYPE_GENERIC, **filter_data).count() \
           + DepartmentModel(main).filter(id__notequal=1, **filter_data).count() == 0
