# coding: utf-8
from flask import (
    g,
)

from intranet.yandex_directory.src.yandex_directory import webmaster
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    scopes_required,
    no_permission_required,
    requires,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import (
    scope,
)
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_shard,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import VerificationAlreadyInProgress
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    json_error_not_found,
)
from intranet.yandex_directory.src.yandex_directory.access_restore.tasks import AccessRestoreTask
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.models import (
    OrganizationAccessRestoreModel,
)
from intranet.yandex_directory.src.yandex_directory.swagger import uses_schema

DOMAIN_START_OWNERSHIP_SCHEMA = {
    'title': 'Domain restore verification',
    'type': 'object',
    'properties': {
        'verification_type': {
            'enum': list(map(str.lower, webmaster.PUBLIC_VERIFICATION_TYPES)),
        },
    },
    'required': ['verification_type'],
    'additionalProperties': False
}


class BaseOwnershipView(View):
    def get_restore_info(self, meta_connection, restore_id):
        return OrganizationAccessRestoreModel(meta_connection) \
            .filter(id=restore_id, new_admin_uid=g.user.passport_uid) \
            .one()


class RestoreOwnershipView(BaseOwnershipView):

    def get_webmaster_info(self, domain, admin_uid):
        info = webmaster.info(
            domain,
            admin_uid,
            ignore_errors='USER__HOST_NOT_ADDED'
        )

        # если домена нет в webmaster сразу его добавим
        if not info.get('data'):
            webmaster.add(
                domain,
                admin_uid,
            )
            info = webmaster.info(
                domain,
                admin_uid,
                ignore_errors='USER__HOST_NOT_ADDED'
            )
        return info

    def get_verification_types(self, domain, admin_uid):

        info = self.get_webmaster_info(domain, admin_uid)

        data = info['data']
        verification_uin = data['verificationUin']

        methods = webmaster.list_applicable(
            domain,
            admin_uid,
        )

        return [
            {
                'method': method.lower(),
                'code': verification_uin,
            }
            for method in methods
        ]

    def get_last_check(self, domain, admin_uid):
        data = self.get_webmaster_info(domain, admin_uid).get('data', {})

        last_check = {}

        if 'lastVerificationAttemptTime' in data:
            last_check = {
                'date': data['lastVerificationAttemptTime'],
                'method': data['verificationType'].lower(),
            }

            if 'lastVerificationFailInfo' in data:
                last_check['fail_type'] = data['lastVerificationFailInfo'].get('type', 'unknown').lower()
                if 'reason' in data['lastVerificationFailInfo']:
                    last_check['fail_reason'] = data['lastVerificationFailInfo'].get('reason', 'unknown').lower()

        return last_check

    @scopes_required([scope.access_restore])
    @no_permission_required
    @requires(org_id=False, user=True)
    def get(self, meta_connection, _, restore_id):
        """
        Получить список способов подтверждения владений доменом через вебмастер и статус подтверждения.

        ```
        {
            "domain": "some.example.com",
            "owned": "true|false",
            "methods": [
                {
                    "method": "file",
                    "code": "123456",
                },
                ...
            ],
            "last_check": {
                "date": "2018-03-02T13:39:49.839Z",
                "method": : "html_file",
                "fail_type": "page_unavailable",   // опционально
                "fail_reason": "connection_error", // опционально
            }
        }
        ```

        ---
        tags:
          - Восстановление доступа
        parameters:
          - in: path
            name: restore_id
            required: true
            type: string
            description: id записи на восстановление
        responses:
          200:
            description: Словарь с информацией.
          404:
            description: Задача с указанным id не найдена.
        """

        new_admin_uid = g.user.passport_uid
        restore_info = self.get_restore_info(meta_connection, restore_id)
        if not restore_info:
            return json_error_not_found()

        lower_domain_name = restore_info['domain'].lower()

        response = {
            'domain': lower_domain_name,
            'owned': webmaster.is_verified(lower_domain_name, new_admin_uid),
            'methods': self.get_verification_types(lower_domain_name, new_admin_uid),
            'last_check': self.get_last_check(lower_domain_name, new_admin_uid),
        }
        return json_response(response)


class RestoreVerifyView(BaseOwnershipView):

    @scopes_required([scope.access_restore])
    @no_permission_required
    @requires(org_id=False, user=True)
    @uses_schema(DOMAIN_START_OWNERSHIP_SCHEMA)
    def post(self, meta_connection, _, data, restore_id):
        """
        Начать подтверждение домена выбранным способом

        ```
        {
            "domain": "some.example.com",
            "owned": "true|false",
        }
        ```

        ---
        tags:
          - Восстановление доступа
        parameters:
          - in: path
            name: restore_id
            required: true
            type: string
            description: id записи на восстановление
        responses:
          200:
            description: Словарь с информацией.
          404:
            description: Задача с указанным id не найдена.
        """

        new_admin_uid = g.user.passport_uid
        restore_info = self.get_restore_info(meta_connection, restore_id)
        if not restore_info:
            return json_error_not_found()

        method = data['verification_type'].upper()
        lower_domain_name = restore_info['domain'].lower()

        response = {
            'domain': lower_domain_name,
            'owned': False,
        }
        if webmaster.is_verified(lower_domain_name, new_admin_uid):
            response['owned'] = True
            org_id = restore_info['org_id']
            shard = get_shard(meta_connection, org_id)
            with get_main_connection(shard=shard, for_write=True) as main_connection:
                try:
                    AccessRestoreTask(main_connection).delay(
                        restore_id=restore_info['id'],
                        org_id=org_id,
                    )
                except DuplicatedTask:
                    pass
        else:
            try:
                webmaster.verify(lower_domain_name, new_admin_uid, method)
            except webmaster.WebmasterError as ex:
                if ex.code == 'webmaster_verify_host__verification_is_in_progress':
                    raise VerificationAlreadyInProgress()
                raise

        return json_response(response)
