# -*- coding: utf-8 -*-

from flask import (
    g,
)

from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    requires,
    permission_required)
from intranet.yandex_directory.src.yandex_directory.common.utils import get_internal_roles_by_uid
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response
)
from intranet.yandex_directory.src.yandex_directory.core.idm import ADMIN_ROLES
from intranet.yandex_directory.src.yandex_directory.core.permission.internal_permissions import common_internal_permission
from intranet.yandex_directory.src.yandex_directory.core.views.base import View


# Эта мапа  показывает, какие ручки доступны пользоватлелям с определенными ролями
# Она заполняется в момент добавления роутов в приложение
ROLES_TO_METHODS = {
    ADMIN_ROLES.admin: [],
    ADMIN_ROLES.support: [],
    ADMIN_ROLES.assessor: [],
    ADMIN_ROLES.bizdev: [],
}


class AdminAllowedRoutesView(View):
    @permission_required([common_internal_permission.allowed_routes])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _):
        """
        Информация о доступных пользователю хэндлерах

        Пользователь идентифицируется с помощью TVM2 user-ticket,
        который передается в хэдере X-Ya-User-Ticket.
        Если тикет некорректый, или его срок годности истёк, то вернётся 401
        Если пользователь не имеет роли в админке в IDM, то вернётся 403

        Пример ответа:
            [
                {'method': 'get', 'route': '/admin/organizations/'},
                {'method': 'get', 'route': '/admin/organizations/search/'},
                ...
            ]

        ---
        tags:
          - Админка
        responses:
          200:
            description: Success
          403:
            description: Request is not available for this service or user
          401:
            description: Invalid or expired TVM 2.0 user- or service-ticket
        """

        user_id = g.user.passport_uid
        roles = get_internal_roles_by_uid(user_id)

        allowed_routes = []
        if ADMIN_ROLES.admin in roles:
            allowed_routes = ROLES_TO_METHODS[ADMIN_ROLES.admin]
        elif ADMIN_ROLES.support in roles:
            allowed_routes = ROLES_TO_METHODS[ADMIN_ROLES.support]
        elif ADMIN_ROLES.assessor in roles:
            allowed_routes = ROLES_TO_METHODS[ADMIN_ROLES.assessor]
        return json_response(
            allowed_routes
        )
