import collections
from datetime import timedelta, datetime

from flask import request

from intranet.yandex_directory.src.yandex_directory.auth.decorators import permission_required, requires
from intranet.yandex_directory.src.yandex_directory.common.db import get_shard, get_main_connection
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationLicenseConsumedInfoModel
from intranet.yandex_directory.src.yandex_directory.core.permission import common_internal_permission
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.common.utils import json_response, Ignore
from intranet.yandex_directory.src.yandex_directory.core.models import ServiceModel
from intranet.yandex_directory.src.yandex_directory.common import schemas
from intranet.yandex_directory.src.yandex_directory.swagger import uses_schema_for_get
from intranet.yandex_directory.src.yandex_directory.core.models import UserModel

TRACKER_LICENSES_CONSUMED_INFO_GET_SCHEMA = {
    'title': 'Get tracker licenses consumed info',
    'type': 'object',
    'properties': {
        'begin_date': schemas.DATE_OR_NULL,
        'end_date': schemas.DATE_OR_NULL,
    },
    'required': [],
    'additionalProperties': True,
}


class AdminTrackerLicensesView(View):
    @uses_schema_for_get(TRACKER_LICENSES_CONSUMED_INFO_GET_SCHEMA)
    @permission_required([common_internal_permission.organizations_services_subscribers_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _, org_id):
        requested_begin_date_str = request.args.get('begin_date')
        requested_end_date_str = request.args.get('end_date')

        begin_date = None
        one_day_delta = timedelta(days=1)
        if requested_begin_date_str is not None:
            begin_date = datetime.strptime(requested_begin_date_str, '%Y-%m-%d').date() - one_day_delta

        end_date = None
        if requested_end_date_str is not None:
            end_date = datetime.strptime(requested_end_date_str, '%Y-%m-%d').date()

        shard = get_shard(meta_connection, org_id)
        service_id = ServiceModel(meta_connection).get_by_slug('tracker', ['id'])['id']
        with get_main_connection(shard=shard) as main_connection:
            items = OrganizationLicenseConsumedInfoModel(main_connection).filter(
                org_id=org_id,
                service_id=service_id,
                begin_date=begin_date,
                end_date=end_date,
            ).order_by('for_date').fields('user_id', 'for_date').all()

            if not items:
                return json_response({})

            uids_by_date = collections.defaultdict(set)
            all_uids = set()
            for item in items:
                date = str(item['for_date'])
                uid = item['user_id']
                uids_by_date[date].add(uid)
                all_uids.add(uid)

            users = UserModel(main_connection).filter(id=all_uids, org_id=org_id, is_dismissed=Ignore)\
                .fields('id', 'nickname', 'first_name', 'last_name').all()
            users_by_uid = {user['id']: user for user in users}

        result_min_date = begin_date + one_day_delta if begin_date is not None else items[0]['for_date']
        result_max_date = end_date if end_date is not None else items[-1]['for_date']

        result = collections.defaultdict(dict)

        cur_date = result_min_date
        while cur_date <= result_max_date:
            cur_date_str = str(cur_date)
            uids_in_cur_date = uids_by_date.get(cur_date_str, set())

            result[cur_date_str]['count'] = len(uids_in_cur_date)
            result[cur_date_str]['users'] = [users_by_uid[uid] for uid in uids_in_cur_date]

            prev_date = cur_date - one_day_delta
            uids_in_prev_date = uids_by_date.get(str(prev_date), set())

            added_uids = uids_in_cur_date - uids_in_prev_date
            deleted_uids = uids_in_prev_date - uids_in_cur_date

            result[cur_date_str]['added'] = [users_by_uid[uid] for uid in added_uids]
            result[cur_date_str]['deleted'] = [users_by_uid[uid] for uid in deleted_uids]

            cur_date += one_day_delta

        return json_response(result)
