# -*- coding: utf-8 -*-
from dateutil.relativedelta import relativedelta
from flask import (
    g,
    request,
)

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory import webmaster
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    requires,
    permission_required,
)
from intranet.yandex_directory.src.yandex_directory.common import schemas
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_shard,
    get_shard_numbers,
    mogrify,
)
from intranet.yandex_directory.src.yandex_directory.common.pagination import Paginator
from intranet.yandex_directory.src.yandex_directory.common.schemas import INTEGER, DATE
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    build_multishard_list_response,
    build_list_response,
    json_response,
    json_error_invalid_value,
    json_error_not_found,
    format_datetime,
    get_user_data_from_blackbox_by_uid,
    Ignore,
    format_date,
    utcnow,
    get_user_id_from_passport_by_login,
    get_object_or_404,
    _get_int_or_none,
    prepare_for_tsquery,
    parse_date,
    get_domain_id_from_blackbox,
)
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_domain_alienate,
    action_service_trial_end,
)
from intranet.yandex_directory.src.yandex_directory.core.exceptions import (
    OrganizationIsWithoutContract,
    BillingException,
    RequiredFieldsMissed,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import OrganizationNotReadyError, InvalidValue
from intranet.yandex_directory.src.yandex_directory.core.maillist_check.tasks import MaillistsCheckTask
from intranet.yandex_directory.src.yandex_directory.core.models import (
    OrganizationModel,
    OrganizationMetaModel,
    UserMetaModel,
    UserModel,
    DepartmentModel,
    GroupModel,
    OrganizationServiceModel,
    UserServiceLicenses,
    ActionModel,
    ServiceModel,
    DomainModel,
    SupportActionMetaModel,
    MaillistCheckModel,
    OrganizationRevisionCounterModel,
    PartnersMetaModel,
    ResourceModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.action import SupportActions
from intranet.yandex_directory.src.yandex_directory.core.models.group import relation_name
from intranet.yandex_directory.src.yandex_directory.core.models.organization import (
    organization_type,
    vip_reason,
)
from intranet.yandex_directory.src.yandex_directory.core.models.service import (
    enable_service,
    trial_status,
    disable_service,
    reason,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.internal_permissions import (
    common_internal_permission,
    support_internal_permission,
    bizdev_internal_permission,
    assessor_internal_pemissions,
)
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    prepare_user,
    ensure_integer,
    get_domains_by_org_id,
    only_attrs,
    get_organization_admin_uid,
    get_model_by_type,
    is_outer_uid,
    int_or_none,
    get_master_domain,
    add_avatar_id_for_user_objects,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.tasks import ChangeOrganizationOwnerTask
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.swagger import uses_schema
from intranet.yandex_directory.src.yandex_directory.webmaster import set_domain_as_owned
from intranet.yandex_directory.src.yandex_directory.core.features.utils import (
    get_organization_features_info,
    is_feature_enabled,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
    get_domains_from_db_or_domenator,
    DomainFilter,
    update_domains_in_db_or_domenator,
    DomainUpdateFilter,
    DomainUpdateData,
)
from intranet.yandex_directory.src.yandex_directory.core.features import USE_DOMENATOR
from intranet.yandex_directory.src.yandex_directory.core.views.organization.view import OrganizationFeaturesChangeView

CHANGE_ORG_TYPE_SCHEMA = {
    'title': 'ChangeOrganizationType',
    'type': 'object',
    'oneOf': [
        {
            'properties': {
                'org_type': {'enum': [
                    organization_type.yandex_project,
                    organization_type.test,
                    organization_type.common,
                    organization_type.cloud,
                ]},
                'comment': schemas.STRING,
            },
            'required': ['org_type', 'comment'],
            'additionalProperties': False,
        },
        {
            'properties': {
                'org_type': {'enum': [organization_type.education, organization_type.charity]},
                'client_id': INTEGER,
                'person_id': INTEGER,
                'comment': schemas.STRING,
            },
            'required': ['org_type', 'client_id', 'comment', 'person_id'],
            'additionalProperties': False,
        },
        {
            'properties': {
                'org_type': {'enum': organization_type.partner_types},
                'partner_id': INTEGER,
                'comment': schemas.STRING,
            },
            'required': ['org_type', 'comment'],
            'additionalProperties': False,
        },
    ]
}

CHANGE_ORG_SUBSCRIPTION_PLAN = {
    'title': 'Change organization subscription plan',
    'type': 'object',
    'oneOf': [
        {
            'properties': {
                'subscription_plan': {'enum': ['paid']},
                'subscription_plan_expires_at': DATE,
                'comment': schemas.STRING,
            },
            'required': ['subscription_plan', 'subscription_plan_expires_at', 'comment'],
            'additionalProperties': False,
        },
        {
            'properties': {
                'subscription_plan': {'enum': ['free']},
                'comment': schemas.STRING,
            },
            'required': ['subscription_plan', 'comment'],
            'additionalProperties': False,
        },
        {
            'properties': {
                'subscription_plan_expires_at': DATE,
                'comment': schemas.STRING,
            },
            'required': ['subscription_plan_expires_at', 'comment'],
            'additionalProperties': False,
        },
    ],
}

CHANGE_OWNER_SCHEMA = {
    'title': 'Change organization owner',
    'type': 'object',
    'properties': {
        'new_owner_login': schemas.STRING,
        'comment': schemas.STRING,
    },
    'required': ['new_owner_login', 'comment'],
    'additionalProperties': False
}

VERIFY_DOMAIN_SCHEMA = {
    'title': 'Verify domain ownership',
    'type': 'object',
    'properties': {
        'comment': schemas.STRING,
    },
    'required': ['comment'],
    'additionalProperties': False
}

CHANGE_SERVICE_LIMITS_SCHEMA = {
    'title': 'Change organization service limits',
    'type': 'object',
    'properties': {
        'user_limit': INTEGER,
        'expires_at': DATE,
        'comment': schemas.STRING,
    },
    'required': ['comment'],
    'additionalProperties': False,
}

ENABLE_SERVICE_LIMITS_SCHEMA = {
    'title': 'Enable organization service limits',
    'type': 'object',
    'properties': {
        'user_limit': INTEGER,
        'expires_at': DATE,
        'comment': schemas.STRING,
    },
    'required': ['comment'],
    'additionalProperties': False,
}

DISABLE_SERVICE_LIMITS_SCHEMA = {
    'title': 'Disable organization service limits',
    'type': 'object',
    'properties': {
        'comment': schemas.STRING,
    },
    'required': ['comment'],
    'additionalProperties': False,
}


ACTION_FEATURE_SCHEMA = {
    'title': 'Enable/disable feature in organization',
    'type': 'object',
    'properties': {
        'comment': schemas.STRING,
    },
    'required': ['comment'],
    'additionalProperties': False,
}

MANAGE_WHITELIST_SCHEMA = {
    'title': 'Add/delete organization to whitelist',
    'type': 'object',
    'properties': {
        'comment': schemas.STRING,
    },
    'required': ['comment'],
    'additionalProperties': False,
}

ORGANIZATION_BLOCK_SCHEMA = {
    'title': 'Block/unblock organization',
    'type': 'object',
    'properties': {
        'comment': {
            'type': 'string',
        },
    },
    'required': ['comment'],
    'additionalProperties': False
}

CONTAINERS_MAX_PER_PAGE = 50


class AdminOrganizationsListView(View):
    allowed_ordering_fields = ['id', 'subscription_plan']

    @permission_required([common_internal_permission.organizations_list_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _):
        response = build_multishard_list_response(
            model=OrganizationModel,
            path=request.path,
            model_filters=self._get_filters(),
            query_params=request.args.to_dict(),
            prepare_result_item_func=_prepare_organization,
            max_per_page=app.config['PAGINATION']['max_per_page_for_admin'],
            order_by=self._get_ordering_fields(),
        )
        return json_response(
            response['data'],
        )

    def _get_filters(self):
        filters = {}

        def _build_list_filter(field_name):
            if request.args.get(field_name):
                values = request.args.get(field_name).split(',')
                if len(values) == 1:
                    values = values[0]
                filters[field_name] = values

        _build_list_filter('source')
        _build_list_filter('subscription_plan')
        _build_list_filter('text')
        _build_list_filter('type')
        _build_list_filter('owner_id')

        if 'owner_id' in filters:
            filters['admin_uid'] = filters['owner_id']
            del filters['owner_id']

        return filters


class AdminOrganizationDetailView(View):
    @permission_required([common_internal_permission.organizations_detail_list_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _, org_id):
        org_id = ensure_integer(org_id, 'org_id')
        meta = OrganizationMetaModel(meta_connection).find(
            {'id': org_id},
            fields=['id', 'shard', 'ready', 'cloud_org_id'],
            one=True,
        )
        if not meta:
            return json_error_not_found()
        shard = meta['shard']
        with get_main_connection(shard=shard) as main_connection:
            organization = OrganizationModel(main_connection).get(
                id=org_id,
                fields=['*', 'billing_info.client_id', 'billing_info.person_id']
            )
            client_id = None
            person_id = None
            balance = None
            first_debt_act_date = None
            if organization.get('billing_info'):
                client_id = organization['billing_info']['client_id']
                person_id = organization['billing_info']['person_id']
                try:
                    balance_info = app.billing_client.get_balance_info(client_id)
                    balance = balance_info['balance']
                    first_debt_act_date = balance_info['first_debt_act_date']
                except BillingException:
                    pass

            del organization['billing_info']

            organization['shard'] = shard
            organization['cloud_org_id'] = meta['cloud_org_id']
            organization['users_count'] = UserModel(main_connection).count(
                filter_data={
                    'org_id': org_id,
                    'user_type': 'user',
                }
            )
            organization['robots_count'] = UserModel(main_connection).count(
                filter_data={
                    'org_id': org_id,
                    'user_type__notequal': 'user',
                }
            )
            organization['departments_count'] = DepartmentModel(main_connection).count(filter_data={'org_id': org_id})
            organization['groups_count'] = GroupModel(main_connection).count(filter_data={'org_id': org_id})
            organization['domains'] = get_domains_by_org_id(meta_connection, main_connection, org_id)
            organization['billing'] = {
                'balance': balance,
                'client_id': client_id,
                'person_id': person_id,
                'first_debt_act_date': format_date(first_debt_act_date, allow_none=True),
            }

        return json_response(
            _prepare_organization(organization, shard),
        )


class AdminOrganizationServicesListView(View):

    @permission_required([common_internal_permission.organizations_services_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _, org_id):
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard) as main_connection:
            data = {}
            for service in OrganizationServiceModel(main_connection) \
                    .filter(org_id=org_id, enabled=Ignore) \
                    .fields('**'):

                data[service['slug']] = AdminOrganizationServicesDetailView \
                    .format_service(main_connection, org_id, service)

        return json_response(data)


class AdminOrganizationServicesDetailView(View):
    @staticmethod
    def format_service(main_connection, org_id, service):
        trial_expired = None
        if service['trial_expires']:
            trial_expired = service['trial_expires'] < utcnow().date()

        user_count = None
        if service['paid_by_license']:
            user_count = UserServiceLicenses(main_connection) \
                .filter(org_id=org_id, service_id=service['service_id']) \
                .count()

        if service['slug'] == 'tracker':
            resources_count = 0
        else:
            resources_count = ResourceModel(main_connection).filter(
                org_id=org_id, service=service['slug']
            ).count()

        return {
            'paid_by_license': service['paid_by_license'],
            'enabled': service['enabled'],
            'ready': service['ready'],
            'ready_at': format_date(service['ready_at'], allow_none=True),
            'trial_expires': format_date(service['trial_expires'], allow_none=True),
            'trial_expired': trial_expired,
            'disable_reason': service['disable_reason'],
            'enabled_at': format_date(service['enabled_at'], allow_none=True),
            'disabled_at': format_date(service['disabled_at'], allow_none=True),
            'last_mail_sent_at': format_date(service['last_mail_sent_at'], allow_none=True),
            'user_count': user_count,
            'user_limit': service['user_limit'],
            'expires_at': format_date(service['expires_at'], allow_none=True),
            'resources_count': resources_count,
            'responsible_id': service['responsible_id'],
        }

    @permission_required([common_internal_permission.organizations_services_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _, org_id, service_slug):
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard) as main_connection:
            service = OrganizationServiceModel(main_connection) \
                .get_by_slug(org_id, service_slug, fields=['**'])

            data = self.format_service(main_connection, org_id, service)

        return json_response(data)

    @uses_schema(CHANGE_SERVICE_LIMITS_SCHEMA)
    @permission_required([bizdev_internal_permission.change_organization_subscription_plan])
    @requires(org_id=False, user=True)
    def patch(self, meta_connection, _, data, org_id, service_slug):
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard, for_write=True) as main_connection:
            OrganizationModel(main_connection).require_partner(org_id)

            service = ServiceModel(meta_connection) \
                .get_licensed_service_by_slug(service_slug)

            org_service = OrganizationServiceModel(main_connection) \
                .get_by_org_service(org_id, service['id'], fields=['**'])

            update_data = {}
            if 'user_limit' in data:
                update_data['user_limit'] = data['user_limit']
            if 'expires_at' in data:
                update_data['expires_at'] = parse_date(data['expires_at'])

            OrganizationServiceModel(main_connection) \
                .update_one(org_service['id'], update_data)

            org_service.update(update_data)

            SupportActionMetaModel(meta_connection).create(
                org_id,
                SupportActions.change_service_limits,
                g.user.passport_uid,
                org_service,
                'organization_service',
                data['comment'],
            )

            data = self.format_service(main_connection, org_id, org_service)

        return json_response(data)


class AdminOrganizationServicesEnableView(View):
    @uses_schema(ENABLE_SERVICE_LIMITS_SCHEMA)
    @permission_required([bizdev_internal_permission.change_organization_subscription_plan])
    @requires(org_id=False, user=True)
    def post(self, meta_connection, _, data, org_id, service_slug):
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard, for_write=True) as main_connection:
            is_partner = OrganizationModel(main_connection).is_partner_organization(org_id)

            update_data = {
                'trial_status': trial_status.expired,
                'trial_expires': (utcnow() - relativedelta(days=1)).date(),
            }
            for param_name in ['user_limit', 'expires_at']:
                if param_name in data:
                    update_data[param_name] = data[param_name]
                elif is_partner and param_name not in data:
                    raise RequiredFieldsMissed([param_name])

            if 'expires_at' in update_data:
                update_data['expires_at'] = parse_date(update_data['expires_at'])

            service = ServiceModel(meta_connection) \
                .get_licensed_service_by_slug(service_slug)

            enable_service(
                meta_connection=meta_connection,
                main_connection=main_connection,
                org_id=org_id,
                service_slug=service_slug,
                author_id=g.user.passport_uid,
                drop_licenses=True,
            )

            org_service = OrganizationServiceModel(main_connection) \
                .get_by_org_service(org_id, service['id'], fields=['**'])

            OrganizationServiceModel(main_connection) \
                .update_one(org_service['id'], update_data)

            action_service_trial_end(
                main_connection,
                org_id=org_service['org_id'],
                author_id=None,
                object_value=service,
            )

            org_service.update(update_data)

            SupportActionMetaModel(meta_connection).create(
                org_id,
                SupportActions.change_service_limits,
                g.user.passport_uid,
                org_service,
                'organization_service',
                data['comment'],
            )

            data = AdminOrganizationServicesDetailView \
                .format_service(main_connection, org_id, org_service)

        return json_response(data, status_code=201)


class AdminOrganizationServicesDisableView(View):
    @uses_schema(DISABLE_SERVICE_LIMITS_SCHEMA)
    @permission_required([bizdev_internal_permission.change_organization_subscription_plan])
    @requires(org_id=False, user=True)
    def post(self, meta_connection, _, data, org_id, service_slug):
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard, for_write=True) as main_connection:

            service = ServiceModel(meta_connection) \
                .get_licensed_service_by_slug(service_slug)

            org_service = OrganizationServiceModel(main_connection) \
                .get_by_org_service(org_id, service['id'], fields=['**'])

            update_data = {
                'user_limit': None,
                'expires_at': None,
            }

            OrganizationServiceModel(main_connection) \
                .update_one(org_service['id'], update_data)

            if org_service['enabled']:
                disable_service(
                    meta_connection=meta_connection,
                    main_connection=main_connection,
                    org_id=org_id,
                    service_slug=service_slug,
                    disable_reason=reason.disabled_by_user,
                    author_id=g.user.passport_uid,
                )

            org_service.update(update_data)

            SupportActionMetaModel(meta_connection).create(
                org_id,
                SupportActions.change_service_limits,
                g.user.passport_uid,
                org_service,
                'organization_service',
                data['comment'],
            )

            data = AdminOrganizationServicesDetailView \
                .format_service(main_connection, org_id, org_service)

        return json_response(data, status_code=201)


class AdminOrganizationAdminsListView(View):
    @permission_required([common_internal_permission.organizations_admins_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _, org_id):
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)
        all_admins = []
        outer_admin = UserMetaModel(meta_connection).get_outer_admins(org_id=org_id, fields=['id'], one=True)
        if outer_admin:
            outer_admin_info = get_user_data_from_blackbox_by_uid(outer_admin['id'])
            outer_admin_info['is_outer'] = True
            all_admins.append(
                outer_admin_info
            )

        with get_main_connection(shard=shard) as main_connection:
            inner_admins = OrganizationModel(main_connection).get_admins(org_id)

            all_admins.extend(
                [prepare_user(
                        main_connection,
                        u,
                        expand_contacts=True,
                        api_version=1
                    ) for u in inner_admins]
            )

        return json_response(all_admins)


def _prepare_organization(organization, shard=None):
    organization['created'] = format_datetime(organization['created'])
    if shard:
        organization['shard'] = shard
    return organization


class AdminOrganizationChangeType(View):
    @uses_schema(CHANGE_ORG_TYPE_SCHEMA)
    @permission_required([support_internal_permission.change_organization_type])
    @requires(org_id=False, user=True)
    def patch(self, meta_connection, _, data, org_id):
        """
        Меняем тип организации на один из: education, yandex_project, test, common, partner
        """
        org_id = ensure_integer(org_id, 'org_id')
        new_org_type = data['org_type']
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(for_write=True, shard=shard) as main_connection:
            OrganizationModel(main_connection).change_organization_type(
                org_id,
                new_org_type,
                author_id=g.user.passport_uid,
                client_id=data.get('client_id'),
                person_id=data.get('person_id'),
            )
            organization = OrganizationModel(main_connection).get(org_id)

            SupportActionMetaModel(meta_connection).create(
                org_id,
                SupportActions.change_organization_type,
                g.user.passport_uid,
                organization,
                'organization',
                data['comment'],
            )
            return json_response(
                _prepare_organization(organization, shard),
            )


class AdminOrganizationChangeSubscriptionPlanView(View):
    @uses_schema(CHANGE_ORG_SUBSCRIPTION_PLAN)
    @permission_required([bizdev_internal_permission.change_organization_subscription_plan])
    @requires(org_id=False, user=True)
    def patch(self, meta_connection, _, data, org_id):
        """
        Меняем тарифный план организации и дату его окончания
        """
        org_id = ensure_integer(org_id, 'org_id')
        subscription_plan = data.get('subscription_plan')
        subscription_plan_expires_at = parse_date(data.get('subscription_plan_expires_at'))
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(for_write=True, shard=shard) as main_connection:
            model = OrganizationModel(main_connection)

            if not model.is_partner_organization(org_id):
                raise OrganizationIsWithoutContract()

            if subscription_plan is None:
                model.update(
                    filter_data={
                        'id': org_id,
                        'subscription_plan': 'paid',
                    },
                    update_data={
                        'subscription_plan_expires_at': subscription_plan_expires_at
                    },
                )
            elif subscription_plan == 'paid':
                model.enable_paid_mode_for_partner_organization(
                    org_id=org_id,
                    author_id=g.user.passport_uid,
                    expires_at=subscription_plan_expires_at,
                )
            elif subscription_plan == 'free':
                model.disable_paid_mode(
                    org_id=org_id,
                    author_id=g.user.passport_uid,
                )

            organization = model.get(org_id)

            SupportActionMetaModel(meta_connection).create(
                org_id,
                SupportActions.change_subscription_plan,
                g.user.passport_uid,
                organization,
                'organization',
                data['comment'],
            )
            return json_response(
                _prepare_organization(organization, shard),
            )


class AdminOrganizationVerifyDomainView(View):
    @permission_required([support_internal_permission.verify_domain])
    @uses_schema(VERIFY_DOMAIN_SCHEMA)
    @requires(org_id=False, user=False)
    def post(self, meta_connection, _, data, org_id, domain_name):
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(for_write=True, shard=shard) as main_connection:
            # проверим, что домен данной организации не подтвержден
            domain = get_domains_from_db_or_domenator(
                meta_connection=meta_connection,
                domain_filter=DomainFilter(name=domain_name, org_id=org_id),
                main_connection=main_connection,
                one=True,
            )
            if domain['owned']:
                raise InvalidValue(message='Organization already owns this domain')

            organization = OrganizationModel(main_connection).get(org_id)

            # проверим наличие другой организации, у которой подтвержден данный домен
            # и если есть такая, сделаем ее домен неподтвержденным
            other_domain = get_domains_from_db_or_domenator(
                meta_connection=meta_connection,
                domain_filter=DomainFilter(name=domain_name, owned=True),
                main_connection=main_connection,
                one=True,
            )
            if other_domain:
                update_domains_in_db_or_domenator(
                    meta_connection,
                    DomainUpdateFilter(name=domain_name, org_id=other_domain['org_id']),
                    DomainUpdateData(owned=False),
                    main_connection=main_connection,
                )
                action_domain_alienate(
                    main_connection,
                    org_id=other_domain['org_id'],
                    author_id=organization['admin_uid'],
                    object_value=other_domain,
                    old_object=None,
                )

            # сделаем домен данной организации подтвержденным
            set_domain_as_owned(
                meta_connection,
                main_connection,
                org_id,
                organization['admin_uid'],
                domain_name,
            )

            SupportActionMetaModel(meta_connection).create(
                org_id,
                SupportActions.verify_domain,
                g.user.passport_uid,
                {},
                'organization',
                data['comment'],
            )

        return json_response(
            data={},
            status_code=201,
        )


class AdminOrganizationTotalCount(View):
    @permission_required([common_internal_permission.organizations_count])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _,):
        shards = get_shard_numbers()
        organizations_total_count = 0
        users_total_count = 0
        for shard in shards:
            with get_main_connection(shard=shard) as main_connection:
                organizations_total_count += OrganizationModel(main_connection).count()
                users_total_count += UserModel(main_connection).count(
                    filter_data={'is_robot': False}
                )

        return json_response({
                'organizations_total_count': organizations_total_count,
                'users_total_count': users_total_count
        })


class AdminOrganizationMaillistsSyncView(View):
    @permission_required([support_internal_permission.organizations_sync_maillist])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _, org_id):
        """
        Отдает статус последней проверки разъезжания рассылок в Директории и BigML для данной организации
        или None, если не было проверки еще

        Пример:

            {
                'updated_at': '2018-09-07T00:06:37.259579+12:00',
                'org_id': 278,
                'ml_is_ok': False,
                'revision': 10,
                'problems': 'Department (all@, 1130000023567860) has problems
                             This users are not subscribed: 1137733427497374
                             This users in department: 1130000099999374'
            }

        """
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard) as main_connection:
            last_check = MaillistCheckModel(main_connection).get(org_id)
            return json_response(last_check)

    @permission_required([support_internal_permission.organizations_sync_maillist])
    @requires(org_id=False, user=False)
    def post(self, meta_connection, _, org_id):
        """
        Синхронизирует рассылки организации через сервис рассылок BigML
        """
        org_id = ensure_integer(org_id, 'org_id')

        with log.name_and_fields('bigml-sync', org_id=org_id):
            shard = get_shard(meta_connection, org_id)
            with get_main_connection(shard=shard, for_write=True) as main_connection:
                MaillistsCheckTask(main_connection).delay(org_id=org_id)

        return json_response(
            data={},
            status_code=201,
        )


class AdminOrganizationStaffListView(View):
    @permission_required([common_internal_permission.organizations_staff_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _, org_id):
        """
        Отдает список контейнеров (пользователей, отделов, групп) организации

        Формат ответа:
            [
                {
                    'type': 'user',
                    'object': {'id': ...}
                },
                {
                    'type': 'department',
                    'object': {'id': ...}
                },
                {
                    'type': 'group',
                    'object': {'id': ...}
                },
                ...
            ]


        #### Фильтрация

        Фильтрация по типу контейнера type (user | department | group):
            /admin/organizations/<org_id>/staff/?type=department

            Если указан type, можно использовать фильтр по id контейнера:
                /admin/organizations/<org_id>/staff/?type=group&id=2

            При type=user можно использовать фильтр role (admin | deputy_admin | user | robot | yamb_bot)
                (фильтрация по пользователем с указанной ролью):
                    /admin/organizations/<org_id>/staff/?type=user&role=user

            При type=group можно использвать фильтр groups.type
                (robots | department_head | tracker-separator-alias | organization_admin
                        | generic | organization_deputy_admin | sergo_group)
                (фильтрация по типу группы):
                    admin/organizations/<org_id>/staff/?type=group&groups.type=generic

        Фильтрация по контейнерам, вложенным в данный контейнер:
            /admin/organizations/<org_id>/staff/?parent_id=111&parent_type=group
                вернет всех пользователей, отделов и групп, непосредственно входящих в группу с id=111

        Фильтрация по контейнерам, членом которого является данный контейнер:
            /admin/organizations/<org_id>/staff/?member_id=111&member_type=department
                вернет все отделы и группы, в которые непосредственно входит отдел с id=111

        Фильтрация по контейнерам, подписанным на указанный сервис:
            admin/organizations/<org_id>/staff/?services.slug=tracker

        В любой из фильтров можно добавить фильтр:
            Фильтрация по текстовым полям контейнеров
                    (как в suggest: фильтр проходит, если есть слово, начинающееся с данного):
                admin/organizations/<org_id>/staff/?text=admin


        Возвращаемые контейнеры сортируются по nickname для пользователей и по label для групп и отделов

        ---
        tags:
          - Админка
        responses:
          200:
            description: Список контейнеров организации
          404:
            description: Контейнер или сервис не найден

        """
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(for_write=False, shard=shard) as main_connection:
            query_params = request.args.to_dict()

            if 'type' in query_params:
                params = {}
                if query_params['type'] == 'user':
                    params['role'] = query_params.get('role')
                if query_params['type'] == 'group':
                    params['groups_type'] = query_params.get('groups.type')
                if query_params.get('id'):
                    container_id = ensure_integer(query_params['id'], 'id')
                    params['container_id'] = container_id

                return self.find_containers_of_type(
                    meta_connection,
                    main_connection,
                    org_id=org_id,
                    type=query_params['type'],
                    text=query_params.get('text'),
                    **params
                )

            if 'parent_id' in query_params and query_params.get('parent_type') == 'group':
                parent_id = ensure_integer(query_params['parent_id'], 'parent_id')
                return self.find_group_members(
                    main_connection,
                    org_id=org_id,
                    parent_id=parent_id,
                    text=query_params.get('text'),
                )

            if 'parent_id' in query_params and query_params.get('parent_type') == 'department':
                parent_id = ensure_integer(query_params['parent_id'], 'parent_id')
                return self.find_department_members(
                    main_connection,
                    org_id=org_id,
                    parent_id=parent_id,
                    text=query_params.get('text'),
                )

            if 'member_id' in query_params and query_params.get('member_type') == 'user':
                member_id = ensure_integer(query_params['member_id'], 'member_id')
                return self.find_user_parents(
                    main_connection,
                    org_id=org_id,
                    member_id=member_id,
                    text=query_params.get('text'),
                )

            if 'member_id' in query_params and query_params.get('member_type') == 'department':
                member_id = ensure_integer(query_params['member_id'], 'member_id')
                return self.find_department_parents(
                    main_connection,
                    org_id=org_id,
                    member_id=member_id,
                    text=query_params.get('text'),
                )

            if 'member_id' in query_params and query_params.get('member_type') == 'group':
                member_id = ensure_integer(query_params['member_id'], 'member_id')
                return self.find_group_parents(
                    main_connection,
                    org_id=org_id,
                    member_id=member_id,
                    text=query_params.get('text'),
                )

            if 'services.slug' in query_params:
                return self.find_service_subscribers(
                    main_connection,
                    org_id=org_id,
                    services_slug=query_params['services.slug'],
                    text=query_params.get('text'),
                )

            return self.find_all_containers(
                main_connection,
                org_id=org_id,
                text=query_params.get('text'),
            )

    def find_containers_of_type(self,
                                meta_connection,
                                main_connection,
                                org_id,
                                type,
                                container_id=None,
                                text=None,
                                role=None,
                                groups_type=None):
        filter_data = {
            'org_id': org_id,
        }
        fields = ['*']
        if text:
            filter_data['suggest'] = text
        if container_id:
            filter_data['id'] = container_id

        if type == 'user':
            if role in ['robot', 'yamb_bot']:
                filter_data['user_type'] = role
            elif role:
                filter_data['role'] = role
                filter_data['user_type'] = 'user'

        if type == 'group':
            if groups_type:
                filter_data['type'] = groups_type

        if type == 'user':
            order_by = 'nickname'
            filter_data['is_dismissed'] = False
            fields.append('karma')
            fields.append('avatar_id')
        else:
            order_by = 'label'
            filter_data['removed'] = False

        prepare_result_item_func = lambda object: self.prepare_container(object, type=type)
        model = get_model_by_type(type)
        response = build_list_response(
            model=model(main_connection),
            model_fields=fields,
            model_filters=filter_data,
            prepare_result_item_func=prepare_result_item_func,
            path=request.path,
            query_params=request.args.to_dict(),
            max_per_page=CONTAINERS_MAX_PER_PAGE,
            order_by=order_by,
        )

        # добавим внешних админов и замов
        if type == 'user':
            if role in ['admin', 'deputy_admin']:
                # добавим админов только на последнюю страницу
                if not response['data']['pages'] or response['data']['page'] == response['data']['pages']:
                    if role == 'admin':
                        outer_admin_ids = UserMetaModel(meta_connection).get_outer_admins(org_id=org_id, fields=['id'])
                    else:
                        outer_admin_ids = UserMetaModel(meta_connection).get_outer_deputy_admins(org_id=org_id, fields=['id'])

                    # сходим в паспорт за расширенной инфо об админах
                    outer_admins = []
                    for outer_admin in outer_admin_ids:
                        if container_id:
                            if outer_admin['id'] != container_id:
                                continue

                        outer_admin_info = get_user_data_from_blackbox_by_uid(outer_admin['id'])

                        if text:
                            if not self.outer_admin_contains_text(text, outer_admin_info):
                                continue

                        outer_admins.append(
                            prepare_result_item_func(
                                self.convert_passport_user_to_connect(
                                    outer_admin_info,
                                    org_id,
                                )
                            )
                        )

                    response['data']['result'].extend(outer_admins)
                # 'total' будет неправльное, так как мы внешних админов и замов добавляем только на последнюю страницу,
                # а на остальных не считаем, поэтому просто удалаем это поле
                del response['data']['total']

        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def outer_admin_contains_text(self, text, outer_admin_info):
        search_fields = [outer_admin_info['first_name'], outer_admin_info['last_name'], outer_admin_info['login']]
        for field in search_fields:
            words = field.lower().split()
            for word in words:
                if word.startswith(text.lower()):
                    return True
        return False

    def convert_passport_user_to_connect(self, passport_user, org_id):
        gender_map = {
            None: 'unspecified',
            '0': 'unspecified',
            '1': 'male',
            '2': 'female',
        }
        return {
            'id': passport_user.get('uid'),
            'org_id': org_id,
            'login': None,
            'email': passport_user.get('default_email'),
            'department_id': None,
            'name': None,
            'gender': gender_map[passport_user.get('sex')],
            'karma': int_or_none(passport_user.get('karma')),
            'position': None,
            'about': None,
            'birthday': passport_user.get('birth_date'),
            'contacts': None,
            'aliases': passport_user.get('aliases'),
            'nickname': passport_user.get('login'),
            'is_dissmissed': False,
            'external_id': None,
            'user_type': 'user',
            'created': None,
            'first_name': passport_user.get('first_name'),
            'last_name': passport_user.get('last_name'),
            'middle_name': None,
            'position_plain': None,
            'recovery_email': None,
            'external': True,
        }

    def make_text_filter(self, main_connection, text, type):
        if not text:
            return ''

        if type == 'user':
            return mogrify(
                main_connection,
                'AND make_user_search_ts_vector2(users.first_name, users.last_name, users.nickname) @@ %('
                'text)s::tsquery',
                {
                    'text': prepare_for_tsquery(text),
                }
            )
        if type == 'department':
            return mogrify(
                main_connection,
                'AND make_user_search_ts_vector2(departments.name_plain, departments.label, \'\') @@ %(text)s::tsquery',
                {
                    'text': prepare_for_tsquery(text),
                }
            )
        if type == 'group':
            return mogrify(
                main_connection,
                'AND make_user_search_ts_vector2(groups.name_plain, groups.label, \'\') @@ %(text)s::tsquery',
                {
                    'text': prepare_for_tsquery(text),
                }
            )

    def find_all_containers(self, main_connection, org_id, text=None):
        container_queries = []
        user_query = '''
    SELECT 'user' as type, id, nickname as label
    FROM users
    WHERE org_id = {org_id}
    AND NOT is_dismissed
    {text_filter}
        '''
        user_query = user_query.format(
            org_id=org_id,
            text_filter=self.make_text_filter(main_connection, text, type='user'),
        )
        container_queries.append(user_query)

        department_query = '''
    SELECT 'department' as type, id, label
    FROM departments
    WHERE org_id = {org_id}
    AND NOT removed
    {text_filter}
        '''
        department_query = department_query.format(
            org_id=org_id,
            text_filter=self.make_text_filter(main_connection, text, type='department'),
        )
        container_queries.append(department_query)

        group_query = '''
    SELECT 'group' as type, id, label
    FROM groups
    WHERE org_id = {org_id}
    AND NOT removed
    {text_filter}
        '''
        group_query = group_query.format(
            org_id=org_id,
            text_filter=self.make_text_filter(main_connection, text, type='group'),
        )
        container_queries.append(group_query)

        response = self.add_paginator(
            main_connection,
            org_id,
            container_queries,
            max_per_page=CONTAINERS_MAX_PER_PAGE,
        )
        add_avatar_id_for_user_objects(response['data']['result'])
        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def find_group_members(self, main_connection, org_id, parent_id, text=None):
        group = get_object_or_404(
            model_instance=GroupModel(main_connection),
            org_id=org_id,
            group_id=parent_id,
        )
        return self.find_resources(main_connection, org_id, group['resource_id'], relation_name=relation_name.include, text=text)

    def find_service_subscribers(self, main_connection, org_id, services_slug, text=None):
        resource_id = OrganizationServiceModel(main_connection) \
                      .get_licensed_service_resource_id(org_id, services_slug)
        return self.find_resources(main_connection, org_id, resource_id, relation_name=relation_name.member, text=text)

    def find_resources(self, main_connection, org_id, resource_id, relation_name, text=None):
        container_queries = []

        user_query = '''
    SELECT 'user' as type, users.id, users.nickname as label
    FROM users
    INNER JOIN resource_relations
            ON resource_relations.user_id = users.id
           AND resource_relations.org_id = users.org_id
    WHERE resource_relations.org_id = {org_id}
    AND resource_relations.resource_id = '{resource_id}'
    AND resource_relations.name = '{relation_name}'
    AND NOT users.is_dismissed
    {text_filter}
        '''
        user_query = user_query.format(
            org_id=org_id,
            resource_id=resource_id,
            relation_name=relation_name,
            text_filter=self.make_text_filter(main_connection, text, type='user'),
        )
        container_queries.append(user_query)

        department_query = '''
    SELECT 'department' as type, departments.id, departments.label as label
    FROM departments
    INNER JOIN resource_relations
            ON resource_relations.department_id = departments.id
           AND resource_relations.org_id = departments.org_id
    WHERE resource_relations.org_id = {org_id}
    AND resource_relations.resource_id = '{resource_id}'
    AND resource_relations.name = '{relation_name}'
    AND NOT departments.removed
    {text_filter}
        '''
        department_query = department_query.format(
            org_id=org_id,
            resource_id=resource_id,
            relation_name=relation_name,
            text_filter=self.make_text_filter(main_connection, text, type='department'),
        )
        container_queries.append(department_query)

        group_query = '''
    SELECT 'group' as type, groups.id, groups.label as label
    FROM groups
    INNER JOIN resource_relations
            ON resource_relations.group_id = groups.id
           AND resource_relations.org_id = groups.org_id
    WHERE resource_relations.org_id = {org_id}
    AND resource_relations.resource_id = '{resource_id}'
    AND resource_relations.name = '{relation_name}'
    AND NOT groups.removed
    {text_filter}
        '''
        group_query = group_query.format(
            org_id=org_id,
            resource_id=resource_id,
            relation_name=relation_name,
            text_filter=self.make_text_filter(main_connection, text, type='group'),
        )
        container_queries.append(group_query)

        response = self.add_paginator(
            main_connection,
            org_id,
            container_queries,
            max_per_page=CONTAINERS_MAX_PER_PAGE,
        )
        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def find_department_members(self, main_connection, org_id, parent_id, text=None):
        get_object_or_404(
            model_instance=DepartmentModel(main_connection),
            org_id=org_id,
            department_id=parent_id,
        )

        container_queries = []

        user_query = '''
    SELECT 'user' as type, users.id, users.nickname as label
    FROM users
    WHERE users.org_id = {org_id}
    AND users.department_id = {parent_id}
    AND NOT users.is_dismissed
    {text_filter}
        '''
        user_query = user_query.format(
            org_id=org_id,
            parent_id=parent_id,
            text_filter=self.make_text_filter(main_connection, text, type='user'),
        )
        container_queries.append(user_query)

        department_query = '''
    SELECT 'department' as type, departments.id, departments.label as label
    FROM departments
    WHERE departments.org_id = {org_id}
    AND departments.parent_id = {parent_id}
    AND NOT departments.removed
    {text_filter}
        '''
        department_query = department_query.format(
            org_id=org_id,
            parent_id=parent_id,
            text_filter=self.make_text_filter(main_connection, text, type='department'),
        )
        container_queries.append(department_query)

        response = self.add_paginator(
            main_connection,
            org_id,
            container_queries,
            max_per_page=CONTAINERS_MAX_PER_PAGE,
        )
        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def find_user_parents(self, main_connection, org_id, member_id, text=None):
        user = get_object_or_404(
            model_instance=UserModel(main_connection),
            org_id=org_id,
            user_id=member_id,
        )
        container_queries = []

        department_query = '''
    SELECT 'department' as type, departments.id, departments.label as label
    FROM departments
    WHERE departments.org_id = {org_id}
    AND departments.id = {member_id}
    AND NOT departments.removed
    {text_filter}
        '''
        department_query = department_query.format(
            org_id=org_id,
            member_id=user['department_id'],
            text_filter=self.make_text_filter(main_connection, text, type='department'),
        )
        container_queries.append(department_query)

        group_query = '''
    SELECT 'group' as type, groups.id, groups.label as label
    FROM groups
    INNER JOIN resource_relations
            ON resource_relations.resource_id = groups.resource_id
           AND resource_relations.org_id = groups.org_id
    WHERE resource_relations.org_id = {org_id}
    AND resource_relations.user_id = {member_id}
    AND resource_relations.name = '{relation_name}'
    AND NOT groups.removed
    {text_filter}
        '''
        group_query = group_query.format(
            org_id=org_id,
            member_id=member_id,
            relation_name=relation_name.include,
            text_filter=self.make_text_filter(main_connection, text, type='group'),
        )
        container_queries.append(group_query)

        response = self.add_paginator(
            main_connection,
            org_id,
            container_queries,
            max_per_page=CONTAINERS_MAX_PER_PAGE,
        )
        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def find_department_parents(self, main_connection, org_id, member_id, text=None):
        department = get_object_or_404(
            model_instance=DepartmentModel(main_connection),
            org_id=org_id,
            department_id=member_id,
        )
        container_queries = []

        department_query = '''
    SELECT 'department' as type, departments.id, departments.label as label
    FROM departments
    WHERE departments.org_id = {org_id}
    AND departments.id = {member_id}
    AND NOT departments.removed
    {text_filter}
        '''
        department_query = department_query.format(
            org_id=org_id,
            member_id=department['parent_id'],
            text_filter=self.make_text_filter(main_connection, text, type='department'),
        )
        container_queries.append(department_query)

        group_query = '''
    SELECT 'group' as type, groups.id, groups.label as label
    FROM groups
    INNER JOIN resource_relations
            ON resource_relations.resource_id = groups.resource_id
           AND resource_relations.org_id = groups.org_id
    WHERE resource_relations.org_id = {org_id}
    AND resource_relations.department_id = {member_id}
    AND resource_relations.name = '{relation_name}'
    AND NOT groups.removed
    {text_filter}
        '''
        group_query = group_query.format(
            org_id=org_id,
            member_id=member_id,
            relation_name=relation_name.include,
            text_filter=self.make_text_filter(main_connection, text, type='group'),
        )
        container_queries.append(group_query)

        response = self.add_paginator(
            main_connection,
            org_id,
            container_queries,
            max_per_page=CONTAINERS_MAX_PER_PAGE,
        )
        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def find_group_parents(self, main_connection, org_id, member_id, text=None):
        get_object_or_404(
            model_instance=GroupModel(main_connection),
            org_id=org_id,
            group_id=member_id,
        )

        group_query = '''
    SELECT 'group' as type, groups.id, groups.label as label
    FROM groups
    INNER JOIN resource_relations
            ON resource_relations.resource_id = groups.resource_id
           AND resource_relations.org_id = groups.org_id
    WHERE resource_relations.org_id = {org_id}
    AND resource_relations.group_id = {member_id}
    AND resource_relations.name = '{relation_name}'
    AND NOT groups.removed
    {text_filter}
        '''
        group_query = group_query.format(
            org_id=org_id,
            member_id=member_id,
            relation_name=relation_name.include,
            text_filter=self.make_text_filter(main_connection, text, type='group'),
        )
        container_queries = [group_query]

        response = self.add_paginator(
            main_connection,
            org_id,
            container_queries,
            max_per_page=CONTAINERS_MAX_PER_PAGE,
        )
        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def add_paginator(self, main_connection, org_id, container_queries, max_per_page=None):

        query_params = request.args.to_dict()
        body_query = '\n    UNION ALL\n'.join(container_queries)

        count_query = '''
SELECT count(*)
FROM (
{body_query}
) as t
        '''
        count_query = count_query.format(body_query=body_query)
        total_count = main_connection.execute(count_query).fetchall()[0][0]
        paginator = Paginator(
            total=total_count,
            page=_get_int_or_none(query_params.get(app.config['PAGINATION']['page_query_param'])),
            per_page=_get_int_or_none(query_params.get(app.config['PAGINATION']['per_page_query_param'])),
            path=request.path,
            query_params=query_params,
            max_per_page=max_per_page,
        )
        if paginator.page > 1:
            skip = (paginator.page - 1) * paginator.per_page
        else:
            skip = 0

        limit = paginator.per_page
        response = paginator.response

        query = '''
SELECT *
FROM (
{body_query}
) as t
ORDER BY t.label
LIMIT {limit} OFFSET {offset}
        '''
        query = query.format(body_query=body_query, limit=limit, offset=skip)
        containers = main_connection.execute(query).fetchall()
        response['data']['result'] = self.prepare_containers(main_connection, containers, org_id)

        return response

    def prepare_containers(self, main_connection, containers, org_id):
        """
        Метод получает список контейнеров (type, id, label) на входе и возвращает список словарей по типу:
            [
                {
                    'type': 'user',
                    'object': {'id': ...}
                },
                {
                    'type': 'department',
                    'object': {'id': ...}
                },
                {
                    'type': 'group',
                    'object': {'id': ...}
                },
                ...
            ]
        """
        result = []
        for type, id, label in containers:
            model = get_model_by_type(type)
            result.append(
                self.prepare_container(
                    object=model(main_connection).find(
                        filter_data={
                            'id': id,
                            'org_id': org_id,
                        },
                        one=True,
                    ),
                    type=type,
                )
            )
        return result

    def prepare_container(self, object, type):
        return {
            'type': type,
            'object': object,
        }


class AdminOrganizationServicesSubscribersView(View):
    @permission_required([common_internal_permission.organizations_services_subscribers_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _, org_id, slug):
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)
        service_id = ServiceModel(meta_connection).get_licensed_service_by_slug(slug)['id']

        with get_main_connection(shard=shard) as main_connection:
            def expand_prepare_user(u):
                return prepare_user(
                    main_connection,
                    u,
                    expand_contacts=True,
                    api_version=1,
                )
            uids_with_licenses = only_attrs(UserServiceLicenses(main_connection).find(
                {
                    'org_id': org_id,
                    'service_id': service_id,
                },
                distinct=True,
            ), 'user_id')
            filters = {
                'org_id': org_id,
                'id': uids_with_licenses,
            }
            query = request.args.get('query')
            if query:
                filters['suggest'] = query

            response = build_list_response(
                model=UserModel(main_connection),
                path=request.path,
                model_filters=filters,
                query_params=request.args.to_dict(),
                prepare_result_item_func=expand_prepare_user,
                max_per_page=app.config['PAGINATION']['max_per_page_for_admin'],
                order_by=self._get_ordering_fields(),
            )

        return json_response(
            response['data'],
            headers=response['headers'],
        )


def _prepare_action(item):
    item['timestamp'] = format_datetime(item['timestamp'])
    return item


class AdminOrganizationsActionsListView(View):
    allowed_ordering_fields = ['id', 'timestamp', 'revision']

    @permission_required([support_internal_permission.get_actions])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _, org_id):
        """
        Пример выдачи:
            {
                ...
                "result": [
                    {
                        "author_id": 1134522881604289,
                        "id": 13577,
                        "name": "service_license_change",
                        "object_type": "organization",
                        "object": {"id": 123, "label": "123", ...},
                        "old_object": {},
                        "org_id": 8090,
                        "timestamp": "2017-10-20T14:15:23.164343Z"
                    }
                    ...
                ]
            }

        """
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)

        with get_main_connection(shard=shard) as main_connection:
            action_model = ActionModel(main_connection)
            response = build_list_response(
                model=action_model,
                path=request.path,
                model_filters=self._get_filters(org_id),
                query_params=request.args.to_dict(),
                prepare_result_item_func=_prepare_action,
                max_per_page=app.config['PAGINATION']['max_per_page_for_admin'],
                order_by=self._get_ordering_fields(),
            )
        return json_response(
            response['data'],
            headers=response['headers'],
            allowed_sensitive_params=['can_users_change_password'],
        )

    def _get_filters(self, org_id):
        filters = {'org_id': org_id}

        def _build_list_filter(field_name):
            if request.args.get(field_name):
                values = request.args.get(field_name).split(',')
                if len(values) == 1:
                    values = values[0]
                filters[field_name] = values

        _build_list_filter('name')

        nickname = request.args.get('nickname')
        if nickname:
            filters['nickname'] = nickname

        return filters


def _prepare_domain(item):
    item['created_at'] = format_date(item['created_at'])
    item['validated_at'] = format_date(item['validated_at'], allow_none=True)
    return item


class AdminOrganizationDomainsView(View):
    @permission_required([common_internal_permission.organizations_domains_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _, org_id):
        org_id = ensure_integer(org_id, 'org_id')

        if is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
            domains = app.domenator.private_get_domains(org_id=org_id)
            length = len(domains)
            return json_response({
                'result': domains,
                'page': 1,
                'per_page': length,
                'pages': 1,
                'total': length,
                'links': {},
                'multishard': False,
            })

        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard) as main_connection:
            response = build_list_response(
                model=DomainModel(main_connection),
                path=request.path,
                model_filters=self._get_filters(org_id),
                model_fields=['*'],
                query_params=request.args.to_dict(),
                prepare_result_item_func=_prepare_domain,
                max_per_page=app.config['PAGINATION']['max_per_page_for_admin'],
                order_by=self._get_ordering_fields(),
            )

        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def _get_filters(self, org_id):
        filters = {'org_id': org_id}

        def _build_list_filter(field_name):
            if request.args.get(field_name):
                filters[field_name] = request.args.get(field_name)

        _build_list_filter('mx')
        _build_list_filter('master')
        _build_list_filter('owned')
        _build_list_filter('delegated')
        _build_list_filter('via_webmaster')
        return filters


class AdminOrganizationChangeOwnerView(View):
    @permission_required([assessor_internal_pemissions.change_organizations_owner])
    @requires(org_id=False, user=False)
    @uses_schema(CHANGE_OWNER_SCHEMA)
    def post(self, meta_connection, _, data, org_id):
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)
        new_owner_login = data['new_owner_login']
        new_owner_uid = get_user_id_from_passport_by_login(new_owner_login)
        if not new_owner_uid:
            return json_error_not_found()

        with get_main_connection(shard=shard, for_write=True) as main_connection, \
                meta_connection.begin_nested():

            # если id внутренний, то он должен принадлежать той же организации
            if not (is_outer_uid(new_owner_uid)
                    or UserModel(main_connection).filter(org_id=org_id, id=new_owner_uid).one()):
                return json_error_not_found()

            old_owner_uid = get_organization_admin_uid(main_connection, org_id)
            try:
                ChangeOrganizationOwnerTask(main_connection).delay(
                    org_id=org_id,
                    old_owner_uid=old_owner_uid,
                    new_owner_uid=new_owner_uid,
                )
            except DuplicatedTask:
                pass
        SupportActionMetaModel(meta_connection).create(
            org_id,
            SupportActions.change_organization_admin,
            g.user.passport_uid,
            {},
            'organization',
            data['comment'],
        )

        return json_response(
            {'status': 'ok'},
            status_code=202
        )


class AdminOrganizationsDomainOwnershipView(View):
    @permission_required([common_internal_permission.organizations_domain_owner_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _, org_id, domain_name):
        """
        Способы подтверждения.

        Отдаёт информацию о том, подтверждён ли домен, и какие способы подтверждения можно использовать.

        Если домен подтверждён, ручка отдаёт:

        ```
        {
            "domain": "some.example.com",
            "status": "owned",
        }
        ```

        Если требуется подтверждение, то:

        ```
        {
            "domain": "some.example.com",
            "status": "need-validation",
            "methods": [
                {
                    "method": "webmaster.file",
                    "filename": "123456.html",
                    "content": "123456",
                    "weight": 0,
                },
                ...
            ],
            "last_check": {
                "date": "2018-03-02T13:39:49.839Z",
                "method": : "webmaster.html_file",
                "fail_type": "page_unavailable",
                "fail_reason": "connection_error",
            }
        }
        ```
        ---
        tags:
          - Домены
        parameters:
          - in: path
            name: domain_name
            required: true
            type: string

        responses:
          200:
            description: В ответе будет словарь со статусом подтверждения домена и возможными способами подтверждения.
        """
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)
        lower_domain_name = domain_name.lower()

        with get_main_connection(shard=shard) as main_connection:
            domain = get_domains_from_db_or_domenator(
                meta_connection=meta_connection,
                domain_filter=DomainFilter(name=lower_domain_name, org_id=org_id),
                main_connection=main_connection,
                one=True,
            )
            admin_uid = get_organization_admin_uid(main_connection, org_id)

        if not domain:
            return json_error_not_found()

        result = {
            'domain': domain_name,
            'status': 'need-validation',
        }
        # Сюда положим информацию о последней проверке, если она была
        last_check = None

        if domain['owned']:
            result['status'] = 'owned'
        else:
            if domain['via_webmaster']:
                info = webmaster.info(
                    lower_domain_name,
                    admin_uid,
                )
                data = info['data']
                verification_status = data.get('verificationStatus')

                if verification_status == 'VERIFIED':
                    result['status'] = 'owned'
                    return json_response(result)

                # если домен в процессе подтверждения  отдаем статус 'in-progress'
                if verification_status == 'IN_PROGRESS':
                    result['status'] = 'in-progress'

                verification_uin = data['verificationUin']
                methods = webmaster.list_applicable(
                    lower_domain_name,
                    admin_uid,
                )

                # Если в ответе есть информация про последнюю попытку проверки,
                # то прокинем её в наш собственный ответ
                if 'lastVerificationAttemptTime' in data:
                    last_check = {
                        'date': data['lastVerificationAttemptTime'],
                        'method': 'webmaster.' + data['verificationType'].lower(),
                    }

                    if 'lastVerificationFailInfo' in data:
                        last_check['fail_type'] = data['lastVerificationFailInfo'].get('type', 'unknown').lower()
                        if 'reason' in data['lastVerificationFailInfo']:
                            last_check['fail_reason'] = data['lastVerificationFailInfo'].get('reason',
                                                                                             'unknown').lower()

                result['methods'] = [
                    {
                        'method': 'webmaster.' + method.lower(),
                        'weight': 0,
                        'code': verification_uin,
                    }
                    for method in methods
                ]

                # Если информация про последнюю проверку известна, то
                # добавим её в ответ
                if last_check:
                    result['last_check'] = last_check
            else:
                result['status'] = 'pdd-verification'
        return json_response(result)


class AdminPartnersList(View):
    allowed_ordering_fields = ['id', 'name']

    @permission_required([common_internal_permission.organizations_list_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _):
        response = build_list_response(
            model=PartnersMetaModel(meta_connection),
            path=request.path,
            model_filters=self._get_filters(),
            model_fields=['*'],
            query_params=request.args.to_dict(),
            max_per_page=app.config['PAGINATION']['max_per_page_for_admin'],
            order_by=self._get_ordering_fields(),
        )

        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def _get_filters(self):
        filters = {}

        def _build_list_filter(field_name):
            if request.args.get(field_name):
                filters[field_name] = request.args.get(field_name)

        _build_list_filter('id')
        _build_list_filter('name')
        return filters


class AdminOrganizationFeaturesListView(View):
    @permission_required([common_internal_permission.organizations_features_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _, org_id):
        """
        Список фич для организации в формате
        {
            'domain-auto-handover': {
                'default': False,
                'description': 'Auto domain handover',
                'enabled': True
             },
            'pdd-proxy-to-connect': {
                'default': False,
                'description': 'Switches traffic of PDD proxy to Connect',
                'enabled': False
            }
            ...
        }
        """

        org_id = ensure_integer(org_id, 'org_id')
        organization = OrganizationMetaModel(meta_connection).filter(id=org_id).one()

        if not organization:
            raise json_error_not_found()

        if not organization['ready']:
            raise OrganizationNotReadyError()

        features = get_organization_features_info(meta_connection, org_id)

        return json_response(features)


class AdminOrganizationFeaturesManageView(View):
    @permission_required([assessor_internal_pemissions.manage_features])
    @requires(org_id=False, user=False)
    @uses_schema(ACTION_FEATURE_SCHEMA)
    def post(self, meta_connection, _, data, org_id, feature_slug, action):
        """
        Включить/выключить фичу в организации.
        """
        return OrganizationFeaturesChangeView._change_feature(meta_connection, org_id, feature_slug, action, data['comment'])


class AdminOrganizationWhitelistManageView(View):
    methods = ['post', 'delete']

    @uses_schema(MANAGE_WHITELIST_SCHEMA)
    @permission_required([bizdev_internal_permission.manage_whitelist])
    @requires(org_id=False, user=True)
    def post(self, meta_connection, _, data, org_id):
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)

        with get_main_connection(shard=shard, for_write=True) as main_connection:
            vip_features = OrganizationModel(main_connection).filter(id=org_id).fields('vip').one()['vip']

            if vip_reason.whitelist not in vip_features:
                vip_features.append('whitelist')
                OrganizationModel(main_connection).update_vip_reasons(org_id, vip_features)
                SupportActionMetaModel(meta_connection).create(
                    org_id,
                    SupportActions.manage_whitelist,
                    g.user.passport_uid,
                    {'whitelist': 'add'},
                    'organization',
                    comment=data['comment'],
                )

        return json_response({}, status_code=201)

    @uses_schema(MANAGE_WHITELIST_SCHEMA)
    @permission_required([bizdev_internal_permission.manage_whitelist])
    @requires(org_id=False, user=True)
    def delete(self, meta_connection, _, data, org_id):
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)

        with get_main_connection(shard=shard, for_write=True) as main_connection:
            vip_features = OrganizationModel(main_connection).filter(id=org_id).fields('vip').one()['vip']

            if vip_reason.whitelist in vip_features:
                vip_features.remove('whitelist')
                OrganizationModel(main_connection).update_vip_reasons(org_id, vip_features)
                SupportActionMetaModel(meta_connection).create(
                    org_id,
                    SupportActions.manage_whitelist,
                    g.user.passport_uid,
                    {'whitelist': 'remove'},
                    'organization',
                    comment=data['comment'],
                )

        return json_response({}, status_code=204)


class OrganizationBlockView(View):

    @uses_schema(ORGANIZATION_BLOCK_SCHEMA)
    @permission_required([assessor_internal_pemissions.block_organization])
    @requires(org_id=False, user=True)
    def post(self, meta_connection, _, data, org_id):
        """
        Блокирует организацию
        ---
        tags:
          - Организация
        parameters:
          - in: path
            name: org_id
            required: true
            type: integer
          - in: body
            name: body
        responses:
          200:
            description: Огранизация заблокирована
        """

        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)

        with get_main_connection(shard=shard, for_write=True) as main_connection:
            OrganizationModel(main_connection).block(org_id)

            domain_name = get_master_domain(main_connection, org_id, True, False)
            if domain_name is not None:
                domain_id = get_domain_id_from_blackbox(domain_name)
                app.passport.block_domain(domain_id)

            organization = OrganizationModel(main_connection).get(org_id)
            SupportActionMetaModel(meta_connection).create(
                org_id,
                SupportActions.block_organization,
                g.user.passport_uid,
                organization,
                'organization',
                data['comment'],
            )

        return json_response(
            data={},
            status_code=200,
        )


class OrganizationUnblockView(View):

    @uses_schema(ORGANIZATION_BLOCK_SCHEMA)
    @permission_required([assessor_internal_pemissions.block_organization])
    @requires(org_id=False, user=True)
    def post(self, meta_connection, _, data, org_id):
        """
        Разблокирует организацию
        ---
        tags:
          - Организация
        parameters:
          - in: path
            name: org_id
            required: true
            type: integer
          - in: body
            name: body
        responses:
          200:
            description: Огранизация разблокирована
        """

        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)

        with get_main_connection(shard=shard, for_write=True) as main_connection:
            OrganizationModel(main_connection).unblock(org_id)

            domain_name = get_master_domain(main_connection, org_id, True, False)
            if domain_name is not None:
                domain_id = get_domain_id_from_blackbox(domain_name)
                app.passport.unblock_domain(domain_id)

            organization = OrganizationModel(main_connection).get(org_id)
            SupportActionMetaModel(meta_connection).create(
                org_id,
                SupportActions.unblock_organization,
                g.user.passport_uid,
                organization,
                'organization',
                data['comment'],
            )

        return json_response(
            data={},
            status_code=200,
        )
