from flask import request

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth.decorators import requires, permission_required
from intranet.yandex_directory.src.yandex_directory.common import schemas
from intranet.yandex_directory.src.yandex_directory.common.components import component_registry
from intranet.yandex_directory.src.yandex_directory.common.pagination import Paginator
from intranet.yandex_directory.src.yandex_directory.common.utils import json_response, NotGiven
from intranet.yandex_directory.src.yandex_directory.core.permission import common_internal_permission
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.meta.repositories.resource_history_repository import ResourceHistoryRepository
from intranet.yandex_directory.src.yandex_directory.swagger import uses_schema_for_get

RESOURCE_HISTORY_GET_SCHEMA = {
    'title': 'Get resource history',
    'type': 'object',
    'properties': {
        'service_slug': schemas.STRING,
        'resource_id': schemas.STRING_OR_INTEGER,
        'name': schemas.STRING,
        app.config['PAGINATION']['page_query_param']: schemas.INTEGER,
        app.config['PAGINATION']['per_page_query_param']: schemas.INTEGER,
    },
    'required': ['service_slug'],
    'additionalProperties': True,
}


class ResourceHistoryView(View):

    @uses_schema_for_get(RESOURCE_HISTORY_GET_SCHEMA)
    @permission_required([common_internal_permission.resource_history_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, _):
        query_params = request.args.to_dict()

        page = int(query_params.get(app.config['PAGINATION']['page_query_param']) or 1)
        per_page = int(query_params.get(app.config['PAGINATION']['per_page_query_param']) or 50)

        offset = (page - 1) * per_page

        resource_history_repository = component_registry().resource_history_repository  # type: ResourceHistoryRepository

        resources = resource_history_repository.find_by_fields({
            'resource_id': query_params.get('resource_id', NotGiven),
            'name': query_params.get('name', NotGiven),
            'service_slug': query_params.get('service_slug', NotGiven),
        }, limit=per_page + 1, offset=offset)

        is_last_page = len(resources) <= per_page
        if not is_last_page:
            del resources[-1]

        paginator = Paginator(
            total=None,
            page=page,
            per_page=per_page,
            path=request.path,
            query_params=query_params,
            max_per_page=50,
            is_last_page=is_last_page,
        )

        response = paginator.response
        response['data']['result'] = list([{
            field: getattr(resource, field)
            for field in list(resource.__table__.columns.keys())
        } for resource in resources])

        return json_response(
            data=response['data'],
            headers=response['headers'],
        )
