# -*- coding: utf-8 -*-
import uuid

from flask import request

from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    requires,
    permission_required)
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_shard,
    get_shard_numbers
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import ImmediateReturn
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    build_list_response,
    json_error_invalid_value,
    json_error_not_found,
    build_multishard_list_response,
    hide_sensitive_params,
    unpickle,
)
from intranet.yandex_directory.src.yandex_directory.core.models import TaskModel
from intranet.yandex_directory.src.yandex_directory.core.permission.internal_permissions import assessor_internal_pemissions
from intranet.yandex_directory.src.yandex_directory.core.task_queue.base import (
    TASK_STATES,
    get_task_names_map,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    only_fields,
    ensure_integer,
)
from intranet.yandex_directory.src.yandex_directory.core.views.base import View

ALLOWED_FIELDS = [
    'id',
    'task_name',
    'state',
    'start_at',
    'created_at',
    'finished_at',
    'author_id',
    'result',
    'traceback',
    'dependencies_count',
    'queue',
    'exception',
    'params',
]

def prepare_task(task):
    """
    Подготовка задачи к выдаче в ручках.
    """
    task['task_name'] = task['task_name'].rsplit('.', 1)[-1]
    task['result'] = unpickle(task['result'])
    return only_fields(task, *ALLOWED_FIELDS)


class AdminTaskDetailView(View):
    @requires(org_id=False, user=False)
    @permission_required([assessor_internal_pemissions.see_tasks])
    def get(self, _, meta_connection, task_id):
        """
        Возвращаем состояние задачи

        tags:
            - Очередь задач
        responses:
            200:
                description: Получаем данные о задаче
            404:
                description: Задача не найдена
            422:
                description: Переданы неправильные данные(не uuid)
        """
        try:
            uuid.UUID(task_id)
        except ValueError:
            return json_error_invalid_value(task_id)
        shards = get_shard_numbers()
        for shard in shards:
            with get_main_connection(for_write=False, shard=shard) as main_connection:
                task = TaskModel(main_connection).get(task_id)
                if task:
                    break
        if task:
            task = prepare_task(task)
            hide_sensitive_params(task)
            return json_response(task)
        else:
            return json_error_not_found()


class AdminTasksDependeciesListView(View):
    @requires(org_id=False, user=False)
    @permission_required([assessor_internal_pemissions.see_tasks])
    def get(self, _, meta_connection, dependent_id):
        try:
            uuid.UUID(dependent_id)
        except ValueError:
            return json_error_invalid_value('dependent_id')
        response = build_multishard_list_response(
            model=TaskModel,
            model_filters={'dependent': dependent_id},
            path=request.path,
            query_params=request.args.to_dict(),
            prepare_result_item_func=prepare_task,
            order_by=self._get_ordering_fields(),
        )
        hide_sensitive_params(response['data'])
        return json_response(
            response['data'],
            headers=response['headers'],
        )


class AdminTasksListView(View):
    allowed_ordering_fields = ['created_at', 'start_at']

    @requires(org_id=False, user=False)
    @permission_required([assessor_internal_pemissions.see_tasks])
    def get(self, meta_connection, main_connection, org_id):
        """
        Возвращается список задач, связанных с переданной организацией,
        либо список задач, от которых зависит переданная задача.

        По умолчанию возвращаются задачи в нетерминальных стейтах ('in-progress', 'suspended', 'free').
        Если нужны задачи в других стейтах - передать их через запятую в аргументе state.
        Пример:

        /admin/tasks/?org_id=123&state=success,rollback

        Если нужны задачи конкретного типа, передать нужный тип в аргументе task_name.
        Пример:

        /admin/tasks/?org_id=123&task_name=CreatePddMaillistTask

        Пример ответа:
            [
                {
                    'id': '038bbc21-579d-4509-9477-392fac485eb1',
                    'task_name': 'CreatePddMaillistTask',
                    'state': 'rollback',
                    'created_at': '2017-08-25T12:17:26.638279+00:00',
                    'start_at': null,
                    'finished_at': null,
                    'author_id': 123123123123,
                    'result': 'account_id=1',
                    'traceback': 'traceback'
                },
                ...
            ]

        ---
        tags:
          - Админка
        responses:
          200:
            description: Success
          403:
            description: Request is not available for this service or user
          401:
            description: Invalid or expired TVM 2.0 user- or service-ticket
          422:
            description: Some validation error
        """
        org_id = ensure_integer(org_id, 'org_id')
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard) as main_connection:
            response = build_list_response(
                model=TaskModel(main_connection),
                model_filters=self._get_filters(org_id),
                path=request.path,
                query_params=request.args.to_dict(),
                prepare_result_item_func=prepare_task,
                order_by=self._get_ordering_fields(),
            )
        hide_sensitive_params(response['data'])
        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def _get_filters(self, org_id):
        if request.args.get('dependent_id'):
            filters = self._build_filter_by_dependent_id(request.args.get('dependent_id'))
        else:
            filters = {'params__contains': {'org_id': org_id}}
        self._build_filter_status(filters)
        self._build_filter_by_task_name(filters)
        return filters

    def _build_filter_status(self, filters):
        field_name = 'state'
        filters[field_name] = [
            TASK_STATES.in_progress,
            TASK_STATES.suspended,
            TASK_STATES.free,
        ]
        if request.args.get(field_name):
            values = request.args.get(field_name).split(',')
            if len(values) == 1:
                filters[field_name] = values[0]
            else:
                filters[field_name] = values

    def _build_filter_by_task_name(self, filters):
        field_name = 'task_name'
        if request.args.get(field_name):
            short_names = request.args.get(field_name).split(',')
            try:
                full_names = [get_task_names_map()[name] for name in short_names]
            except KeyError:
                raise ImmediateReturn(json_error_invalid_value(field_name))
            if len(full_names) == 1:
                full_names = full_names[0]
            filters[field_name] = full_names

    def _build_filter_by_dependent_id(self, dependent_id):
        try:
            uuid.UUID(dependent_id)
        except ValueError:
            raise ImmediateReturn(json_error_invalid_value('dependent_id'))
        return {'dependent': dependent_id}
