# -*- coding: utf-8 -*-
from flask import g
from copy import copy
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    requires,
    internal,
    no_scopes,
    permission_required,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    get_object_or_404,
    get_user_contacts_from_blackbox_by_uid,
    json_error_not_found,
    get_internal_roles_by_uid,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    UserModel,
    UserMetaModel,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.internal_permissions import (
    common_internal_permission,
    support_internal_permission,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    ensure_integer,
    prepare_user,
)
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.common.db import get_main_connection
from intranet.yandex_directory.src.yandex_directory.core.views.users import UserView


class AdminUserDetailView(View):
    @permission_required([common_internal_permission.user_detail_view])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, main_connection, user_id):
        user_id = ensure_integer(user_id, 'user_id')
        meta_user = get_object_or_404(
            UserMetaModel(meta_connection),
            user_id=user_id,
            fields=['organization.shard']
        )

        with get_main_connection(shard=meta_user['organization']['shard']) as main_connection:
            model = UserModel(main_connection)
            fields = copy(model.simple_fields)
            fields.add('avatar_id')
            user = get_object_or_404(
                model_instance=UserModel(main_connection),
                user_id=user_id,
                fields=fields,
            )

            return json_response(prepare_user(
                main_connection,
                user,
                expand_contacts=True,
                api_version=1,
            ))


class UserContactsView(UserView):

    @internal
    @permission_required([support_internal_permission.get_user_contacts])
    @no_scopes
    @requires(org_id=False, user=False)
    def get(self, meta_connection, main_connection, user_id):
        """
        Получаем телефон и внешние почты пользователя

        Пример ответа:

            {
                'uid': '1337228',
                'phone_number': '88005553535',
                'email': ["krutoy_paren1999@pilesos.com"]
            }

        ---
        tags:
          - Сотрудники
        parameters:
          - in: path
            name: user_id
            required: true
            type: integer
        responses:
          200:
            description: Словарь с данными для пользоватеоя
          404:
            description: Юзер не найден
        """

        user_id = ensure_integer(user_id, 'user_id')
        user_contacts = get_user_contacts_from_blackbox_by_uid(user_id)
        if (user_contacts):
            return json_response(user_contacts)
        else:
            return json_error_not_found()


class AdminAdminDetailView(View):
    """
    Отдаем список ролей пользователя админки
    """
    @permission_required([common_internal_permission.user_roles])
    @requires(org_id=False, user=False)
    def get(self, meta_connection, main_connection):
        roles = get_internal_roles_by_uid(g.user.passport_uid)
        return json_response({'roles': roles})
