# -*- coding: utf-8 -*-
from functools import wraps

from flask import g

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory.auth.scopes import check_scopes, ScopeCheckMethod
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    NoScopesError,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    check_permissions,
    force_text,
)

from intranet.yandex_directory.src.yandex_directory.core.permission import permission_descriptions


def no_auth(function):
    """
    Декоратор для методов, которые не требуют авторизации

    По умолчанию, мы считаем, что каждая view требует аутентификации
    и авторизации, но если весь механизм авторизации
    нужно отключить, то необходимо использовать декоратор `@no_auth`.

    ВНИМАНИЕ, мало какая view должны быть без аутентификации.
    """
    setattr(function, 'require_auth', False)
    function = no_permission_required(function)
    return function


def internal(function):
    """
    Декоратор для методов, которые могут быть использованы
    только внутренними сервисами.
    """
    setattr(function, 'internal', True)
    return function


def requires(org_id=True, user=True):
    """
    Задает необходимые условия для авторизации доступа к ручкам API.
    """

    def wrapper(function):
        setattr(function, 'requires_org_id', org_id)
        setattr(function, 'requires_user', user)
        return function

    return wrapper


def no_permission_required(function):
    """Этот декоратор нужно использовать для того, чтобы
    явно указать, что вьюшка не требует наличия каких-то особенных прав
    у пользователя.
    """
    function.permissions = []
    return function


def permission_required(permissions, object_type=None, any_permission=False):
    """
    Декоратор, задающий права для метода
    Args:
        permissions (list): необходимые права, список строк из мн-ва
            core.views.permissions.global_permissions и
            core.views.permissions.group_permissions
        object_type (string): тип объекта
            {'group', 'user', 'department', 'resource'}
            Нужен для проверки прав на один объект с его id
        any_permission (bool): проверка должна проходить если у пользователя есть хотя бы одно право из списка
            В этом случае, дальнейшие проверки прав должны проходить внутри view-функции
    """
    if not hasattr(permissions, '__iter__') or isinstance(permissions, (str, bytes)):
        raise TypeError('First argument must be iterable object')

    def decorator(view_func):
        @wraps(view_func)
        def _wrapped_view(view_instance,
                          meta_connection,
                          main_connection,
                          *args,
                          **kwargs):
            object_id = None
            if object_type:
                object_id = kwargs.get(object_type + '_id')

            check_permissions(
                meta_connection,
                main_connection,
                permissions,
                object_type,
                object_id,
                any_permission=any_permission,
            )
            return view_func(
                view_instance,
                meta_connection,
                main_connection,
                *args,
                **kwargs
            )

        all_descriptions = [
            permission_descriptions.get(permission, '')
            for permission in permissions
        ]
        permissions_description = ' '.join(map(force_text, all_descriptions))
        view_doc = force_text(view_func.__doc__)
        if '{permissions}' in view_doc:
            view_doc = view_doc.replace(
                '{permissions}',
                permissions_description
            )
        _wrapped_view.__doc__ = view_doc
        new_permissions = permissions
        if hasattr(view_func, 'permissions'):
            new_permissions = permissions + view_func.permissions
        _wrapped_view.permissions = new_permissions

        return _wrapped_view

    return decorator


def scopes_required(scopes, method=ScopeCheckMethod.OR):
    def decorator(view_func):
        assert isinstance(scopes, list), 'Аргумент scopes_required должен быть списком'

        @wraps(view_func)
        def _wrapped_view(view_instance,
                          meta_connection,
                          main_connection,
                          *args,
                          **kwargs):
            # Независимо от того, откуда пришёл запрос, проверяем
            # наличие нужных вьюхе скоупов
            if not check_scopes(g.scopes, scopes, method):
                with log.fields(scopes=g.scopes,
                                required_scopes=scopes,
                                method=method):
                    log.warning('No required scope')

                raise NoScopesError(*scopes)

            return view_func(
                view_instance,
                meta_connection,
                main_connection,
                *args,
                **kwargs
            )

        _wrapped_view.scopes = scopes
        return _wrapped_view

    return decorator


def no_scopes(func):
    """Этот декоратор нужен для того, чтобы явно указать, что ручка
    не требует каких-либо скоупов.
    """
    func.scopes = []
    return func
