# -*- coding: utf-8 -*-
import os
import json
import time
import uuid
import re

from flask import (
    request,
    Response,
    g,
)

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log, access_log

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    is_yandex_team_uid,
    get_user_role,
)
from intranet.yandex_directory.src.yandex_directory.core.features import USE_CLOUD_PROXY
from intranet.yandex_directory.src.yandex_directory.core.features.utils import is_feature_enabled
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    APIError,
    NotAYambTestOrganizationError,
    UserNotFoundInBlackbox,
    HeaderShouldBeIntegerError,
    HeaderIsRequiredError,
    CatalogRouteForbiddenError,
    InternalRouteError,
    AuthenticationError,
    AuthorizationError,
    OrganizationNotReadyError,
)
from intranet.yandex_directory.src.yandex_directory.common import backpressure
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    get_karma,
    log_work_time,
    validate_data_by_schema,
    coerce_to_int_or_none,
    get_auth_fields_for_log,
    json_error_exception,
    json_error_not_found,
    Ignore,
    json_error,
    split_by_comma,
)
from intranet.yandex_directory.src.yandex_directory.auth.service import Service
from intranet.yandex_directory.src.yandex_directory.auth.scopes import (
    scope,
    check_scopes,
)
from intranet.yandex_directory.src.yandex_directory.auth.user import (
    User,
    TeamUser,
)
from intranet.yandex_directory.src.yandex_directory.auth.utils import (
    hide_auth_from_headers,
    obfuscate_ticket,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    RobotServiceModel,
    OrganizationMetaModel,
    UserMetaModel,
    ServiceModel,
    OrganizationServiceModel,
    OrganizationRevisionCounterModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.service import INTERNAL_ADMIN_SERVICE_SLUG
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_meta_connection,
    get_main_connection,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.users.base import is_outer_admin
from intranet.yandex_directory.src.yandex_directory.core.cloud.utils import create_cloud_user
from intranet.yandex_directory.src.yandex_directory.core.exceptions import CloudUnavailableError
from intranet.yandex_directory.src.yandex_directory.core.utils.ycrid_manager import ycrid_manager

# ручка для замера времени ответа без работы с БД
from intranet.yandex_directory.src.yandex_directory.core.utils.credential_sanitizer import CredentialSanitizer

SIMPLE_PING_URL = '/simple-ping/'

MAX_UID_VALUE = 9223372036854775807  # 2^63-1 Postgres Bigint
SIGNAL_NAME_TRANSTAB = str.maketrans('<-> ', '____')


def get_authorization_tokens():
    """Example
    Authorization: OAuth vF9dft4qmT
    Authorization: SessionId vF9dft4qmT
    """
    headers = request.headers

    # Если есть такой заголовок, то в нём передан сервисный TVM 2.0 тикет
    if 'X-Ya-Service-Ticket' in headers:
        yield 'tvm2', headers['X-Ya-Service-Ticket']

    # Если есть такой заголовок, то в нём передан TVM тикет
    if 'Ticket' in headers:
        yield 'tvm', headers['Ticket']

    try:
        name, token = headers.get('Authorization', '').split(' ')
        yield name.lower(), token
    except ValueError:
        pass


def authorize(meta_connection,
              org_id=None,
              service=None,
              user=None,
              scopes=None,
              require_org_id=False,
              require_user=False,
              internal_only=False):
    """Проверяем условия, при которых пользователя можно считать авторизованным.
    Код вынесен в отдельную функцию, чтобы было проще его тестировать.
    Подробное объяснение условий смотри в authentication.rst,
    раздел "Авторизация".
    """
    shard = None
    revision = None
    scopes = scopes or set()
    organization_ready = True

    # Вычисляем и проверяем org_id

    if user and not (service and service.identity == INTERNAL_ADMIN_SERVICE_SLUG):
        filter_data = {'is_dismissed': False}
        if user.is_cloud:
            filter_data['cloud_uid'] = user.cloud_uid
        else:
            filter_data['id'] = user.passport_uid

        org_ids = []
        if user.has_cloud_header:
            # К нам пришёл пользователь с облачным заголовком. Нужно сходить в Я.Орг и
            # полистить его организации, досоздать пользователя в тех организациях, где его не хватает
            try:
                org_ids = create_cloud_user(meta_connection, user)
            except Exception:
                # rpc недоступен, ну что же, отдадим пустой список организаций
                log.exception('Cannot create cloud user')
        if user.is_cloud:
            # если у пользователя только облачный заголовок, то в его org_ids будут только облачные организации
            user.org_ids = org_ids
        else:
            meta_users = UserMetaModel(meta_connection).find(
                filter_data=filter_data,
                fields=['id', 'org_id'],
                order_by=['created', 'org_id'],
            )
            user.org_ids = [item['org_id'] for item in meta_users]

        if org_id is not None and org_id not in user.org_ids:
            # Если в запросе был указан org_id и он не совпадает
            # ни с одной организацией пользователя, то это ошибка,
            # так как пытаются работать с организацией к которой пользователь
            # не принадлежит.
            raise AuthorizationError(
                'Wrong organization (org_id: {}, user.uid: {})'.format(org_id, user.passport_uid),
            )
        # Если пользователь всего в одной организации,
        # то берем id единственной организации пользователя
        if len(user.org_ids) == 1:
            org_id = user.org_ids[0]

        # Для обратной совместимости при переезде на мультиорг
        if org_id is None and not is_outer_admin(meta_connection, user.passport_uid, is_cloud=user.is_cloud):
            org_id = user.org_ids[0] if user.org_ids else None

    if org_id is not None:
        # Если известен org_id, то надо проверить, что
        # такая организация есть в базе.
        organization_meta = OrganizationMetaModel(meta_connection).get(org_id)
        if not organization_meta:
            OrganizationMetaModel(meta_connection).raise_organization_not_found_exception(org_id)
        g.organization_meta = organization_meta
        if request.headers.get('x-disable-cloud-proxy'):
            g.use_cloud_proxy = False
        else:
            g.use_cloud_proxy = is_feature_enabled(meta_connection, org_id, USE_CLOUD_PROXY)

        shard = organization_meta['shard']

        # Если организация пока не готова, то считаем, что её нет
        # если вдруг для view нужен org_id, то позже мы кинем про это ошибку.
        if not organization_meta['ready']:
            # Раньше мы тут бросали исключение AuthorizationError, но
            # это приводило к проблемам, описанным в тикете:
            # https://st.yandex-team.ru/DIR-2993
            org_id = None
            shard = None
            organization_ready = False
        else:
            with get_main_connection(shard) as main_connection:
                revision = OrganizationRevisionCounterModel(main_connection).get(org_id)['revision']
                if user:
                    user.role = get_user_role(meta_connection, main_connection, org_id,
                                              uid=user.get_cloud_or_passport_uid(), is_cloud=user.is_cloud)

                if service and not service.identity == INTERNAL_ADMIN_SERVICE_SLUG:
                    # Вычисление организации закончилось, теперь надо проверить ещё пару вещей:

                    # 1. Может ли сервис работать с данной организацией – подключен
                    #    ли он для неё или быть может запрос имеет специальный scope?
                    if not check_scopes(scopes, [scope.work_with_any_organization]):
                        organization_service = OrganizationServiceModel(main_connection).find(
                            {'org_id': org_id, 'service_id': service.id, 'enabled': Ignore}
                        )
                        # Если не подключен, то это ошибка
                        if not organization_service:
                            raise AuthorizationError(
                                'Service (identity: {}, id: {}) is not enabled'.format(
                                    service.identity, service.id
                                ),
                                'service_is_not_enabled'
                            )
                        elif not organization_service[0]['enabled']:
                            raise AuthorizationError(
                                'Service (identity: {}, id: {}) was disabled for this organization.'.format(
                                    service.identity, service.id
                                ),
                                'service_was_disabled'
                            )

                    # 2. Может ли сервис работать от имени указанного пользователя –
                    #    есть ли у него нужные скоупы чтобы работать с любым пользователем
                    #    или быть может он пытается работать от имени своего роботного аккаунта?
                    if user and not check_scopes(scopes, [scope.work_on_behalf_of_any_user]):
                        # Сервису "из под пользователя" можно ходить лишь
                        # при наличии scope work_on_behalf_of_any_user,
                        # либо если это робот сервиса.
                        service_robot = None
                        if not user.is_cloud:
                            service_robot = RobotServiceModel(main_connection).filter(
                                org_id=org_id,
                                uid=user.passport_uid,
                                service_id=service.id,
                            ).count()
                        # В противном случае, должна быть 403 ошибка
                        if not service_robot:
                            raise AuthorizationError(
                                'Unable to work on behalf of the user'
                            )
    else:
        g.use_cloud_proxy = False

    # Потом проверяем какие объекты, скоупы и права нужны для текущей ручки.
    if require_org_id and not organization_ready:
        raise OrganizationNotReadyError()

    if require_user and user is None:
        raise AuthorizationError(
            'User is required for this operation')

    if require_org_id and org_id is None:
        raise AuthorizationError(
            'Organization is required for this operation')

    if internal_only and (service is None or not service.is_internal):
        raise InternalRouteError()

    return org_id, shard, revision


def get_view_function():
    endpoint = request.endpoint
    view = app.view_functions.get(endpoint)
    if view:
        # ручки swagger роутятся как функции, а у нас как методы класса
        if getattr(view, 'view_class', None):
            method_name = view.view_class._methods_map.get((request.method.lower(), request.api_version))
            if method_name:
                return getattr(view.view_class(), method_name)

    return view


class CloseServiceBySmokeTestResultsMiddleware(object):
    """
    Если запрос пришел не в ручку /ping/ и по результатам backpressure.is_need_to_close_service
    нужно закрыть бекэнд - отвечаем 503
    """

    def __init__(self, app):
        app.before_request(self.process_request)

    def process_request(self):
        if isinstance(request.endpoint, str) and not request.endpoint.startswith('external.PingView') \
                and backpressure.is_need_to_close_service():
            log.error('Service is temporarily unavailable by smoke tests results')
            return json_error(
                503,
                'service_unavailable',
                'Service unavailable',
            )


class DebugSwitcherMiddleware(object):
    def __init__(self, app):
        app.before_request(self.process_request)

    def process_request(self):
        if request.headers.get('x-debug'):
            g.x_debug = True


class OrganizationRevisionMiddleware(object):
    def __init__(self, app):
        app.after_request(self.process_response)

    def process_response(self, response, *args, **kwargs):
        # org_revision хранится в g-объекте и проставляется при аутентификации
        # и при создании новых записей в таблицу actions
        revision = getattr(g, 'revision', None)
        if revision is not None:
            response.headers.add_header('X-Revision', revision)
            # Стандарт требует, чтобы etag был обертнут в кавычки
            response.headers.add_header('Etag', '"{0}"'.format(revision))
        return response


class RequestIDMiddleware(object):
    def __init__(self, app):
        app.before_request(self.process_request)
        app.after_request(self.process_response)

        try:
            import uwsgi
        except ImportError:
            self._save_request_id_to_uwsgi_variables = lambda: None
        else:
            # Если приложение запущено под uwsgi, то сохраняем в него request_id для его собственных логов
            # http://uwsgi-docs.readthedocs.io/en/latest/LogFormat.html#user-defined-logvars
            self._save_request_id_to_uwsgi_variables = lambda: uwsgi.set_logvar('request_id', g.request_id)

    def process_request(self):
        request_id = self._get_request_id()
        if request_id:
            if self.is_request_id_valid(request_id):
                g.request_id = request_id
            else:
                log.trace().error('Invalid request id')
                return json_error(
                    422,
                    'invalid_request_id',
                    'X-Request-ID has invalid format',
                )
        else:
            g.request_id = self._generate_id()

        self._save_request_id_to_uwsgi_variables()

    def process_response(self, *args, **kwargs):
        if 'request_id' in g:
            args[0].headers['X-Request-ID'] = g.request_id

        return args[0]

    def _get_request_id(self):
        return request.headers.get('x-request-id')

    def _generate_id(self):
        return uuid.uuid4().hex

    def is_request_id_valid(self, request_id):
        return len(request_id) <= 100 and re.match(r'^[\w\d-]+$', request_id)


class LogMiddleware(object):
    start_time_request = None

    def __init__(self, app):
        app.before_request(self.process_request)
        app.after_request(self.process_response)

    def process_request(self):
        ycrid_manager.set_from_request(request)
        self.start_time_request = time.time()

    def process_response(self, response):
        request_time = time.time() - self.start_time_request

        uid = ''
        if hasattr(g, 'user'):
            uid = getattr(g.user, 'passport_uid', '')
        headers = CredentialSanitizer.get_headers_list_with_sanitized_credentials(request.headers)
        uri = '%s?%s' % (request.path, request.query_string.decode('utf-8'))

        access_log.info(
            'http request %s: %s %.3fms %s' % (request.method, uri, request_time, response.status_code),
            extra={
                'proto': request.environ.get('SERVER_PROTOCOL'),
                'method': request.method,
                'uid': uid,
                'org_id': getattr(g, 'org_id', '') or '',
                'uri': uri,
                'headers': ' '.join(['"%s: %s"' % (k, v) for k, v in headers]),
                'status': response.status_code,
                'request_time': request_time,
            }
        )

        response_code_mask = str(response.status_code)[0] + 'xx'
        service = g.service.identity if hasattr(g, 'service') and g.service else 'unknown'
        view = g.view_class.__class__.__name__ if hasattr(g, 'view_class') and g.view_class else 'unknown'
        method = g.method.__name__ if hasattr(g, 'method') and g.method else 'unknown'

        app.stats_aggregator.inc(f'response_code_service_{service}_{response_code_mask}_summ')

        app.stats_aggregator.inc(f'response_code_view_{view}_{method}_{response_code_mask}_summ')
        app.stats_aggregator.add_to_bucket(f'response_time_view_{view}_{method}', request_time)

        ycrid_manager.reset()
        return response


def get_client_ip_from(headers):
    # может быть тут стоит озаботиться защитой от
    # IP спуфинга, описанного тут:
    # http://esd.io/blog/flask-apps-heroku-real-ip-spoofing.html
    #
    # или дождаться решения тикета
    # https://st.yandex-team.ru/QLOUDDEV-295
    # и просто брать IP из заголовка
    # X-Forwarded-For-Y

    result = headers.get('x-real-ip')

    if not result:
        raise RuntimeError('Unknown client ip')

    return result


def get_iam_token(headers):
    return headers.get('x-user-iam-token')


def get_user_ip_from(headers):
    user_ip = headers.get('x-user-ip')
    if not user_ip:
        log.warning('Header X-User-IP is required')
        raise HeaderIsRequiredError('X-User-IP')

    return user_ip


def get_internal_service_by_token(token):
    # Отдельная функция тут нужна для удобства тестирования,
    # так как её просто запатчить.
    return app.config['INTERNAL_SERVICES_BY_TOKEN'].get(token)


def authenticate_by_token(token, headers, no_more_tokens):
    if not app.config['INTERNAL']:
        raise AuthenticationError(
            'Our API requires OAuth authentication')

    internal_service = get_internal_service_by_token(token)

    if internal_service:
        scopes = internal_service['scopes']

        # Попытаемся найти в базе сервис, соответстующий
        # этому старому токену
        with get_meta_connection() as meta_connection:
            service_from_db = ServiceModel(meta_connection) \
                .filter(slug=internal_service['identity']) \
                .fields('id', 'scopes') \
                .one()

        # Если сервис в базе нашли, то проставим его id,
        # чтобы потом можно было определить включён он в организации или нет.
        # И скоупы его тоже будем брать из базы.
        # Это сделано для того, чтобы можно было отключать Ямб в организациях
        # и при этом не форсить его переходить на TVM2:
        # https://st.yandex-team.ru/DIR-4949
        if service_from_db:
            service_id = service_from_db['id']
            scopes = service_from_db['scopes']
        else:
            service_id = None

        service = Service(
            id=service_id,
            name=internal_service['name'],
            identity=internal_service['identity'],
            is_internal=True,
            ip=get_client_ip_from(headers),
        )
        org_id = coerce_to_int_or_none(
            headers.get('x-org-id'),
            HeaderShouldBeIntegerError,
            'X-ORG-ID',
        )
        user = get_user_from_headers(headers)

        return {
            'auth_type': 'token',
            'service': service,
            'user': user,
            'org_id': org_id,
            'scopes': scopes,
        }

    # Исключения выбрасываем только если нет других
    # способов для аутентификации запроса
    if no_more_tokens:
        raise AuthenticationError(
            'Bad credentials',
            headers={'WWW-Authenticate': "Token"}
        )


def authenticate_by_oauth(token, headers, no_more_tokens):
    """
    Авторизация по OAuth токенам
    Про внешние сервисы https://docs.ws.yandex-team.ru/architecture/services.html
    """
    # X-Real-Ip
    user_ip = get_client_ip_from(headers)

    # https://doc.yandex-team.ru/blackbox/reference/method-oauth.xml
    oauth_info = app.blackbox_instance.oauth(
        headers_or_token=token,
        userip=user_ip,
        by_token=True,
        dbfields=app.config['BLACKBOX']['dbfields']
    )
    if oauth_info.get('status') == 'VALID':
        uid = oauth_info.get('uid')
        client_id = oauth_info['oauth']['client_id']
        # oauth_info['oauth']['scope'] => 'service-slug:scope1 service-slug:scope2'
        scopes = [_f for _f in oauth_info['oauth']['scope'].strip().lower().split(' ') if _f]

        org_id = coerce_to_int_or_none(
            headers.get('x-org-id'),
            HeaderShouldBeIntegerError,
            'X-ORG-ID',
        )

        user = None
        service = None

        with get_meta_connection() as meta_connection:
            # проверим добавлен ли сервис с таким client_id  в директорию
            service_from_db = ServiceModel(meta_connection).find(
                filter_data={
                    'client_id': client_id,
                },
                one=True,
            )

            if service_from_db:
                # пока считаем любой сервис, известный Директории - внутренним

                service = Service(
                    id=service_from_db['id'],
                    name=oauth_info['oauth']['client_name'],
                    identity=service_from_db['slug'],
                    # если сервис добавлен в директорию, то ему доступны внутренние ручки
                    # они с декоратором @internal
                    is_internal=True,
                    # с OAuth мы всегда считаем, что в ручку пришёл
                    # конечный пользователь а не бэк сервиса
                    ip=user_ip,
                )
                scopes.extend(service_from_db['scopes'])

            # Если в токене содержится uid, то запомним этого пользователя
            if uid:
                user = User(
                    passport_uid=uid,
                    ip=user_ip,
                    iam_token=get_iam_token(headers),
                )

        if not user and not service:
            # Исключения выбрасываем только если нет других
            # способов для аутентификации запроса
            if no_more_tokens:
                error = 'No user and service'
                raise AuthenticationError(
                    error,
                    headers={
                        'WWW-Authenticate': "OAuth error='%s'" % error
                    }
                )
            else:
                return None
        else:
            return {
                'auth_type': 'oauth',
                'service': service,
                'user': user,
                'org_id': org_id,
                'scopes': scopes,
            }
    else:
        # Исключения выбрасываем только если нет других
        # способов для аутентификации запроса
        if no_more_tokens:
            # https://beta.wiki.yandex-team.ru/oauth/newservice/#kakpodderzhatoauthvsvojomservise
            raise AuthenticationError(
                'Bad credentials',
                headers={
                    'WWW-Authenticate': "OAuth error='%s'" % oauth_info.get('error', 'No more tokens')
                }
            )
        else:
            return None


def get_user_from_headers(headers):
    # В качестве UID выбираем либо то что передано,
    # либо пробуем найти uid в заголовке.
    #
    # Извне может быть передан список UID
    # которые извлечёны из TVM тикета. Их может быть
    # больше одного, если пользователь использует мультиавторизацию.
    # В этом случае, должен быть явно указан текущий UID в заголовке
    # X-UID, если не указан, а извне передано больше одного уида,
    # то считаем, что пользователь не аутентифицирован.
    passport_uid = headers.get('x-uid')
    cloud_uid_from_headers = headers.get('x-cloud-uid')
    request_id = request.headers.get('x-request-id')

    # uid должен быть натуральным числом
    passport_uid = coerce_to_int_or_none(
        passport_uid,
        HeaderShouldBeIntegerError,
        'X-UID',
    )

    if passport_uid or cloud_uid_from_headers:
        user_ip = get_user_ip_from(headers)
        karma = None

        if not cloud_uid_from_headers:
            # Пока вычисляем карму лишь для модифицирующих методов
            if request.method.lower() != 'get':
                karma = get_karma(passport_uid, ip=user_ip)

            if passport_uid > MAX_UID_VALUE or passport_uid < 0:
                raise HeaderShouldBeIntegerError('X-UID')

        user_data = {
            'ip': user_ip,
            'karma': karma,
            'passport_uid': passport_uid,
            'cloud_uid': cloud_uid_from_headers,
            'iam_token': get_iam_token(headers),
            'request_id': request_id,
        }
        if cloud_uid_from_headers and passport_uid is None:
            user_data['is_cloud'] = True

        return User(**user_data)


def parse_service_ticket(ticket):
    try:
        data = app.tvm2_client.parse_service_ticket(ticket)
    except Exception:
        data = None
    return data


def authenticate_by_tvm2(token, headers, no_more_tokens):
    """
    Авторизация по TVM тикетам
    :param token: TVM тикет
    """

    if not app.config['INTERNAL']:
        raise AuthenticationError(
            'Our API requires OAuth authentication')

    # X-Real-Ip
    user_ip = get_client_ip_from(headers)

    # При логгировании мы отрезаем последнюю часть
    # токена, как советовал @cerevra в переписке про
    # проблему парсинга тикетов от Почты.
    obfuscated_ticket = obfuscate_ticket(token)

    with log.fields(ticket=obfuscated_ticket):
        data = parse_service_ticket(token)
        if not data:
            log.warning('Unable to verify ticket')

            # Исключения выбрасываем только если нет других
            # способов для аутентификации запроса
            if no_more_tokens:
                raise AuthenticationError(
                    'Bad credentials',
                    headers={
                        'WWW-Authenticate': "TVM error='Unable to verify ticket.'"
                    }
                )
            else:
                return None

        with get_meta_connection() as meta_connection:
            service_model = ServiceModel(meta_connection)
            service = service_model.find(
                filter_data={
                    'tvm2_client_ids__contains': data.src,
                },
                one=True,
            ) or service_model.find(
                filter_data={
                    'team_tvm2_client_ids__contains': data.src,
                },
                one=True) or {}

            if not service:
                log.error('Unknown service')

                # Исключения выбрасываем только если нет других
                # способов для аутентификации запроса
                if no_more_tokens:
                    raise AuthenticationError(
                        'Unknown service',
                        headers={
                            'WWW-Authenticate': "TVM error='Unknown service.'"
                        }
                    )
                else:
                    return None

            with log.fields(service=service):
                org_id = headers.get('x-org-id')
                if org_id:
                    try:
                        org_id = int(org_id)
                    except ValueError:
                        log.warning('X-ORG-ID should be an integer')
                        # Исключения выбрасываем только если нет других
                        # способов для аутентификации запроса
                        if no_more_tokens:
                            raise HeaderShouldBeIntegerError('X-ORG-ID')
                        else:
                            return None

                # Для DIR-3233 мы запрещаем доступ для yamb-test ко всем организациям,
                # кроме описанных в YAMB_TEST_ORGANIZATIONS
                if service['slug'] == 'yamb-test' and org_id not in app.config['YAMB_TEST_ORGANIZATIONS']:
                    # Исключения выбрасываем только если нет других
                    # способов для аутентификации запроса
                    if no_more_tokens:
                        raise NotAYambTestOrganizationError()
                    else:
                        return None

                # Если сервис - внутренняя админка, то он должен ещё передать TVM 2.0 user ticket
                team_user = None
                if service['slug'] == INTERNAL_ADMIN_SERVICE_SLUG:
                    team_user = get_user_from_tvm2(headers)
                    if not isinstance(team_user, TeamUser):
                        raise CatalogRouteForbiddenError()

                with log.fields(org_id=org_id):
                    # TVM сервисы могут передавать список UID пользователя
                    # и их может быть больше одного при мультиавторизации.
                    # Тут мы определяем конкретного пользователя, от имени которого
                    # пытаются выполнить запрос.
                    # Можно ли им работать от имени этого уида, определяется на этапе
                    # авторизации.

                    # Если есть team_user, это значит, что к нам пришла внутренняя админка
                    # с TVM2 пользовательским тикетом
                    user = team_user or get_user_from_headers(headers)

                    return {
                        'auth_type': 'tvm2',
                        'service': Service(
                            id=service.get('id'),
                            name=service['name'],
                            identity=service['slug'],
                            # Если сервис добавлен в директорию, то ему доступны внутренние ручки
                            # они с декоратором @internal.
                            # Когда у нас будет Marketplace, то нам надо будет уметь различать
                            # внутренние сервисы и внешние.
                            is_internal=True,
                            ip=user_ip,
                        ),
                        # Для TVM 2.0 токенов список скоупов пока берём
                        # из своей собственной базы.
                        'scopes': service['scopes'] or [],
                        'user': user,
                        'org_id': org_id,
                        'user_ticket': headers.get('X-Ya-User-Ticket')
                    }


def get_user_from_tvm2(headers):
    user_ticket = headers.get('X-Ya-User-Ticket')
    if not user_ticket:
        log.warning('Header X-Ya-User-Ticket is required')
        raise HeaderIsRequiredError('X-Ya-User-Ticket')
    user_data = app.tvm2_client.parse_user_ticket(user_ticket)
    if not user_data:
        raise AuthenticationError(
            'Bad credentials',
            headers={
                'WWW-Authenticate': "TVM error='Unable to verify ticket.'"
            }
        )
    provided_uid = headers.get(
        'x-uid',
        user_data.default_uid
    )
    # uid должен быть натуральным числом
    provided_uid = coerce_to_int_or_none(
        provided_uid,
        HeaderShouldBeIntegerError,
        'X-UID',
    )

    user_id = None
    for uid in user_data.uids:
        if uid == provided_uid:
            user_id = uid
            break

    if not user_id:
        raise UserNotFoundInBlackbox()

    user_ip = get_user_ip_from(headers)
    if is_yandex_team_uid(user_id):
        user = TeamUser(passport_uid=user_id, ip=user_ip)
    else:
        user = User(passport_uid=user_id, ip=user_ip)
    return user


AUTHENTICATORS_ADMINS_VIEW = dict(
    tvm2=authenticate_by_tvm2,
)

AUTHENTICATORS_USERS_VIEW = dict(
    oauth=authenticate_by_oauth,
    token=authenticate_by_token,
    tvm2=authenticate_by_tvm2,
)


class AuthMiddleware(object):
    def __init__(self, app):
        app.before_request(self.authenticate_and_authorize)

    @log_work_time
    def authenticate_and_authorize(self):
        # сначала считаем что пользователь не аутентифицирован и про
        # сервис мы тоже ничего не знаем
        g.service = None
        g.user = None
        g.user_ticket = None
        g.org_id = None
        g.shard = None
        g.scopes = []

        try:
            if request.method == 'OPTIONS':
                return

            if not request.endpoint:
                return

            view_function = get_view_function()
            # по умолчанию, считаем, что каждая view требует аутентификации
            # и авторизации, но если весь механизм авторизации
            # нужно отключить, то необходимо использовать декоратор `@no_auth`.
            #
            # ВНИМАНИЕ, мало какая view должны быть без аутентификации
            requires_authentication = getattr(view_function, 'require_auth', True)

            if requires_authentication:
                response = self._authenticate()
                auth_fields = get_auth_fields_for_log(response)
                with log.fields(auth=auth_fields):
                    log.debug('Request was authenticated')
                    self._authorize(response, view_function)
                    log.debug('Request was authorized')

        except APIError as e:
            with log.fields(
                    name='middlewares.auth-middleware',
                    response_code=e.status_code,
                    response=e.message,
                    response_headers=hide_auth_from_headers(e.headers),
                    url=request.url,
                    request_headers=hide_auth_from_headers(request.headers),
            ):
                log.warning('Auth failed: {}'.format(e.message))
            data = {'message': e.message}
            if e.code:
                data['code'] = e.code
            response = json_error(
                e.status_code,
                e.code,
                e.message,
                headers=e.headers,
                **e.params
            )
            return response
        except Exception as ex:
            log.trace().error('Unhandled exception')
            return json_error_exception(ex)

    def _authenticate(self):
        auth_tokens = list(get_authorization_tokens())

        if auth_tokens:
            tokens_left = len(auth_tokens)

            # DIR-5516
            # тут нужно определить тип ручки
            view_type = 'users_view'
            if request.path.startswith('/admin/'):
                view_type = 'admins_view'

            # Начинаем проверять все пришедшие токены
            for auth_type, token in auth_tokens:
                tokens_left -= 1
                no_more_tokens = (tokens_left == 0)

                if view_type == 'users_view':
                    authenticator = AUTHENTICATORS_USERS_VIEW.get(auth_type)
                else:
                    authenticator = AUTHENTICATORS_ADMINS_VIEW.get(auth_type)
                response = None
                if authenticator:
                    response = authenticator(token, request.headers, no_more_tokens)
                # Исключение выбрасывается только в том случае, если
                # перепробованы все переданные токены – например если
                # не удалось ни TVM 2.0 ни TVM 1.0 токен проверить.
                if response is None and no_more_tokens:
                    raise AuthenticationError(
                        'Bad credentials')
                else:
                    return response
        else:  # пользователь не передал никакого токена
            raise AuthenticationError(
                'Our API requires authentication')

    def _authorize(self, auth_response, view_function):
        # если параметры доступа к ручке не заданы декоратором
        # `requires`, то считаем, что и пользователь и org_id
        # обязательно должны быть известны
        requires_org_id = getattr(view_function, 'requires_org_id', True)
        requires_user = getattr(view_function, 'requires_user', True)

        # по умолчанию, считаем все ручки внешними
        # чтобы сделать внутреннюю ручку, её нужно обернуть декоратором
        # `@internal`
        internal = getattr(view_function, 'internal', False)

        with get_meta_connection() as meta_connection:
            org_id, shard, revision = authorize(
                meta_connection,
                org_id=auth_response['org_id'],
                service=auth_response['service'],
                user=auth_response['user'],
                scopes=auth_response['scopes'],
                require_org_id=requires_org_id,
                require_user=requires_user,
                internal_only=internal,
            )
            # Функция authorize возвращает id организации
            # с которой должен работать данный запрос.
            g.org_id = org_id
            g.shard = shard
            g.revision = revision
            g.auth_type = auth_response.get('auth_type')
            g.user = auth_response.get('user')
            g.user_ticket = auth_response.get('user_ticket')
            g.service = auth_response.get('service')
            g.scopes = auth_response['scopes']


class IncomingRequestStatsMiddleware(object):
    """
    Пишет в Голован статистику входящих в сервис запросов.
    """

    def __init__(self, app):
        app.after_request(self.after_request)

    def normalize_signal_name(self, name):
        # '/v<int:api_version>/users/<int:user_id>/aliases/<alias>/ ' -> v_api_version__users__user_id__aliases__alias
        return name.translate(SIGNAL_NAME_TRANSTAB). \
            replace('int:', ''). \
            replace('/', '__'). \
            strip('_-')

    def __get_service_name(self):
        if hasattr(g, 'service') and hasattr(g.service, 'identity'):
            service_slug = g.service.identity.lower()
        else:
            service_slug = 'not_defined'
        return self.normalize_signal_name(service_slug)

    def _stats_incoming_requests(self):
        signals = []
        for signal in signals:
            try:
                app.stats_aggregator.inc(signal)
            except ValueError:
                with log.fields(signal=signal):
                    log.error('Invalid signal name')

    def after_request(self, response, *args, **kwargs):
        # Отдельные графики по числу запросов от каждого из потребителей API
        try:
            self._stats_incoming_requests()
        except Exception as exc:
            log.exception('Error during add request count stats: "%s"', str(exc))

        return response


class DCMiddleware(object):
    """Выдает в заголовке X-DC идентификатор датацентра.
    """

    def __init__(self, app):
        self.dc = os.environ.get('DEPLOY_NODE_DC', os.environ.get('QLOUD_DATACENTER'))
        # заголовок прописываем только если запущены
        # в Qloud
        if self.dc:
            app.after_request(self.after_request)

    def after_request(self, response, *args, **kwargs):
        response.headers.add_header('X-DC', self.dc)
        return response


class ValidateOutputData(object):
    """
    Проверяет, что формат возвращаемых данных соответвует схеме.
    Используется в тестах.
    """

    def __init__(self, app):
        # Включаем проверку только в автотестах
        if app.testing:
            app.after_request(self.validate)

    @log_work_time
    def validate(self, *args, **kwargs):
        view_function = get_view_function()
        out_schema = getattr(view_function, 'out_schema', {})
        response = args[0]
        # Будем проверять только неошибочные ответы
        if out_schema and response.status_code < 400:
            data = json.loads(response.data)
            result = data.get('result', data)
            errors = validate_data_by_schema(result, schema=out_schema)
            if errors:
                with log.fields(errors=errors[0]['message']):
                    log.error('Output format does not match to validation schema')

                return json_error(
                    500,
                    'outputs_schema_is_invalid',
                    ('Output\'s schema is invalid and '
                     'has following errors: {errors}'),
                    errors=errors,
                )
        return response


class APIVersionMiddleware(object):
    """
    Сохраняет в request.api_version запрошенную версию API.
    Если запрошена неподдерживаемая версия, возвращает ошибку с кодом 400.
    """

    def __init__(self, app):
        app.before_request(self.set_api_version)
        app.before_request(self.set_route)

    def set_api_version(self):
        try:
            request.api_version = int(request.environ.get('API_VERSION', 1))
        except (ValueError, TypeError):
            request.api_version = 0

        # https://st.yandex-team.ru/DIR-3571
        # Делаем исключение по версионированию для /monitoring/ & /ping/

        view = app.view_functions.get(request.endpoint)
        # для внешних blueprint
        if view and not hasattr(view, 'view_class'):
            return

        if view and view.view_class.__name__ in app.config['VIEW_CLASSES_WITHOUT_VERSION']:
            # Доступна только 1я версия. Сделано для удобства, чтобы можно было в qloud закрыть за роутом этот метод
            if request.api_version == 1:
                return
            return json_error_not_found()

        if request.api_version not in app.config['SUPPORTED_API_VERSIONS']:
            log.trace().error('Unsupported api version')
            return json_error(
                400,
                'unsupported_api_version',
                'Supported API versions: %s' % app.config['SUPPORTED_API_VERSIONS'],
            )

    def set_route(self):
        route = str(request.url_rule)
        if route and route[0] != '/':
            route = '/' + route

        api_version = request.api_version

        version = '/v{}'.format(api_version)
        if api_version:
            route = route.replace('/v<int:api_version>', version)
            route = route.replace('/v<api_version>', version)
        else:
            route = version + route
        # для роутов типа /users/ припишем версию -> /v1/users/
        if not route.startswith('/v'):
            route = version + route

        g.route = route.lower()
