import threading
import time

import getpass
from tvmauth import BlackboxTvmId as BlackboxClientId
from tvm2 import TVM2

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import get_meta_connection
from intranet.yandex_directory.src.yandex_directory.core.models import (
    ServiceModel,
    WebHookModel,
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log


def get_tvm2_ticket(destination):
    return app.tvm2_client.get_service_ticket(str(destination))


class TicketProvider:
    def __init__(self):
        self.is_inited = False
        self.local_cache = {}
        self.apps = {}

    def get_destinations(self):
        blackbox_app = app.config['BLACKBOX']['type']
        team_blackbox_app = app.config['TEAM_BLACKBOX']['type']
        destinations = {
            blackbox_app: app.config['BLACKBOX_TVM_CLIENT_ID'],
            team_blackbox_app: app.config['TEAM_BLACKBOX_TVM_CLIENT_ID'],
            'disk': app.config['DISK_TVM_CLIENT_ID'],
            'bigml': app.config['BIGML_TVM_CLIENT_ID'],
            'forms': app.config['FORMS_TVM_CLIENT_ID'],
            'gozora': app.config['ZORA_TVM_CLIENT_ID'],
            'metrika': app.config['METRIKA_TVM_CLIENT_ID'],
            'gendarme': app.config['GENDARME_TVM_CLIENT_ID'],
            'fouras': app.config['FOURAS_TVM_CLIENT_ID'],
            'yandexsprav': app.config['YANDEXSPRAV_TVM_CLIENT_ID'],
            'idm_b2b': app.config['IDM_B2B_TVM_CLIENT_ID'],
            'direct': app.config['DIRECT_TVM_CLIENT_ID'],
            'partner_disk': app.config['PARTNER_DISK_TVM_CLIENT_ID'],
            'billing': app.config['BILLING_TVM2_CLIENT_ID'],
            'billing_invoice': app.config['BILLING_INVOICE_API_TVM2_CLIENT_ID'],
            'alice_b2b': app.config['ALICE_B2B_TVM_CLIENT_ID'],
            'domenator': app.config['DOMENATOR_TVM_CLIENT_ID'],
            'sso_config': app.config['SSO_CONFIG_TVM_CLIENT_ID'],
        }

        # Также получаем тикеты для всех вебхуков, у которых указан tvm client id.
        with get_meta_connection() as meta_connection:
            webhooks = WebHookModel(meta_connection).find(
                fields=['url', 'tvm_client_id'],
            )
            for hook in webhooks:
                if hook['tvm_client_id']:
                    destinations[hook['url']] = hook['tvm_client_id']
        return destinations

    def get_clients(self):
        clients = {}
        with get_meta_connection() as meta_connection:
            services = ServiceModel(meta_connection).find(
                fields=['slug', 'tvm2_client_ids']
            )
        for service in services:
            if service.get('tvm2_client_ids'):
                clients['service_' + service['slug']] = service['tvm2_client_ids']
        return clients

    def get_tvm_id_set(self):
        destinations = self.get_destinations()
        clients = self.get_clients()
        destionation_set = set(destinations.values())
        client_set = set()
        for tvm_id_or_list in clients.values():
            if isinstance(tvm_id_or_list, (int, str)):
                client_set.add(tvm_id_or_list)
            else:
                client_set.update(tvm_id_or_list)
        return client_set, destionation_set

    def update_services(self):
        # Словарик приложение -> client_id
        destinations = self.get_destinations()
        self.apps = destinations
        self.team_blackbox_app = destinations[app.config['TEAM_BLACKBOX']['type']]
        self.is_inited = True

    def __getitem__(self, service_slug):
        if service_slug in self.local_cache:
            return self.local_cache[service_slug]

        if not self.is_inited:
            self.update_services()

        client_id = self.apps[service_slug]

        return app.tvm2_client.get_service_ticket(str(client_id))

    def __setitem__(self, key, value):
        self.local_cache[key] = value

    def __iter__(self):
        for key in self.local_cache:
            yield key
        for key in self.apps:
            yield key


tickets = TicketProvider()
tickets_services_updater_thread = None


def tickets_services_updater():
    while True:
        with log.name_and_fields('tvm_keys_updater'):
            try:
                tickets.update_services()
            except Exception:
                log.trace().error('Unable to update TVM keys')
                # Если была ошибка, то будем ждать не 10 минут,
                # а лишь минуту
                time.sleep(60)
            else:
                # поспим, прежде чем обновлять снова
                time.sleep(60 * 10)


def setup_tvm(app):
    global tickets_services_updater_thread

    if app.config['ENVIRONMENT'] in ('autotests', 'development', getpass.getuser()):
        app.tvm2_client = None
        return

    TVM2._instance = None
    client_set, destionation_set = tickets.get_tvm_id_set()
    app.tvm2_client = TVM2(
        client_id=app.config['TVM_CLIENT_ID'],
        secret=app.config['TVM_SECRET'],
        blackbox_client=BlackboxClientId.ProdYateam,
        allowed_clients=client_set,
        destinations=destionation_set,
    )

    tickets.update_services()

    tickets_services_updater_thread = threading.Thread(
        target=tickets_services_updater,
    )
    tickets_services_updater_thread.daemon = True
    tickets_services_updater_thread.start()
