# -*- coding: utf-8 -*-

import os

import flask

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.caching.utils import make_key
from intranet.yandex_directory.src.yandex_directory.core.models.organization import (
    OrganizationModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.user import (
    UserMetaModel,
)
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import (
    get_permissions,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.internal_permissions import get_admin_permissions
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    is_outer_uid,
)


# кэш для метаданных про организацию
# нам на каждый запрос приходится ходить в базу за получением номера шарда
# чтобы этого не делать, мы можем доставать информацию из этого кэша
from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
    get_domains_from_db_or_domenator,
    DomainFilter,
)

#cache_size = 0 if os.environ.get('ENVIRONMENT') == 'autotests' else 1000
#users_meta_cache = LRUCache(maxsize=cache_size)
#users_meta_lock = RLock()

#org_cache = LRUCache(maxsize=cache_size)
#org_lock = RLock()


# Отключаем. Проблемы с инвалидацией при откате миграции.
# @cached(users_meta_cache,
#         key=ignore_first_args(1)(hashkey),
#         lock=users_meta_lock)
def get_users_meta(meta_connection, uid):
    users_meta = UserMetaModel(meta_connection).find(
        filter_data={'id': uid},
        fields=[
            '*',
            'organization.*',
        ],
    )
    return users_meta


# Отключаем. Проблемы с инвалидацией при откате миграции.
# @cached(org_cache,
#         lock=org_lock)
def get_organization_by(shard, org_id, fields=None):
    with get_main_connection(shard) as main_connection:
        model = OrganizationModel(main_connection)
        return model.get(org_id, fields=fields)


class User(object):
    def __init__(self, passport_uid=None, ip=None, *, cloud_uid=None, karma=None, is_cloud=False,
                 iam_token=None, request_id=None):
        if not is_cloud:
            self.passport_uid = int(passport_uid)
            self.is_outer = is_outer_uid(self.passport_uid)
            self.cloud_uid = cloud_uid
        else:
            self.passport_uid = passport_uid
            if self.passport_uid is not None:
                self.passport_uid = int(self.passport_uid)
            self.cloud_uid = cloud_uid
            self.is_outer = True

        self.is_cloud = is_cloud
        self.has_cloud_header = self.cloud_uid is not None
        self.ip = ip
        self.iam_token = iam_token
        # Это поле заполняется в момент авторизации
        self.org_ids = []
        self.role = None
        self.karma = karma
        self.request_id = request_id

    def get_cloud_or_passport_uid(self):
        if self.is_cloud:
            return self.cloud_uid
        else:
            return self.passport_uid

    def __repr__(self):
        outer = 'outer' if self.is_outer else 'not_outer'
        return f'<User({outer}, {self.is_cloud}) passport_uid={self.passport_uid}, cloud_uid={self.cloud_uid}>'

    def has_permissions(self,
                        meta_connection,
                        main_connection,
                        permissions,
                        object_type=None,
                        object_id=None,
                        any_permission=False,
                        org_id=None):
        # Кэшируем, если только известен номер ревизии, чтобы не пропустить
        # изменения в правах доступа. (Например смену тарифного плана орг.)
        if flask.g.revision is None:
            return self._has_permissions(
                meta_connection,
                main_connection,
                permissions,
                object_type,
                object_id,
                any_permission,
                org_id
            )
        else:
            key = make_key(
                prefix='has_permissions',
                key_data=(
                    flask.g.revision,
                    self.passport_uid,
                    org_id,
                    object_type,
                    object_id,
                    sorted(permissions),
                    any_permission
                )
            )

            result = app.cache.get(key)
            if result is None:
                result = self._has_permissions(
                    meta_connection,
                    main_connection,
                    permissions,
                    object_type,
                    object_id,
                    any_permission,
                    org_id
                )
                app.cache.set(key, result)

            return result

    def _has_permissions(self,
                         meta_connection,
                         main_connection,
                         permissions,
                         object_type=None,
                         object_id=None,
                         any_permission=False,
                         org_id=None):
        """
        Проверяет есть ли у пользователя все права (или одно, если any_permission=True) из списка permissions
        Args:
            permissions (list|tuple): список строк из мн-ва
                core.views.permissions.global_permissions и
                core.views.permissions.group_permissions и т.д
                Наличие этого права доступа и будем проверять
            object_type (string): тип объекта
                {'group', 'user', 'department', 'resource'}
                Нужен для проверки прав на один объект с его id
            object_id (int): id пользователя, департамента или группы
            any_permission (bool): проверка должна проходить если у пользователя есть хотя бы одно право из списка
        """
        assert isinstance(permissions, (list, tuple)), 'permissions must be a list or tuple'

        allow_permissions = get_permissions(
            meta_connection,
            main_connection,
            self.passport_uid,
            object_type,
            object_id,
            org_id=org_id,
            is_cloud=self.is_cloud,
        )

        return self._return_permissions(permissions, allow_permissions, any_permission)

    @staticmethod
    def _return_permissions(permissions, allow_permissions, any_permission):
        if any_permission:
            result = any([p for p in permissions if p in allow_permissions])
        else:
            result = set(permissions).issubset(allow_permissions)
        return result


class TeamUser(User):
    def __repr__(self):
        return f'<Team user uid={self.passport_uid}>'

    def has_permissions(self,
                        meta_connection,
                        main_connection,
                        permissions,
                        object_type=None,
                        object_id=None,
                        any_permission=False,
                        org_id=None,
                        ):
        assert isinstance(permissions, (list, tuple)), 'permissions must be a list or tuple'

        allow_permissions = get_admin_permissions(self.passport_uid)

        return self._return_permissions(permissions, allow_permissions, any_permission)
