# -*- coding: utf-8 -*-
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.common.exceptions import APIError

from intranet.yandex_directory.src.yandex_directory.common.utils import (
    log_service_response,
    url_join,
)
from functools import (
    partial,
)

from intranet.yandex_directory.src.yandex_directory.common import http_client


class BigmlError(APIError):
    code = 'unknown_bigml_error'
    message = 'Unknown BigML error'
    description = 'Сервис рассылок ответил ошибкой'

    def __init__(self, error_code, error_message, params):
        super(BigmlError, self).__init__(**params)
        self.error_code = error_code
        self.code = 'bigml_' + str(error_code).lower()
        self.message = error_message


def process_response(bigml_response):
    if bigml_response.status_code >= 500:
        raise BigmlError(bigml_response.status_code, 'BigNL not available', {})
    if bigml_response.status_code == 404:
        raise BigmlError(bigml_response.status_code, 'BigNL 404 error', {})

    json_response = bigml_response.json()
    error = json_response['status'] == 'error'
    if error:
        error_response = json_response['response']
        with log.fields(error=error_response):
            log.error('BigML returned error')
            raise BigmlError(
                error_response.get('code'),
                error_response.get('message') or 'No message',
                error_response.get('params'),
            )
    return json_response


def _make_request(method, uri, get_params=None, post_params=None):
    ticket = tvm.tickets['bigml']
    headers = {
        'X-Ya-Service-Ticket': ticket,
    }
    bigml_url = app.config['BIGML_API']
    if not bigml_url:
        raise RuntimeError('BIGML base URL is unknown. Set BIGML_API variable')

    url = url_join(bigml_url, uri, query_params=get_params)
    response = http_client.request(method, url, json=post_params, headers=headers)

    log_service_response('bigml', method, url, post_params, response)
    return process_response(response)


_get = partial(_make_request, 'get')
_post = partial(_make_request, 'post')


def get_maillist_subscribers(maillist_uid):
    data = _get('/api/v1/support', get_params={'list_uid': maillist_uid})
    return [item['uid'] for item in data['response']]


def sync_organization(org_id):
    return _post('/api/v1/sync/directory', post_params={'org_id': org_id})
