# -*- coding: utf-8 -*-
import random
import datetime
from copy import deepcopy

from flask_script import Option
from flask import g

from subprocess import check_output

from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.common.commands.base import (
    TransactionalCommand,
    CommandError,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    UserModel,
    GroupModel,
    DepartmentModel,
    ServiceModel,
    DomainModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.organization import (
    OrganizationModel,
)
from intranet.yandex_directory.src.yandex_directory.common.trace import Node
from intranet.yandex_directory.src.yandex_directory.common import json
from intranet.yandex_directory.src.yandex_directory.common.utils import force_text, utcnow
from intranet.yandex_directory.src.yandex_directory.common.trace import trace_node

from intranet.yandex_directory.src.yandex_directory.common.db import (
    set_permanent_meta_connection,
    set_permanent_main_connection,
)

from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_organization_add,
    action_organization_delete,

    action_domain_add,
    action_domain_delete,
    action_domain_master_modify,

    action_department_add,
    action_department_delete,
    action_department_modify,
    action_department_alias_add,
    action_department_alias_delete,

    action_group_add,
    action_group_delete,
    action_group_modify,
    action_group_admins_change,
    action_group_alias_add,
    action_group_alias_delete,

    action_user_add,
    action_user_modify,
    action_user_alias_add,
    action_user_alias_delete,
    # тестируется за счёт вызова метода модели
#    action_user_dismiss,

    action_security_user_grant_organization_admin,
    action_security_user_revoke_organization_admin,
    action_security_user_avatar_changed,
    action_security_user_password_changed,
    action_security_user_unblocked,
    action_security_user_blocked,

    action_service_enable,
    action_service_disable,
    action_service_set_ready,
    action_service_license_change,
    action_service_trial_end,

    action_organization_subscription_plan_change,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    create_organization as original_create_organization,
    build_email,
)

def create_organization(
        meta_connection,
        main_connection,
        name={'ru': 'Яндекс'},
        label='not_yandex_test',
        domain_part='.ws.autotest.yandex.ru',
        admin_uid=None,
        source='autotest',
        language='ru',
        tld='ru',
        root_dep_label=None,
        ready=True
    ):

    if not admin_uid:
        admin_uid = random.randint(113*10**13, 114*10**13)

    admin_nickname = 'admin'
    admin_gender = 'male'
    admin_birthday = utcnow().date()

    domain = label + domain_part

    info = original_create_organization(
        meta_connection,
        main_connection,
        name=name,
        domain_part=domain_part,
        label=label,
        language=language,
        admin_uid=admin_uid,
        admin_nickname=admin_nickname,
        admin_first_name='Admin',
        admin_last_name='Adminovich',
        admin_gender=admin_gender,
        admin_birthday=admin_birthday,
        source=source,
        tld=tld,
        root_dep_label=root_dep_label,
        ready=ready,
    )

    dep_email = build_email(main_connection, root_dep_label, info['organization']['id'])
    # чтобы диффы событий были с правильными счетчиками
    # надо обновить счетчик на корневом отделе
    DepartmentModel(main_connection).update_members_count(
        1,
        info['organization']['id'],
    )
    return info



DIRECTORY_ROOT_DEPARTMENT = 1


def make_header(text, level):
    template = '{1}\n{0}\n\n'

    if level == 0:
        template = '{0}\n{1}\n{0}\n\n'
        char = '='
    elif level == 1:
        char = '='
    elif level == 2:
        char = '-'
    elif level == 3:
        char = '~'
    elif level == 4:
        char = '^'
    elif level == 5:
        char = "+"
    elif level == 6:
        char = "'"
    elif level == 7:
        char = "`"
    elif level == 8:
        char = '#'
    else:
        raise RuntimeError('Header levels > 8 are not unsupported')

    underline = char * len(text)
    return template.format(underline, text)


def indent(text, spaces):
    indent_string = ' ' * spaces
    lines = text.split('\n')
    lines = (indent_string + line
             for line in lines)
    new_text = '\n'.join(lines)
    return new_text


def tree2rst(doc, level=0):
    """Принимает дерево событий и вызовов, а на выходе отдает
    текст в reStructured формате.
    """

    if isinstance(doc, Node):
        title = '{0}: {1}'.format(force_text(doc.type), force_text(doc.name))
        data = [tree2rst(item, level=level+1)
                for item in doc.data]
        return '{0}\n{1}'.format(
            indent('**{0}**'.format(title), level * 2),
            '\n'.join(data)
        )
    else:
        dumped_doc = json.dumps(
            doc,
            indent=4,
            ensure_ascii=False,
            sort_keys=True,
        )
        snippet = '.. code:: json\n\n{0}\n'.format(
            indent(dumped_doc, 3)
        )
        return indent(snippet, level * 2)


def get_uniq_items_from_tree(doc, item_type='action'):
    result = set([])
    if isinstance(doc, Node):
        if doc.type == item_type:
            result.add(doc.name)

        for item in doc.data:
            result.update(get_uniq_items_from_tree(item, item_type))

    return result



def grep_uniq_items_from_code(prefix='action'):
    """Ищет по коду и возвращает коды действий.

    Список содержит уникальные имена, и используется
    для проверки того, что все на все действия или события
    генерится документация.
    """
    command = (
        "git grep ' {0}_[^ .]*(' | "
        "awk '{{print $NF}}' | "
        "sed 's/(.*//' | sort -u"
    ).format(prefix)

    output = check_output(command, shell=True)
    ignore = set([
        '',
        # это на самом деле не настоящая функция, а временная переменная
        'action_method',
        # и эта переменная тоже - используется в качестве параметра
        'event_func',
        # это просто функция, которая генерит resource_grant_changed
        'event_resource_changed_for_each_resource',
    ])
    lines = output.split('\n')
    lines = [line.replace(prefix + '_', '')
             for line in lines
             if line not in ignore]
    result = set(lines)

    # добавим ещё и содержимое модулей
    if prefix == 'action':
        from intranet.yandex_directory.src.yandex_directory.core import actions as module
    else:
        from intranet.yandex_directory.src.yandex_directory.core import events as module

    result.update(
        name.replace(prefix + '_', '')
        for name in dir(module)
        if name.startswith(prefix + '_')
    )
    return result


class Command(TransactionalCommand):
    """Генерит файлик в reStructured text, с описанием действий и событий.

    Кстати вот так можно сгенерить список всех действий вызываемых в коде:

    git grep ' action_[^ .]*(' | awk '{print $NF}' | sed 's/(.*//' | sort -u

    А так список событий:

    git grep ' event_[^ .]*(' | awk '{print $NF}' | sed 's/(.*//' | sort -u

    """

    name = 'build-events-doc'
    option_list = (
        Option(
            '--output',
            '-o',
            dest='file_path',
            required=True,
            help='Filename to write results.',
        ),
        # делаем параметр --shard необязательным, так как в TransactionalCommand
        # он обязательный
        Option(
            '--shard',
            '-s',
            dest='shard',
            type=int,
            required=False,
            default=1,
            help='Shard number.',
        )
    )

    def run(self, file_path, *args, **kwargs):
        # Эта команда предназначена для запуска в дев-окружении,
        # поэтому, чтобы не тащить модуль mock
        # мы импортируем его здесь.
        #
        # Дело в том, что все manage команды импортируются
        # при старте manage.py и если этот импорт перенести
        # наверх файла, то любая management команда в проде или
        # тестинге будет падать из-за того, что нет модуля mock
        from unittest.mock import patch
        from testutils import mocked_passport

        with self.meta_connection.begin_nested() as meta_transaction,\
             self.main_connection.begin_nested() as main_transaction,\
             mocked_passport():

            set_permanent_meta_connection(self.meta_connection)
            set_permanent_main_connection(self.main_connection)

            self.service_model = ServiceModel(self.meta_connection)
            self.user_model = UserModel(self.main_connection)
            self.group_model = GroupModel(self.main_connection)
            self.department_model = DepartmentModel(self.main_connection)
            self.domain_model = DomainModel(self.main_connection)

            self.organization = create_organization(
                self.meta_connection,
                self.main_connection,
                name={'ru': 'Дока для событий'},
                label='event-docs',
                source='event-docs',
            )
                
            self.org_id = self.organization['id']
            self.admin_uid = 100500 #  self.organization['admin_uid']

            self.robots_dep = self.department_model.create(
                {'ru': 'Robots'},
                self.org_id,
                parent_id=1,
                label='robots',
            )
            self.robots_dep_id = self.robots_dep['id']

            self.bob_uid = 100500
            self.bob = self.user_model.create(
                self.bob_uid,
                'bob',
                {
                    'first_name': {'ru': 'Bob'},
                    'last_name': {'ru': 'Hopkins'},
                },
                email='bob@example.com',
                gender='male',
                org_id=self.org_id,
                department_id=self.robots_dep_id,
                position={'ru': 'Разработчик'},
                contacts=[
                    {'type': 'skype', 'value': 'the-bob'},
                ],
                aliases=['jerry'],
            )

            # этим обработчиком мы будем симулировать
            # передачу события подписчику
            def handled(*args, **kwargs):
                pass

            EVENTS = grep_uniq_items_from_code('event')
            SUBSCRIPTIONS = {}
            for event in EVENTS:
                SUBSCRIPTIONS[event] = [
                    {
                        'callback': handled,
                        # Эта настройка никак не влияет на callback
                        # так как в функцию всегда передаются расширенные
                        # данные, включающие поле content
                        # 'settings': {
                        #     'expand_content': True,
                        # },
                    }
                ]

            # теперь начнём отслеживать цепочки событий
            # Чтобы не было обращений к паспорту, надо запатчить reset_admin_option
            with trace_node('doc', 'Автоматическая документация') as doc:

                # будем вызывать разные действия
                self.simulate_user_actions()
                self.simulate_group_actions()
                self.simulate_department_actions()
                self.simulate_organization_actions()
                self.simulate_service_actions()
                self.simulate_organization_subscription_plan_changed()
                self.simulate_domain_actions()

                # проверим, что все действия и события покрыты документацией
                values = (
                    ('action', 'действия'),
                    ('event', 'события'),
                )
                for item_type, item_desc in values:
                    from_code = grep_uniq_items_from_code(item_type)
                    from_tree = get_uniq_items_from_tree(doc, item_type)
                    diff = from_code - from_tree
                    if diff:
                        print('Внимание, эти {0} остались непокрытыми:\n{1}\n'.format(
                            item_desc,
                            '\n'.join(sorted(diff))
                        ))


                # и потом соберём их все в файл
                with open(file_path, 'w') as f:
                    rst_doc = tree2rst(doc)
                    f.write('=================\n')
                    f.write(' Примеры событий\n')
                    f.write('=================\n')
                    f.write(rst_doc.encode('utf-8'))
                    f.write("""
    .. raw:: html

       <style>
            dt { cursor: pointer; }
       </style>
       <script src="https://yastatic.net/jquery/3.1.0/jquery.min.js"></script>

       <script>
            jQuery('pre').hide();
            jQuery('dl').removeClass('first');
            jQuery('dt').click(function on_click(ev) {
                ev.preventDefault();
                $(this).next().find('pre').toggle();
            });
       </script>
    """)

            # и конечно, за собой надо подчистить базу,
            # чтобы не осталось лишнего
            main_transaction.rollback()

    def simulate_organization_actions(self):
        self.simulate(
            'Работа с организациями',
            self.simulate_organization_add,
            self.simulate_organization_delete,
        )

    def simulate_organization_add(self):
        """Добавлена новая организация."""

        action_organization_add(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.organization,
        )

    def simulate_organization_delete(self):
        """Организация была удалена."""
        # тут не генерируется event: organization_deleted (не понятно пока, правильно это или нет, но action про
        # удаление организации используется только в одном месте - в функции rollback организации и в этой же функции
        # руками происходит генерация event-a прямо через send_all_*)
        action_organization_delete(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.organization,
        )

    def simulate_department_actions(self):
        self.simulate(
            'Работа с отделами',
            self.simulate_department_add,
            self.simulate_department_delete,
            self.simulate_department_rename,
            self.simulate_department_move,
            self.simulate_department_alias_add,
            self.simulate_department_alias_delete,
        )

    def simulate_department_rename(self):
        """Переименование отдела."""

        new_department = deepcopy(self.robots_dep)
        new_department['name'] = {'ru': 'Робототехника'}

        action_department_modify(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=new_department,
            old_object=self.robots_dep,
        )

    def simulate_department_move(self):
        """Перенос отдела в другой отдел."""

        other_dep = self.department_model.create(
            {'ru': 'Другой отдел'},
            self.org_id,
        )

        # делаем копию и меняем родительский отдел
        new_department = deepcopy(self.robots_dep)
        new_department['parent_id'] = other_dep['id']

        action_department_modify(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=new_department,
            old_object=self.robots_dep,
        )

    def simulate_department_add(self):
        """Добавление нового отдела"""
        action_department_add(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.robots_dep,
        )

    def simulate_organization_subscription_plan_changed(self):
        """Смена тарифного плана в организации"""
        action_organization_subscription_plan_change(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.organization,
            content={'subscription_plan': 'paid'},
        )

    def simulate_department_delete(self):
        """Удаление отдела"""
        self.department_model.remove(
            self.robots_dep_id,
            self.org_id,
            self.admin_uid,
            force=True,
        )

    def simulate_department_alias_add(self):
        """Добавление алиаса отдела"""
        action_department_alias_add(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.robots_dep,
            content={
                'alias_added': 'big_department'
            }
        )

    def simulate_department_alias_delete(self):
        """Удаление алиаса отдела"""
        action_department_alias_delete(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.robots_dep,
            content={
                'alias_deleted': 'small_department'
            }
        )

    def simulate_user_actions(self):
        self.simulate(
            'Работа с сотрудниками',
            self.simulate_user_add,
            self.simulate_user_modify,
            self.simulate_user_move_to_department,
            self.simulate_security_user_grant_organization_admin,
            self.simulate_security_user_revoke_organization_admin,
            self.simulate_security_user_avatar_changed,
            self.simulate_security_user_password_changed,
            self.simulate_security_user_unblocked,
            self.simulate_security_user_blocked,
            self.simulate_user_dismiss,
            self.simulate_user_alias_add,
            self.simulate_user_alias_delete,
        )

    def simulate_user_dismiss(self):
        """Сотрудника уволили."""

        self.user_model.dismiss(
            self.org_id,
            self.bob_uid,
            author_id=self.admin_uid,
            skip_disk=True,
            skip_passport=True,
        )

    def simulate_security_user_password_changed(self):
        """У сотрудника сменили пароль."""
        action_security_user_password_changed(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.bob['id'],
        )

    def simulate_security_user_blocked(self):
        """Сотрудника заблокировали."""
        action_security_user_blocked(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.bob['id'],
        )

    def simulate_security_user_unblocked(self):
        """Сотрудника разаблокировали."""
        action_security_user_unblocked(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.bob['id'],
        )

    def simulate_security_user_avatar_changed(self):
        """У сотрудника сменили аватарку."""
        action_security_user_avatar_changed(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.bob['id'],
        )

    def simulate_security_user_grant_organization_admin(self):
        """Сотрудника сделали администратором организации"""
        action_security_user_grant_organization_admin(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.bob,
        )

    def simulate_security_user_revoke_organization_admin(self):
        """У сотрудника отобрали право администрировать организацию"""
        action_security_user_revoke_organization_admin(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.bob,
        )

    def simulate_user_add(self):
        "Добавление сотрудника"
        action_user_add(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.bob,
        )

    def simulate_user_modify(self):
        """Изменение сотрудника"""
        # сделаем вид, что раньше у Боба была позиция Тестировщик
        old_bob = deepcopy(self.bob)
        old_bob['position'] = {'ru': 'Тестировщик'}
        # и один из контактов был удалён
        old_bob['contacts'].append(
            {
                'type': 'email',
                'value': 'bob@home.com',
            }
        )

        action_user_modify(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.bob,
            old_object=old_bob,
        )

    def simulate_user_move_to_department(self):
        """Перевод сотрудника в другой отдел"""
        # сделаем вид, что раньше у Боб числился в рутовом отделе
        old_bob = deepcopy(self.bob)
        old_bob['department_id'] = 1
        # а потом его перевели в роботов
        assert self.bob['department_id'] == self.robots_dep_id

        action_user_modify(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.bob,
            old_object=old_bob,
        )

    def simulate_user_alias_add(self):
        action_user_alias_add(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.bob,
            content={
                'alias_added': 'jerry'
            }
        )

    def simulate_user_alias_delete(self):
        action_user_alias_delete(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=self.bob,
            content={
                'alias_deleted': 'larry'
            }
        )

    def simulate_group_actions(self):
        self.simulate(
            'Работа с командами',
            self.simulate_group_add,
            self.simulate_group_rename,
            self.simulate_group_member_addition,
            self.simulate_group_member_deletion,
            self.simulate_group_delete,
            self.simulate_group_admins_change,
            self.simulate_group_alias_add,
            self.simulate_group_alias_delete,
        )

    def simulate(self, description, *items):
        with trace_node('section', description):
            for func in items:
                with trace_node('case', func.__doc__.strip()):
                    with self.main_connection.begin_nested() as transaction:
                        func()
                        # Перед тем, как откатить транзакцию, надо выполнить
                        # асинхронные таски, иначе в результате могут быть ошибки
                        
                        # TODO: подопнуть выполнение тасков в новой очереди
                        # after_successful_transaction_pipeline.execute()
                        transaction.rollback()

    def prepare_group(self):
        """Создает данные для того, чтобы можно было симулировать действия, связанные с командой
        """
        some_group = self.group_model.create(
            name='Some other group',
            org_id=self.org_id,
            members=[
                {
                    'type': 'user',
                    'id': self.bob['id'],
                }
            ]
        )
        group = self.group_model.create(
            name='Blah',
            org_id=self.org_id,
            admins=[self.admin_uid],
            members=[
                {
                    'type': 'user',
                    'id': self.admin_uid,
                },
                {
                    'type': 'group',
                    'id': some_group['id'],
                },
                {
                    'type': 'department',
                    'id': self.robots_dep_id,
                },
            ],
        )

        # группу надо забрать из базы вместе со всеми мемберами,
        # иначе код не сможет сформировать событие для отправки в вебхук
        group = self.group_model.get(
            self.org_id,
            group['id'],
            fields=[
                '*',
                'members.*',
                'admins.*',
            ],
        )
        return group

    def simulate_group_admins_change(self):
        """Добавление нового админа команды."""
        group = self.prepare_group()
        new_group = deepcopy(group)
        # прикинемся, будто в список админа добавили Боба
        new_group['admins'].append(self.bob)

        action_group_admins_change(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=new_group,
            old_object=group,
        )

    def simulate_group_add(self):
        """Добавление команды"""
        group = self.prepare_group()
        action_group_add(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=group,
        )

    def simulate_group_delete(self):
        """Удаление команды"""
        group = self.prepare_group()
        action_group_delete(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=group,
        )

    def simulate_group_rename(self):
        """Смена названия команды """

        group = self.prepare_group()
        group_before_update = deepcopy(group)

        group_before_update['name'] = {'ru': 'Старое название команды'}
        group['name'] = {'ru': 'Новое название команды'}

        action_group_modify(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=group,
            old_object=group_before_update,
        )

    def simulate_group_member_addition(self):
        """Добавление в команду человека"""
        group = self.prepare_group()
        group_before_update = deepcopy(group)

        # сделаем вид, что раньше Боба не было в мемберах
        # просто убрав его из списка
        members = group_before_update['members']
        members = [item
                   for item in members
                   if item['type'] != 'user']
        group_before_update['members'] = members

        action_group_modify(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=group,
            old_object=group_before_update,
        )

    def simulate_group_member_deletion(self):
        """Удаление человека из команды"""
        group = self.prepare_group()
        group_before_update = deepcopy(group)

        # сделаем вид, что в новой группе нет Боба
        members = group['members']
        members = [item
                   for item in members
                   if item['type'] != 'user']
        group['members'] = members

        action_group_modify(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=group,
            old_object=group_before_update,
        )

    def simulate_group_alias_add(self):
        """Добавление алиаса команды"""
        group = self.prepare_group()
        action_group_alias_add(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=group,
            content={
                'alias_added': 'VIP'
            }
        )

    def simulate_group_alias_delete(self):
        """Удаление алиаса команды"""
        group = self.prepare_group()
        action_group_alias_delete(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=group,
            content={
                'alias_deleted': 'luxury'
            }
        )

    def simulate_service_actions(self):
        self.simulate(
            'Работа с сервисами',
            self.simulate_service_enabled,
            self.simulate_service_disabled,
            self.simulate_service_ready,
            self.simulate_service_license_changed,
        )

    def prepare_service(self):
        """
        Создаем какой-нибудь сервис.
        """
        return self.service_model.create(
            'slug',
            'Service Name',
            client_id='JDjsdfHjsdfJHGDJF',
            tvm_client_id=421234,
            scopes=[scope.work_with_any_organization,],
            robot_required=False,
        )

    def simulate_service_enabled(self):
        """Включение сервиса"""
        service = self.prepare_service()
        action_service_enable(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=service,
        )
        # удаляем, чтоб потом можно было снова сгенерировать такой же сервис
        self.service_model.delete_one(service['id'])

    def simulate_service_disabled(self):
        """Отключение сервиса"""
        service = self.prepare_service()
        action_service_disable(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=service,
        )
        # удаляем, чтоб потом можно было снова сгенерировать такой же сервис
        self.service_model.delete_one(service['id'])

    def simulate_service_ready(self):
        """Сервис готов"""
        service = self.prepare_service()
        action_service_set_ready(
            self.main_connection,
            org_id=self.org_id,
            author_id=None,
            object_value=service,
            old_object={},
        )
        # удаляем, чтоб потом можно было снова сгенерировать такой же сервис
        self.service_model.delete_one(service['id'])

    def simulate_domain_actions(self):
        self.simulate(
            'Работа с доменами',
            self.simulate_domain_add,
            self.simulate_domain_delete,
            self.simulate_domain_master_modify,
        )

    def prepare_domain(self, domain_name=None):
        """Создаём какой-нибудь домен"""
        name = domain_name if domain_name else 'example.org'
        return self.domain_model.create(name, self.org_id)

    def simulate_domain_add(self):
        """Добавление домена"""
        domain = self.prepare_domain()
        action_domain_add(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=domain,
            old_object=None,
        )

    def simulate_domain_master_modify(self):
        """У организации изменился основной домен."""

        old_domain = self.prepare_domain('blah.minor.ru')
        new_domain = self.prepare_domain('newdomain.com')
        action_domain_master_modify(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=new_domain,
            old_object=old_domain,
        )

    def simulate_domain_delete(self):
        """Удаление домена"""
        domain = self.prepare_domain()
        action_domain_delete(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=domain,
            old_object=domain,
        )

    def simulate_service_license_changed(self):
        """Изменение лицензий сервиса"""
        service = self.prepare_service()
        action_service_license_change(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=service,
        )
        # удаляем, чтоб потом можно было снова сгенерировать такой же сервис
        self.service_model.delete_one(service['id'])

    def simulate_service_trial_end(self):
        """Окончание триала сервиса"""
        service = self.prepare_service()
        action_service_trial_end(
            self.main_connection,
            org_id=self.org_id,
            author_id=self.admin_uid,
            object_value=service,
        )
        # удаляем, чтоб потом можно было снова сгенерировать такой же сервис
        self.service_model.delete_one(service['id'])
