# encoding: UTF-8

from functools import wraps
from inspect import getargspec
from inspect import getcallargs
from inspect import ismethod

from flask import g

_CACHE_G_ATTR = '__appcontext_cache__'


def _get_cache():
    try:
        return getattr(g, _CACHE_G_ATTR)
    except AttributeError:
        value = {}
        setattr(g, _CACHE_G_ATTR, value)
        return value


def _make_cache_key(func, cached_args, args, kwargs):
    call_args = getcallargs(func, *args, **kwargs)
    return (func,) + tuple(call_args[arg] for arg in cached_args)


def appcontext_cached(*cached_args):
    """
    Декоратор, который кэширует результат выполнения функции в текущем контексте
    приложения Flask. Т.е. будет возвращать одно и тоже знаение в пределах
    одного HTTP-запроса или внутри контексного менеджера ``app.app_context()``.

    Пример использования:
    ::
        from flask import request
        from werkzeug.local import LocalProxy

        @appcontext_cached
        def get_current_meta_organization():
            try:
                org_id = int(request.headers['X-Org-ID'])
            except KeyError:
                return None
            except ValueError:
                # Здесь можно было бы вернуть 400-й код для невалидного X-Org-ID
                return None

            repository = component_registry().meta_organization_repository
            return repository.get_or_none(org_id)

        component_registry().current_meta_organization = \
            LocalProxy(get_current_meta_organization)
    """

    if not cached_args or isinstance(cached_args[0], str):
        return lambda f: appcontext_cached(f, *cached_args)
    else:
        func = cached_args[0]
        cached_args = cached_args[1:]

        argspec = getargspec(func)
        known_args = set(argspec.args or []) | set(argspec.keywords or [])

        unknown_args = set(cached_args) - known_args
        if unknown_args:
            raise TypeError(
                'function %s has no specified arguments: %s' % (
                    func.__name__,
                    ', '.join(unknown_args),
                )
            )

        @wraps(func)
        def decorator(*args, **kwargs):
            cache = _get_cache()
            cache_key = _make_cache_key(func, cached_args, args, kwargs)

            try:
                return cache[cache_key]
            except KeyError:
                value = func(*args, **kwargs)
                cache[cache_key] = value
                return value

        decorator.__cached_func__ = func
        decorator.__cached_args__ = cached_args

        return decorator


def reset_appcontext_cached(func, *args, **kwargs):
    """
    Сбрасывает значение функции в текущем контексте приложения Flask.
    """

    if ismethod(func):
        args = (func.__self__,) + args

    cache = _get_cache()
    cache_key = _make_cache_key(
        func.__cached_func__,
        func.__cached_args__,
        args,
        kwargs,
    )

    cache.pop(cache_key, None)  # Безопасно удаляем элемент
