# -*- coding: utf-8 -*-
import xmlrpc.client
from xmlrpc.client import ProtocolError
from socket import error
import os
from cachetools import TTLCache, cached
from threading import RLock

import dateutil.parser

from decimal import Decimal
from retrying import retry

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.billing.utils import billing_errors_handler
from intranet.yandex_directory.src.yandex_directory.core.exceptions import BillingClientIdMismatch
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.directory_logging.http_requests_logger import HttpRequestLogService

"""
Клиент для походов в API Биллинга (Баланса)
Описание API https://wiki.yandex-team.ru/balance/xmlrpc/

В любых походах в API биллинга нужно указывать uid оператора.
Это человек, который осуществляет данное действие.
Указывать его нужно обязательно в виде строки.

"Клиент" в биллинге - наша организация.
"Плательщик" по сути - набор реквизитов.
"Представитель клиента" - человек, который может платить и смотреть заказы с актами на balance.yandex.ru


В тестинге Биллинг использует продакшеновые UID-ы, поэтому для ENVIRONMENT=testing
мы используем отдельный BillingClientWithUidReplacement
"""

# id сервиса в Биллинге, меняться не должен
WORKSPACE_SERVICE_ID = 202

# 1="ООО Яндекс",
# 2="ООО Яндекс.Украина",
# 3="КазНет Медиа",
# 4="Yandex Inc",
# 7="Yandex Europe AG",
# 8="Yandex Turkey"
#
# todo: пока непонятно как будет выбираться валюта для оплаты
#       возможно, нужно будет выбирать валюту в зависимости от фирмы
FIRM_IDS = {
    'yandex': 1,
    'yandex_europe_ag': 7,
}

TRUST_PAYMENT_METHOD = 'trust_web_page'

# у нас должны будут отличаться id банков для резидентов и не резидентов
# который мы должны передавать при создании договора или оферты
# пока изестен только id банка для резидентов
BANK_IDS = {
    'resident': 21,
}

if os.environ.get('ENVIRONMENT') == 'autotests':
    products_price_cache = None
else:
    products_price_cache = TTLCache(
        maxsize=1000,
        ttl=300,
    )
products_price_lock = RLock()


def _retry_on_protocol_error(exc, error_code):
    return isinstance(exc, ProtocolError) and error_code == getattr(exc, 'errcode')


def _retry_on_socket_error(err):
    return isinstance(err, error)


def retry_on_connection_error(exc):
    return any([
        _retry_on_protocol_error(exc, 502),
        _retry_on_protocol_error(exc, 504),
        _retry_on_socket_error(exc),
    ])


class TVM2Transport(xmlrpc.client.SafeTransport):
    def send_content(self, connection, request_body):
        connection.putheader('X-Ya-Service-Ticket', tvm.tickets['billing'])
        return super().send_content(connection, request_body)

    def make_connection(self, host):
        conn = super(TVM2Transport, self).make_connection(host)
        conn.timeout = app.config['BILLING_TIMEOUT']
        return conn


class BillingClient(object):
    def __init__(self, endpoint, token, manager_uid):
        self.endpoint = endpoint
        self.server = xmlrpc.client.ServerProxy(endpoint, allow_none=True, transport=TVM2Transport())
        self.token = token
        self.manager_uid = manager_uid  # uid менеджера, обслуживающего организацию

    def _prepare_uid(self, uid):
        """
        Превращает uid в строку, Биллинг может ломаться на больших UID-ах в числовом виде
        """
        if uid is not None:
            return str(uid)

    def get_client_contracts(self, client_id):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getclientcontracts
        Находит договоры (контракты) данного клиента в Биллинге

        Args:
            client_id - id клиента
        """
        params = {
            'ClientID': client_id,
            'Signed': 0,                # чтобы неподписанные договоры тоже возвращались
        }
        with log.fields(client_id=client_id):
            log.info('Getting client contracts')
            all_contracts = self._make_request('GetClientContracts', params=params)
            service_contracts = []
            for contract in all_contracts:
                if WORKSPACE_SERVICE_ID in contract['SERVICES']:
                    service_contracts.append(contract)
            return service_contracts

    def create_natural_person(self, uid, client_id, first_name, last_name, middle_name, phone, email):
        """
        Создаёт плательщика в Биллинге (физ лицо)
        https://wiki.yandex-team.ru/balance/xmlrpc/#objazatelnyepoljaxjeshavzavisimostiottype
        """
        params = {
            'type': 'ph',
            'client_id': client_id,
            'fname': first_name,
            'lname': last_name,
            'mname': middle_name,
            'phone': phone,
            'email': email,
        }
        with log.fields(
            uid=uid,
            client_id=client_id,
            first_name=first_name,
            last_name=last_name,
            middle_name=middle_name,
            phone=phone,
            email=email,
        ):
            log.info('Creating natural person in Billing')
            return self._make_request('CreatePerson', params=params, uid=uid)

    def create_legal_person(self,
                            uid,
                            client_id,
                            name,
                            long_name,
                            phone,
                            email,
                            postal_code,
                            postal_address,
                            legal_address,
                            inn,
                            kpp,
                            bik,
                            account):
        """
        Создаёт плательщика в Биллинге (юридическое лицо)
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createperson

        Args:
            uid (int) - uid оператора
            client_id (int) - id клиента
            name (str) - название организации
            long_name (str) - полное название организации с формой собственности
            phone (str) - телефон
            email (str) - контантый ящик
            postal_code (str)- почтовый индекс
            postal_address (str)- адрес для доставки корреспонденции/фактический адрес
            legal_address (str)- юридический адрес
            inn (str)- ИНН
            kpp (str)- КПП - необязательно для ИП(ИНН 12-ти значный)
            bik (str)- БИК банка
            account (str)- расчетный счет в банке
        """
        params = {
            'type': 'ur',
            'client_id': client_id,
            'name': name,
            'longname': long_name,
            'phone': phone,
            'email': email,
            'postcode': postal_code,
            'postaddress': postal_address,
            'legaladdress': legal_address,
            'inn': inn,
            'kpp': kpp or '',  # если КПП не указан, например для ИП, нужно передавать пустую строку в Биллинг
            'bik': bik,
            'account': account,
        }
        with log.fields(
            uid=uid,
            client_id=client_id,
            person_name=name,
            long_name=long_name,
            phone=phone,
            email=email,
            postal_code=postal_code,
            postal_address=postal_address,
            legal_address=legal_address,
            inn=inn,
            kpp=kpp,
            bik=bik,
            account=account,
        ):
            log.info('Creating legal person in Billing')
            return self._make_request('CreatePerson', params=params, uid=uid)

    def find_client(self, client_id, user_uid):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.findclient

        """
        params = {
            'ClientID': client_id,
            'PassportID': user_uid,
        }
        with log.fields(client_id=client_id, user_uid=user_uid):
            log.info('Finding client in Billing')
            return self._make_request('FindClient', params)

    def create_natural_person_client(self, uid, name, email, phone):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createclient

        Args:
            uid - паспортный uid оператора
            name - название клиента
            email - email клиента
            phone - телефон
        """
        # todo: обработка ошибок
        with log.fields(uid=uid, client_name=name, email=email, phone=phone):
            log.info('Creating natural person client in Billing')
            _, _, client_id = self.create_client(uid=uid, name=name, email=email, phone=phone)
        return client_id

    def create_legal_person_client(self, uid, name, email, phone):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createclient

        Args:
            uid - паспортный uid оператора
            name - название клиента
            email - email клиента
            phone - телефон
        """
        # todo: обработка ошибок
        with log.fields(uid=uid, client_name=name, email=email, phone=phone):
            log.info('Creating legal person client in Billing')
            _, _, client_id = self.create_client(uid=uid, name=name, email=email, phone=phone)
        return client_id

    def create_client(self, uid, name, email, phone):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createclient

        Args:
            uid - паспортный uid оператора
            name - название клиента
            email - email клиента
            phone - телефон
        """
        params = {
            'name': name,
            'email': email,
            'phone': phone,
            'currency': 'RUR',  # todo: поменять когда появятся не резиденты
        }
        return self._make_request('CreateClient', params, uid=uid)

    def create_offer(self,
                     uid,
                     client_id,
                     person_id):
        """
        Создаём оферту в Биллинге для клиента с id == client_id
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createoffer

        Args:
            client_id - id клиента
            person_id - id плательщика
            manager_uid - uid менеджера, обслуживающего организацию
        """
        params = {
            'client_id': client_id,
            'person_id': person_id,
            'firm_id': FIRM_IDS['yandex'],
            'bank_details_id': BANK_IDS['resident'],
            'manager_uid': self.manager_uid,
            'currency': 'RUR',  # todo: поменять когда появятся не резиденты
            'payment_type': 3,  # постоплата
            'payment_term': app.config['BILLING_PAYMENT_TERM'],  # срок оплаты счетов в днях
            'services': [WORKSPACE_SERVICE_ID],
            'start_dt': None,
        }
        with log.fields(**params):
            log.info('Creating offer in Billing')
            return self._make_request('CreateOffer', params, uid=uid)

    def create_contract(self,
                        uid,
                        client_id,
                        person_id):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createcommoncontract

        Args:
            uid - uid оператора
            client_id - id клиента
            person_id - id плательщика
            manager_uid - uid менеджера, обслуживающего организацию
        """
        params = {
            'client_id': client_id,
            'person_id': person_id,
            'firm_id': FIRM_IDS['yandex'],
            'bank_details_id': BANK_IDS['resident'],
            'manager_uid': self.manager_uid,
            'payment_type': 3,  # постоплата
            'currency': 'RUR',  # todo: поменять когда появятся не резиденты; ISO код валюты (одно из EUR, KZT, BYR, USD, RUB, UAH)
            'payment_term': app.config['BILLING_PAYMENT_TERM'],  # срок оплаты счетов в днях
            'services': [WORKSPACE_SERVICE_ID],
            'start_dt': None,
        }
        with log.fields(**params):
            log.info('Creating contract in Billing')
            return self._make_request('CreateCommonContract', params, uid=uid)

    def get_balance_info(self, client_id):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getpartnerbalance

        Возвращает информацию о балансе и дате начала задолженности для клиента:
        {
            'balance': -100,
            'first_debt_act_date': '2017-01-01'
        }
        Args:
            uid (int) - uid оператора
            client_id (int) - id клиента
        """
        active_id = self.get_active_workspace_contract_id(client_id)
        contract_ids = [active_id] if active_id else [-1]
        with log.fields(contract_id=active_id):
            log.info('Getting balance for contract')
            return self._get_balances_info(contract_ids=contract_ids)

    def _get_balances_info(self, contract_ids):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getpartnerbalance

        Для каждого contract_id из списка получаем баланс и
        дату начала задолженности FirstDebtFromDT (либо None, если ее нет).
        >> Для постоплатных сервисов положительная разница между суммой поступлений (ReceiptSum)
        >> и суммой актов (ActSum) означает, что акт полностью оплачен

        Акты выставляются за потребленные услуги, т.о, баланс = "поступишие деньги - выставленные счета"
        Возвращает словарь вида:
        {
            'balance': Decimal('100'),
            'first_debt_act_date': None,
            'receipt_sum': Decimal('500'),
            'act_sum': Decimal('400'),
        }

        Args:
            contract_ids (list) - список договоров (у нас это [<id активного договора>])
        """
        balances_info = self._make_flat_request('GetPartnerBalance', WORKSPACE_SERVICE_ID, contract_ids)
        act_sum = 0
        receipt_sum = 0
        first_debt_act_date = None

        if balances_info:
            b = balances_info[0]
            # преобразуем из строк в Decimal чтобы ничего не потерять
            act_sum = Decimal(b['ActSum'])
            receipt_sum = Decimal(b['ReceiptSum'])
            if b.get('FirstDebtFromDT'):
                first_debt_act_date = dateutil.parser.parse(b['FirstDebtFromDT']).date()

        return {
            'balance': receipt_sum - act_sum,
            'first_debt_act_date': first_debt_act_date,
            'act_sum': act_sum,
            'receipt_sum': receipt_sum,
        }

    def create_client_user_association(self,
                                       uid,
                                       client_id,
                                       user_uid):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createuserclientassociation

        Args:
            uid - uid оператора
            client_id - id клиента
            user_uid - uid представителя
        """
        with log.fields(uid=uid, client_id=client_id, user_uid=user_uid):
            log.info('Creating client representative')
            return self._make_flat_request(
                'CreateUserClientAssociation',
                self._prepare_uid(uid),
                client_id,
                self._prepare_uid(user_uid),
            )

    def get_request_choices(self, uid, request_id):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getrequestchoices

        Args:
            uid - uid оператора
            request_id - id недовыставленного счета
        """
        params = {
            "OperatorUid": self._prepare_uid(uid),
            "RequestID": request_id,
        }
        return self._make_flat_request('GetRequestChoices', params)

    def create_pay_request(self,
                           uid,
                           request_id,
                           person_id,
                           contract_id,
                           payment_method_id=None,
                           ):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/paymentviatrustprocess/?section=5&goanchor=h-5&originpath=/balance/xmlrpc/paymentviatrustprocess#balance.payrequest

        Args:
            uid - uid оператора
            request_id - id запроса из create_invoice_request
            person_id - id плательщика
            contract_id - id договора
        """
        params = {
                'RequestID': request_id,
                'PaymentMethodID': payment_method_id or TRUST_PAYMENT_METHOD,
                'Currency': 'RUB',  # пока только в рублях принимаем, поменять когда появятся нерезиденты
                'PersonID': person_id,
                'ContractID': contract_id,
            }
        with log.fields(uid=uid, request_id=request_id, contract_id=contract_id, person_id=person_id):
            log.info('Creating or updating request')
            return self._make_flat_request('PayRequest', self._prepare_uid(uid), params)

    def get_request_payment_methods(self, uid, request_id):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/paymentviatrustprocess/?section=5&goanchor=h-5&originpath=/balance/xmlrpc/paymentviatrustprocess#balance.getrequestpaymentmethods

        Args:
            uid - uid оператора
            request_id - id запроса из create_invoice_request
        """
        params = {
            'RequestID': request_id,
            'OperatorUid': self._prepare_uid(uid),
        }
        with log.fields(uid=uid, request_id=request_id):
            log.info('Getting request payment methods')
            return self._make_flat_request('GetRequestPaymentMethods', params)

    def create_or_update_order(self,
                               uid,
                               client_id,
                               product_id,
                               ):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createorupdateordersbatch

        Args:
            uid - uid оператора
            client_id - id клиента
            product_id - id продукта
        """
        orders = [
            {
                # Если логика ServiceOrderID=client_id будет меняться, нужно учесть, что
                # ServiceOrderId не должен быть больше 20000000000000, биллинг значения больше использует у себя внутри
                # client_id больше этого числа не бывает
                'ServiceOrderID': client_id,
                'ServiceID': WORKSPACE_SERVICE_ID,
                'ClientID': client_id,
                'ProductID': product_id,
            }
        ]
        with log.fields(uid=uid, client_id=client_id, product_id=product_id):
            log.info('Creating or updating order')
            return self._make_flat_request('CreateOrUpdateOrdersBatch', self._prepare_uid(uid), orders)

    def create_invoice_request(self,
                               uid,
                               client_id,
                               quantity,
                               return_path=None):
        """
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createrequest2

        Args:
            uid - uid оператора
            client_id - id клиента
            quantity (str) - кол-во единиц товара
        """
        orders = [
            {
                # Если логика ServiceOrderID=client_id будет меняться, нужно учесть, что
                # ServiceOrderId не должен быть больше 20000000000000, биллинг значения больше использует у себя внутри
                # client_id больше этого числа не бывает
                'ServiceOrderID': client_id,
                'ServiceID': WORKSPACE_SERVICE_ID,
                'ClientID': client_id,
                'Qty': quantity,
            }
        ]
        params = {
            'InvoiceDesireType': 'charge_note',  # charge_note означает, что квитанция создается на пополнение счета
            'ReturnPath': return_path,
        }
        with log.fields(uid=uid, client_id=client_id, quantity=quantity, return_path=return_path):
            log.info('Creating invoice request')
            return self._make_flat_request('CreateRequest2', self._prepare_uid(uid), client_id, orders, params)

    def get_passport_by_uid(self,
                            uid,
                            user_uid,
                            ):
        """
        Возвращает паспортную информацию о пользователе и id клиента, к которому он привязан если есть
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getpassportbyuid

        Args:
            uid - uid оператора
            user_uid - uid представителя
        """
        params = {'RepresentedClientIds': True}
        return self._make_flat_request('GetPassportByUid', self._prepare_uid(uid), self._prepare_uid(user_uid), params)

    def create_client_user_association_if_needed(self,
                                                 uid,
                                                 client_id,
                                                 user_uid,
                                                 ):
        """
        Проверяет, является ли пользователь user_uid представителем клиента client_id,
         если нет - привязывает пользователя к client_id, если uid привязан к другому клиенту,
         то вернем ошибку, это не стандартная ситуация
        Args:
            uid - uid оператора
            client_id - id клиента
            user_uid - uid представителя
        """
        old_client_id = self.get_user_client(uid)
        if old_client_id != client_id:
            if old_client_id:
                raise BillingClientIdMismatch
            self.create_client_user_association(uid, client_id, user_uid)

    def get_client_acts(self, client_id):
        """
        Возвращает акты клиента
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getclientacts

        Args:
            client_id - id клиента
        Возвращает список словарей вида:
            [{'AMOUNT': '1.14',
              'DT': <DateTime '20170718T19:19:10' at 7f21e409bb90>,
              'ID': 65196072,
              'PAID_AMOUNT': '0',
              'PAYMENT_TERM_DT': <DateTime '20170930T00:00:00' at 7f21e409b7a0>,
              'ROWS': [{'AMOUNT': '1.14', 'PRODUCT_ID': 508507, 'QTY': '0.06'}]}]
        """
        return self._make_flat_request('GetClientActs', self.token, {'ClientID': client_id})

    def create_invoice(self, uid, request_id, person_id, paysys_id, contract_id):
        """
        Создает обычный предоплатный счет
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createinvoice
        Args:
            uid - uid пользователя
            request_id - id недовыставленного счета
            person_id - идентификатор плательщика
            paysys_id - id платежной системы
            contract_id - id договора
        """

        params = {
            "RequestID": request_id,
            "PaysysID": paysys_id,
            "PersonID": person_id,
            "ContractID": contract_id,
        }
        return self._make_flat_request('CreateInvoice', self._prepare_uid(uid), params)

    @cached(products_price_cache,
            lock=products_price_lock)
    def get_products_price(self):
        """
        Возвращает цены продуктов.
        https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getproducts

        Returns:
            [{'Name': u'Услуги "Яндекс.Коннект - Трекер 11–100',
              'Prices': [{
                'Currency': 'RUB',
                'DT': <DateTime '20170809T00:00:00' at 7f19864cb290>,
                'Price': '209'
              }],
              'ProductID': 508510,
              'Rate': '1',
              'Unit': u'чел/мес'}]
        """
        all_products = self._make_flat_request('GetProducts', self.token)
        return dict((product['ProductID'], int(product['Prices'][0]['Price'])) for product in all_products)

    def get_contract_print_form(self, client_id):
        """
        Возвращает PDF печатной формы договора
        https://wiki.yandex-team.ru/Balance/XmlRpc/#balance.getcontractprintform
        Returns:
            строка base64
        """
        contract_ids = self.get_client_contracts(client_id)
        for contract in contract_ids:
            if contract['IS_ACTIVE']:
                active_id = contract['ID']
                break
        else:
            active_id = contract_ids[0]['ID']
        with log.fields(contract_id=active_id, client_id=client_id):
            log.info('Getting print form for contract')
            return self._make_flat_request('GetContractPrintForm', active_id).data

    def get_client_persons(self, client_id):
        """
        Возвращает список плательщиков клиента
        """
        with log.fields(client_id=client_id):
            log.info('Getting client persons')
            return self._make_flat_request('GetClientPersons', client_id)

    def get_user_client(self, uid):
        passport_data = self.get_passport_by_uid(uid, uid)
        client_id = passport_data.get('ClientId')
        if not client_id:
            represented_clients = passport_data.get('RepresentedClientIds')
            if represented_clients:
                client_id = represented_clients[0]

        return client_id

    def get_active_workspace_contract_id(self, client_id):
        contracts = self.get_client_contracts(client_id)
        active_id = None
        for contract in contracts:
            if contract['IS_ACTIVE']:
                active_id = contract['ID']
                break
        return active_id

    def get_and_valid_user_client(self, uid):
        client_id = self.get_user_client(uid)
        if client_id:
            active_id = self.get_active_workspace_contract_id(client_id)
            if active_id:
                raise BillingClientIdMismatch
        return client_id

    def get_free_user_client(self, uid):
        # грязный хак, пока со стороны биллинга нельзя создать
        # на один client_id несколько оферт с одним и тем же сервисом
        try:
            return self.get_and_valid_user_client(uid)
        except BillingClientIdMismatch:
            # если на клиенте уже есть договор с коннектом, притворимся,
            # что клиента нет, чтобы создать новый
            return

    def get_user_persons(self, uid):
        """
        Ищем привязанные к uid client_id и плательщиков в биллинге
        """
        client_id = self.get_free_user_client(uid)
        persons = []
        if client_id:
            persons_info = self.get_client_persons(client_id)
            for person in persons_info:
                if person.get('HIDDEN') != '1':
                    if person.get('TYPE') == 'ur':
                        persons.append({
                            'client_id': int(person['CLIENT_ID']),
                            'person_id': int(person['ID']),
                            'email': person.get('EMAIL'),
                            'legal_address': person.get('LEGALADDRESS'),
                            'phone': person.get('PHONE'),
                            'postal_code': person.get('POSTCODE'),
                            'postal_address': person.get('POSTADDRESS'),
                            'long_name': person.get('LONGNAME'),
                            'inn': person.get('INN'),
                            'kpp': person.get('KPP'),
                            'bik': person.get('BIK'),
                            'account': person.get('ACCOUNT'),
                            'type': 'legal',
                        })
                    elif person.get('TYPE') == 'ph':
                        persons.append({
                            'client_id': int(person['CLIENT_ID']),
                            'person_id': int(person['ID']),
                            'email': person.get('EMAIL'),
                            'phone': person.get('PHONE'),
                            'first_name': person.get('FNAME'),
                            'last_name': person.get('LNAME'),
                            'middle_name': person.get('MNAME'),
                            'type': 'natural',
                        })
        return persons

    def _make_request(self, method_name, params, uid=None):
        """
        Вызывает метод Биллинга с именем method_name и аргументами [uid, массив параметров]
        или [массив параметров], если uid не передан
        """
        if uid:
            # uid в Биллинг должен передаваться в виде строки
            args = (self._prepare_uid(uid), params)
        else:
            args = (params, )
        return self._make_flat_request(method_name, *args)

    @billing_errors_handler
    @retry(stop_max_attempt_number=3,
           wait_incrementing_increment=50,
           retry_on_exception=retry_on_connection_error)
    def _make_flat_request(self, method_name, *args):
        """
        Вызывает метод Биллинга с именем method_name и аргументами args
        """
        with log.name_and_fields('billing', method_name=method_name, params=','.join(map(str, args))):
            log.info('Making request to billing API')
            log_record = HttpRequestLogService.start(method_name, self.endpoint, {})
            try:
                response = getattr(self.server.Balance, method_name)(*args)
                log_record.finish('Unknown')
            except Exception as e:
                log_record.error(e)
                raise

            with log.fields(response=response):
                log.info('Billing API response')
            return response
