# -*- coding: utf-8 -*-
from xml.etree.ElementTree import fromstring, ParseError

from functools import wraps
from xmlrpc.client import Fault
from intranet.yandex_directory.src.yandex_directory.core.exceptions import (
    BillingMissingField,
    BillingInvalidField,
    BillingUnknownError,
    BillingUidNotFoundInPassport,
    BillingClientIdNotFound,
)

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common.utils import get_environment


def billing_errors_handler(func):
    @wraps(func)
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Fault as f:
            with log.name_and_fields('billing', fault=f.faultString):
                log.trace().error('Billing returned error')

                return make_exception_from_billing_fault(f.faultString)

    return wrapper


def make_exception_from_billing_fault(fault_resp):
    try:
        xml_fault = fromstring(fault_resp)
        code = [c.text for c in xml_fault.findall('code') if not c.text.isdigit()][0]
    except ParseError:
        code = None 

    codes_invalid_fields = {
        'INVALID_INN': 'inn',
        'WRONG_EMAIL': 'email',
        'WRONG_BIK': 'bik',
        'WRONG_ACCOUNT': 'account',
    }

    billing_fields = {
        'fname': 'first_name',
        'lname': 'last_name',
        'longname': 'long_name',
        'mname': 'middle_name',
        'postcode': 'postal_code',
        'postaddress': 'postal_address',
        'legaladdress': 'legal_address',
    }

    if code in list(codes_invalid_fields.keys()):
        raise BillingInvalidField(codes_invalid_fields[code])

    if code == 'MISSING_MANDATORY_PERSON_FIELD':
        billing_field = xml_fault.find('field').text
        raise BillingMissingField(billing_fields.get(billing_field, billing_field))

    if code == 'PASSPORT_NOT_FOUND':
        uid = xml_fault.find('object-id').text
        raise BillingUidNotFoundInPassport(uid)

    if code == 'CLIENT_NOT_FOUND' and get_environment().startswith('testing'):
        client_id = xml_fault.find('object-id').text
        raise BillingClientIdNotFound(client_id)

    raise BillingUnknownError()
