# encoding: UTF-8

import pickle
import weakref

from redis.sentinel import Sentinel

from intranet.yandex_directory.src import settings


class NullCache:
    def exists(self, key, from_master=False):
        return False

    def get(self, key, from_master=False):
        return None

    def set(self, key, value, ttl=None):
        pass


class RedisCache:
    def __init__(self, sentinel, app):
        self.sentinel = sentinel
        self.app = weakref.proxy(app)
        self.master = self.sentinel.master_for(settings.REDIS_DB_NAME, password=settings.REDIS_PASSWORD)
        self.replica = self.sentinel.slave_for(settings.REDIS_DB_NAME, password=settings.REDIS_PASSWORD)

    def exists(self, key, from_master=False):
        if from_master:
            data = self.master.exists(key)
        else:
            data = self.replica.exists(key)
        return data

    def get(self, key, from_master=False):
        with self.app.stats_aggregator.log_work_time('cache_get'):
            if from_master:
                data = self.master.get(key)
            else:
                data = self.replica.get(key)
            if data is None:
                self.app.stats_aggregator.inc('cache_miss_summ')
            else:
                self.app.stats_aggregator.inc('cache_hit_summ')
        if data is not None:
            data = pickle.loads(data)
        return data

    def set(self, key, value, ttl=None):
        with self.app.stats_aggregator.log_work_time('cache_set'):
            value = pickle.dumps(value)
        self.master.set(key, value, ex=ttl)


def setup_caching(app):
    if settings.CACHE_ENABLED:
        sentinel = Sentinel([(host, 26379) for host in settings.REDIS_HOSTS], socket_timeout=0.1)
        app.cache = RedisCache(sentinel, app)
    else:
        app.cache = NullCache()
