# -*- coding: utf-8 -*-
import json
import sys
import abc

from collections import defaultdict

from intranet.yandex_directory.src.yandex_directory.common.commands.base import BaseCommand, Option
from intranet.yandex_directory.src.yandex_directory.common.exceptions import APIError


def get_exceptions(m):
    module_name = m.__name__
    for obj in list(vars(m).values()):
        if (isinstance(obj, type) and
            issubclass(obj, APIError) and
            obj is not APIError and
            obj.__module__ == module_name):
            yield obj


def get_classes():
    modules = [
        (name, list(get_exceptions(m)))
        for name, m in list(sys.modules.items())
        if m is not None and name.startswith('intranet.yandex_directory.src.yandex_directory')
    ]
    modules = [item for item in modules if item[1]]

    # Теперь построим словарь по которому можно будет определить,
    # есть ли у класса потомки
    subclasses = defaultdict(list)
    for name, classes in modules:
        for cls in classes:
            parent = cls.mro()[0]
            subclasses[parent].append(cls)
    return modules, subclasses


def show(classes, subclasses, only_missing=False):
    from colorama import Fore, Style

    seen_codes = defaultdict(list)
    has_duplicates = defaultdict(set)
    no_description = []

    for module, exceptions in classes:
        print(module)
        for exc in exceptions:
            description = getattr(exc, 'description', None)
            has_subclasses = subclasses[exc]
            has_desc = isinstance(description, str)
            class_name = exc.__name__
            full_name = '{}.{}'.format(module, class_name)

            # Если у класса есть потомки, то отсутствие у него описания не должно считаться проблемой
            if not has_subclasses and not has_desc:
                no_description.append(full_name)

            if only_missing == False or not has_desc:
                duplicate_in = seen_codes[exc.code]

                print(' ', class_name)
                print('   ', exc.code, end=' ')

                if duplicate_in:
                    has_duplicates[exc.code].add(full_name)
                    has_duplicates[exc.code].update(duplicate_in)
                    print(Fore.RED + 'Такой же код в {}'.format(', '.join(has_duplicates)) + Style.RESET_ALL)
                else:
                    print('')

                if has_desc:
                    print('   ', Fore.GREEN + exc.description.encode('utf-8') + Style.RESET_ALL)
                else:
                    print('   ', Fore.RED + 'Нет описания' + Style.RESET_ALL)

                seen_codes[exc.code].append(full_name)

    if has_duplicates:
        print('')
        print(Fore.RED + 'В этих классах код сообщения об ошибке повторяется:')
        print('')
        for key, classes in sorted(has_duplicates.items()):
            print(key)
            for cls in classes:
                print('  ', cls)
        print(Style.RESET_ALL)

    return has_duplicates or no_description


def save(classes, filename):
    if not filename.endswith('.tjson'):
        print('Мы сохраняем данные в формате tjson, и файл должен иметь расширение ".tjson"')
        sys.exit(1)

    with open(filename, 'w') as f:
        keys = {}
        data = dict(keysets=dict(api_messages=dict(keys=keys)))

        for module, module_classes in classes:
            for cls in module_classes:
                full_name = module + '.' + cls.__name__
                desc = getattr(cls, 'description', None)
                if desc and not isinstance(desc, abc.abstractproperty):
                    keys[cls.code] = dict(
                        info=dict(context=full_name),
                        translations=dict(ru=dict(form=desc)),
                    )
        json.dump(data, f)


class Command(BaseCommand):
    """
    Показывает ошибки унаследованные от APIErrors и их описания.
    """
    name = 'api-errors'
    option_list = (
        Option('--missing', '-m',
               action='store_true', required=False,
               help='Показывать только классы для которых нет описания.'),
        Option('--save-to', '-s',
               required=False,
               help='Сохранить описания в файл в формате tjson, годный для загрузки в Танкер.'),
    )

    def run(self, missing, save_to):
        from colorama import Fore, Style

        classes, subclasses = get_classes()
        has_problems = show(classes, subclasses, missing)

        if save_to:
            if has_problems:
                print(Fore.RED + 'С некоторыми классами ошибок есть проблем. Сначала их надо исправить.' + Style.RESET_ALL)
                sys.exit(1)

            save(classes, save_to)
