# -*- coding: utf-8 -*-
import json
import requests

from intranet.yandex_directory.src.yandex_directory.common.commands.base import AllShardsCommand, Option
from intranet.yandex_directory.src.yandex_directory.core.models.user import UserModel
from intranet.yandex_directory.src.yandex_directory.core.models.department import DepartmentModel
from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel


def get_uid_cookie(uids):
    # uids - список всех uid-ов
    # На выходе dict: {
    #   <uid>: {
    #       'Session_id': <session_id>,
    #       'sessionid2': <sessionid2>,
    #   }
    # }
    # генерируем стрессовую куку по этому описанию:
    # https://wiki.yandex-team.ru/nagruzochnoetestirovanie/stresssessionid/

    url = 'https://adm.yandex-team.ru/api/makesession'
    str_uids = ','.join([str(uid) for uid in uids])
    data = {
        'uids': str_uids,
        'policy': 'longer',
        'get_safe': 1,
    }
    response = requests.post(url, data=data)
    result = json.loads(response.content)
    if result.get('status') != 'OK':
        return {}

    result_dict = {}
    for uid in uids:
        uid = str(uid)
        result_dict[uid] = {
            'Session_id': result['sessions'][uid],
            'sessionid2': result['sslsessions'][uid],
        }
    return result_dict


def get_currect_department(cookie, org_id, dep_id):
    return {
        "method": "GET",
        "uri": "/api/department?id=%d&extension=shallow" % dep_id,
        "headers": {
            "Cookie": "Session_id=%s; sessionid2=%s; yandex_ws_visited=yes" % (
                cookie['Session_id'],
                cookie['sessionid2']
            ),
            "x-org-id": org_id,
        }
    }


def get_currect_user(cookie, org_id, user_id):
    return {
        "method": "GET",
        "uri": "/api/user?id=%d&extension=shallow" % user_id,
        "headers": {
            "Cookie": "Session_id=%s; sessionid2=%s; yandex_ws_visited=yes" % (
                cookie['Session_id'],
                cookie['sessionid2']
            ),
            "x-org-id": org_id,
        }
    }


def get_current_group(cookie, org_id, group_id):
    return {
        "method": "GET",
        "uri": "/api/group?id=%d&extension=deep&type=generic" % group_id,
        "headers": {
            "Cookie": "Session_id=%s; sessionid2=%s; yandex_ws_visited=yes" % (
                cookie['Session_id'],
                cookie['sessionid2']
            ),
            "x-org-id": org_id,
        }
    }


class Command(AllShardsCommand):
    """
    Запросы в виде json сначала отправляем в файл, а потом уже прогоняем
    через GenerateAmmoCommand.
    """
    name = 'generate-requests-json-for-ammo'

    option_list = (
        Option('org_id'),
        Option('--numbers', '-n', dest='numbers', required=False, default=1),
        Option('--apart', '-a', dest='apart', required=False, default=True),
    )

    def get_dashboard(self, cookie):
        return [{
            "method": "GET",
            "uri": "/",
            "headers": {
                "Cookie": "Session_id=%s; sessionid2=%s; yandex_ws_visited=yes" % (
                    cookie['Session_id'],
                    cookie['sessionid2']
                )
            }
        }]

    def get_departments(self, cookie):
        return [{
            "method": "GET",
            "uri": "/api/department?extension=deep",
            "headers": {
                "Cookie": "Session_id=%s; sessionid2=%s; yandex_ws_visited=yes" % (
                    cookie['Session_id'],
                    cookie['sessionid2']
                ),
                "x-org-id": self.org_id,
            }
        }]

    def get_users(self, cookie):
        return [{
            "method": "GET",
            "uri": "/api/users?extension=deep",
            "headers": {
                "Cookie": "Session_id=%s; sessionid2=%s; yandex_ws_visited=yes" % (
                    cookie['Session_id'],
                    cookie['sessionid2']
                ),
                "x-org-id": self.org_id,
            }
        }]

    def get_groups(self, cookie):
        return [{
            "method": "GET",
            "uri": "/api/group?extension=deep&type=generic",
            "headers": {
                "Cookie": "Session_id=%s; sessionid2=%s; yandex_ws_visited=yes" % (
                    cookie['Session_id'],
                    cookie['sessionid2']
                ),
                "x-org-id": self.org_id,
            }
        }]

    def get_departments_details(self, cookie):
        reqs = []
        for d in self.all_deps:
            reqs.append(
                get_currect_department(cookie, self.org_id, d['id'])
            )
        return reqs

    def get_users_details(self, cookie):
        reqs = []
        for d in self.all_users:
            reqs.append(
                get_currect_user(cookie, self.org_id, d['id'])
            )
        return reqs

    def get_groups_details(self, cookie):
        reqs = []
        for d in self.all_groups:
            reqs.append(
                get_current_group(cookie, self.org_id, d['id'])
            )
        return reqs

    def gen_requests(self, cookie):
        yield self.get_dashboard(cookie)[0]
        yield self.get_users(cookie)[0]
        yield self.get_departments(cookie)[0]
        yield self.get_groups(cookie)[0]

        for u in self.all_users:
            yield get_currect_user(cookie, self.org_id, u['id'])
        for d in self.all_deps:
            yield get_currect_department(cookie, self.org_id, d['id'])
        for g in self.all_groups:
            yield get_currect_department(cookie, self.org_id, g['id'])

    def requests_to_file(self):
        """
        Совместно, все запросы отправляем в один файл
        :return:
        """
        with open('portal_requests.json', 'w') as f:
            all_requests = []
            for uid in self.uid_cookies:
                cookie = self.uid_cookies[uid]
                for r in self.gen_requests(cookie):
                    all_requests.append(r)
            f.write(json.dumps(all_requests))

    def apart_requests_to_files(self):
        """
        Разделяем запросы по файлам, в соответсвии с GET-методами
        :return:
        """
        filenames_functions = {
            'dashboard.json': self.get_dashboard,
            'departments.json': self.get_departments,
            'users.json': self.get_users,
            'groups.json': self.get_groups,
            'departments_details.json': self.get_departments_details,
            'users_details.json': self.get_users_details,
            'groups_details.json': self.get_groups_details,
        }

        for fn in filenames_functions:
            with open(fn, 'w') as f:
                func = filenames_functions[fn]
                for uid in self.uid_cookies:
                    cookie = self.uid_cookies[uid]
                    # генерим пачку запросов от имени пользователя с таким uid
                    results = func(cookie)
                    # пишем их в файл построчно
                    serialized_results = (
                        json.dumps(result) + '\n'
                        for result in results
                    )
                    f.writelines(serialized_results)

    def run(self, org_id, numbers=1, apart=True):
        # numbers - кол-во стрессовых кук, от имени которых будем
        # дергать методы
        # apart - разделить запросы по файлам
        numbers = int(numbers)
        self.org_id = int(org_id)

        org_users = UserModel(self.main_connection).find(
            filter_data={
                'org_id': self.org_id,
            },
            limit=numbers
        )
        # Одним запросом в adm.yandex-team.ru получаем все куки
        self.uid_cookies = get_uid_cookie([u['id'] for u in org_users])

        self.all_users = UserModel(self.main_connection).find(
            filter_data={
                'org_id': self.org_id,
            }
        )
        self.all_deps = DepartmentModel(self.main_connection).find(
            filter_data={
                'org_id': self.org_id,
            }
        )
        self.all_groups = GroupModel(self.main_connection).find(
            filter_data={
                'org_id': self.org_id,
            }
        )
        if apart:
            self.apart_requests_to_files()
        else:
            self.requests_to_file()
