# -*- coding: utf-8 -*-
import os
import subprocess

import psycopg2
import psycopg2.extras

try:
    from library.python import resource
    import yatest.common
    IS_ARCADIA = True
except ImportError:
    IS_ARCADIA = False

from intranet.yandex_directory.src.yandex_directory.common.commands.base import BaseCommand, Option
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common import db


CREATE_MIGRATION_TABLE_COMMAND = """
CREATE TABLE IF NOT EXISTS migration_history (
    migration VARCHAR(255) NOT NULL,
    applied TIMESTAMP DEFAULT NOW() NOT NULL
);
"""


def _get_all_public_tables(cursor):
    cursor.execute(
        "SELECT table_name "
        "FROM information_schema.tables "
        "WHERE table_schema = 'ydir'",
    )
    return [i['table_name'] for i in cursor.fetchall()]


def get_migration_files_for_db(db_alias):
    prefix = 'intranet/yandex_directory/src/yandex_directory/core/db/migrations/%s/' % db_alias

    return [
        file_name for file_name in
        resource.resfs_files(prefix=prefix)
        if file_name.endswith('.sql')
    ]


def migrate(rebuild=False, verbose=False, path=None, admin_user=None):
    # Create db if it does not exist
    print('Starting migration process')
    for db_alias, db_config in list(app.config['DATABASES'].items()):
        for shard, shard_config in list(db_config.items()):
            hosts = ','.join(db.sorted_hosts(shard_config['master']['host']))
            connection_info = {
                'host': hosts,
                'port': shard_config['master']['port'],
                'database': shard_config['master']['database'],
                'user': shard_config['master']['user'],
                'password': shard_config['master']['password'],
                'cursor_factory': psycopg2.extras.RealDictCursor,
                'target_session_attrs': 'read-write',
            }
            try:
                connection = psycopg2.connect(**connection_info)
                connection.set_isolation_level(psycopg2.extensions.ISOLATION_LEVEL_AUTOCOMMIT)
                cursor = connection.cursor()
                if rebuild:
                    cursor.execute('DROP TABLE IF EXISTS %s CASCADE' % ','.join(_get_all_public_tables(cursor)))
            except psycopg2.OperationalError as e:
                if 'does not exist' in str(e) \
                   or 'No such database' in str(e):

                    print('Creating database %s' % db_alias)
                    connection_info['database'] = 'postgres'
                    connection = psycopg2.connect(**connection_info)
                    connection.set_isolation_level(psycopg2.extensions.ISOLATION_LEVEL_AUTOCOMMIT)
                    connection.cursor().execute('CREATE DATABASE %s' % shard_config['master']['database'])
                    db.rebuild(db.engines)
                else:
                    # если это было какое-то другое исключение, то нужно прокинуть его,
                    # потому что возможно это какие-то проблемы с базой
                    raise

    # run migrations for every db

    for db_alias, shards in list(db.engines.items()):
        if db_alias not in app.config['MIGRATE_DBS']:
            continue

        master_engines = [shard['master'] for shard in list(shards.values())]

        print('')
        print('Database "%s"' % db_alias)
        if not path:
            path = os.path.join(app.config['BASE_PATH'], 'yandex_directory/core/db/migrations')

        # Выполняем на всех шардах
        for engine in master_engines:
            command = (
                '{command_path} ',
                '-vvv ' if verbose else '',
                '-l 0.1 '
                '-t 1000 ',
                '-d {path}/{db_alias} ',
                '-c "',
                'dbname={database} ',
                'host={host} ',
                'port={port} ',
                'user={admin_user} ',
                'target_session_attrs=read-write ',
            )
            command = ''.join(command)
            if engine.db_info['connection_info'].get('password'):
                command += 'password={password} '
            command += '" '

            conn_info = engine.db_info['connection_info']
            if IS_ARCADIA:
                path_pgmigrate = 'contrib/python/yandex-pgmigrate/bin/pgmigrate'
                try:
                    command_path = yatest.common.binary_path(path_pgmigrate)
                except AttributeError:
                    # only for local pycharm tests
                    command_path = os.path.join(os.environ["Y_PYTHON_SOURCE_ROOT"], path_pgmigrate)

            else:
                command_path = 'pgmigrate'

            base_command = command.format(
                command_path=command_path,
                path=path,
                db_alias=db_alias,
                admin_user=admin_user or app.config['MIGRATE_DATABASE_USER'],
                database=conn_info['database'],
                host=conn_info['host'],
                port=conn_info['port'],
                password=conn_info.get('password'),
            )
            if rebuild and (app.config['ENVIRONMENT'] in app.config['PRODUCTION_ENVIRONMENTS'] or app.config['ENVIRONMENT'] == 'testing'):
                print('WARNING: rebuild option only for development environment.')
            elif rebuild:
                if not IS_ARCADIA or os.environ.get('PYCHARM_TEST_RUN'):
                    subprocess.check_call(base_command + ' clean', shell=True)
                else:
                    yatest.common.execute(
                        base_command + ' clean',
                        shell=True,
                        check_exit_code=True
                    )

            print(base_command)
            if not IS_ARCADIA or os.environ.get('PYCHARM_TEST_RUN'):
                subprocess.check_call(base_command + ' migrate', shell=True)
            else:
                yatest.common.execute(
                    base_command + ' migrate',
                    shell=True,
                    check_exit_code=True
                )


class Command(BaseCommand):
    name = 'migrate'
    option_list = (
        Option(
            '--rebuild',
            '-r',
            dest='rebuild',
            default=False,
            action='store_true',
            help='Drop all tables and migrate from zero.'
        ),
        Option(
            '--verbose',
            '-v',
            dest='verbose',
            default=False,
            action='store_true',
            help='Output verbose information.'
        ),
    )

    def run(self, rebuild, verbose):
        path = '/root/app/yandex_directory/core/db/migrations'
        migrate(
            rebuild=rebuild,
            verbose=verbose,
            path=path,
        )
