# encoding: UTF-8

"""
Методы для обратной совместимости кода в старом и новом подходе
"""



from contextlib import contextmanager
from functools import wraps

from flask import g
from sqlalchemy.orm import Session

from intranet.yandex_directory.src.yandex_directory.auth.scopes import check_scopes
from intranet.yandex_directory.src.yandex_directory.common.db import get_main_connection
from intranet.yandex_directory.src.yandex_directory.common.utils import check_permissions
from intranet.yandex_directory.src.yandex_directory.common.utils import force_text
from intranet.yandex_directory.src.yandex_directory.core.permission import permission_descriptions
from intranet.yandex_directory.src.yandex_directory.meta.components import get_meta_session
from intranet.yandex_directory.src.yandex_directory.common.exceptions import NoScopesError
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log


def permission_required_compat(
        permissions,
        object_type=None,
        any_permission=False,
):
    """
    Декоратор, задающий права для метода
    Args:
        permissions (list): необходимые права, список строк из мн-ва
            core.views.permissions.global_permissions и
            core.views.permissions.group_permissions
        object_type (string): тип объекта
            {'group', 'user', 'department', 'resource'}
            Нужен для проверки прав на один объект с его id
        any_permission (bool): проверка должна проходить если у пользователя есть хотя бы одно право из списка
            В этом случае, дальнейшие проверки прав должны проходить внутри view-функции
    """
    if not hasattr(permissions, '__iter__') or isinstance(permissions, (str, bytes)):
        raise TypeError('First argument must be iterable object')

    def decorator(view_func):
        @wraps(view_func)
        def _wrapped_view(view_instance, *args, **kwargs):
            object_id = None
            if object_type:
                object_id = kwargs.get(object_type + '_id')

            meta_session = get_meta_session()
            shard_number = getattr(g, 'shard', None)
            if shard_number is None:
                check_permissions(
                    meta_session.connection(),
                    None,
                    permissions,
                    object_type,
                    object_id,
                    any_permission=any_permission,
                )
            else:
                with get_main_connection(shard_number) as main_connection:
                    check_permissions(
                        meta_session.connection(),
                        main_connection,
                        permissions,
                        object_type,
                        object_id,
                        any_permission=any_permission,
                    )

            return view_func(view_instance, *args, **kwargs)

        all_descriptions = [
            permission_descriptions.get(permission, '')
            for permission in permissions
        ]
        permissions_description = ' '.join(map(force_text, all_descriptions))
        view_doc = force_text(view_func.__doc__)
        if '{permissions}' in view_doc:
            view_doc = view_doc.replace(
                '{permissions}',
                permissions_description
            )
        _wrapped_view.__doc__ = view_doc
        new_permissions = permissions
        if hasattr(view_func, 'permissions'):
            new_permissions = permissions + view_func.permissions
        _wrapped_view.permissions = new_permissions

        return _wrapped_view

    return decorator


@contextmanager
def session_compat(connection, *args, **kwargs):
    """
    Контектсный менеджер для обратной совместимости с meta_connection,
    main_connection.
    """
    session = Session(connection, *args, **kwargs)
    try:
        yield session
    finally:
        session.expunge_all()


def scopes_required_compat(scopes):
    def decorator(view_func):
        assert isinstance(scopes, list), 'Аргумент scopes_required должен быть списком'

        @wraps(view_func)
        def _wrapped_view(view_instance,
                          *args,
                          **kwargs):
            # Независимо от того, откуда пришёл запрос, проверяем
            # наличие нужных вьюхе скоупов
            if not check_scopes(g.scopes, scopes):
                with log.fields(scopes=g.scopes,
                                required_scopes=scopes):
                    log.warning('No required scope')

                raise NoScopesError(*scopes)

            return view_func(
                view_instance,
                *args,
                **kwargs
            )

        _wrapped_view.scopes = scopes
        return _wrapped_view

    return decorator

