# -*- coding: utf-8 -*-
import hashlib
import os
import threading
import time
from collections import defaultdict
from contextlib import contextmanager
from functools import wraps

from flask import g
from psycopg2.extras import Json as PgSQLJson
from retrying import retry
from sqlalchemy import create_engine
from sqlalchemy import event
from sqlalchemy.engine import Engine
from sqlalchemy.exc import (
    IntegrityError,
    OperationalError,
)

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common import json
from intranet.yandex_directory.src.yandex_directory.common.caching.utils import make_key
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ReadonlyConnectionError,
    UnknownShardError,
    ReadonlyModeError,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    is_debug_logging_enabled,
    build_dsn,
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log, requests_log

DATABASES_DATA = {}

# Эти переменные используются только в юнит-тестах и при рендеринге
# документации, для того, чтобы get_main_connection и get_meta_connection
# выдавали всегда один и тот же коннект, созданный в рамках теста и не было
# проблем с тем, что данные поменялись в какой-то параллельной транзакции,
# а тест их не видит, потому что у него другой коннект.
_meta_connection = None
_main_connection = None


def set_permanent_main_connection(connection):
    global _main_connection
    _main_connection = connection


def set_permanent_meta_connection(connection):
    global _meta_connection
    _meta_connection = connection


def _retry_on_db_error(exc, error_msg):
    return isinstance(exc, OperationalError) and error_msg in str(exc.orig)


def retry_on_close_connect_to_server(exc):
    """
    Проверка на необходимость ретрая при разрыве соединения к БД
    :param exc: Исключение
    :type exc: Exception
    :rtype: bool
    """
    return _retry_on_db_error(exc, 'SSL connection has been closed unexpectedly')


def retry_if_cannot_connect_to_server(exc):
    """
    Проверка на необходимость ретрая попытки соеденения к БД
    :param exc: Исключение
    :type exc: Exception
    :rtype: bool
    """
    return (
            _retry_on_db_error(exc, 'pgbouncer cannot connect to server') or
            _retry_on_db_error(exc, 'timeout expired')
    )


def before_cursor_execute(conn,
                          cursor,
                          statement,
                          parameters,
                          context,
                          executemany):
    conn.info.setdefault('query_start_time', []).append(time.time())


def get_sql_profile_fields(conn, statement, parameters):
    start_time = conn.info['query_start_time'][-1]
    end_time = time.time()
    total = end_time - start_time
    return {
        'connection': conn.engine,
        'query': statement,
        'parameters': parameters,
        'start_time': start_time,
        'end_time': end_time,
        'total_time': total,
    }


def send_stats_to_golovan(conn,
                         cursor,
                         statement,
                         parameters,
                         context,
                         executemany):

    sql_profile_fields = get_sql_profile_fields(conn, statement, parameters)
    shard = '{alias}_{shard}_{role}'.format(**conn.engine.db_info)
    query_type = 'other'
    query_simplify = sql_profile_fields['query'].lstrip().lower()
    for type in ['select', 'insert', 'update', 'delete']:
        if query_simplify.startswith(type):
            query_type = type

    if app.config['LOG_SQL_QUERIES']:
        requests_log.debug(
            'SQL QUERY TO %s:\n%s\n%s' % (
                shard,
                sql_profile_fields['query'],
                sql_profile_fields['parameters'],
            ),
            extra={
                'request_time': sql_profile_fields['total_time']
            },
        )

    app.stats_aggregator.add_to_bucket(
        metric_name='database_response_time',
        value=sql_profile_fields['total_time'],
    )
    app.stats_aggregator.add_to_bucket(
        metric_name='database_response_time_{}_{}'.format(query_type, shard),
        value=sql_profile_fields['total_time'],
    )

    app.stats_aggregator.inc(
        metric_name='database_requests_{}_{}_summ'.format(query_type, shard)
    )
    app.stats_aggregator.inc(
        metric_name='database_requests_summ'
    )

    # Если были установлены какие-то функции-хуки,
    # то передадим в них информацию о запросах.
    # Это используется для тестов, чтобы считать количество запросов к базе
    # с помощью контекстного менеджера with catched_sql_queries() as sql.
    #
    # Каждый хук должен быть функцией, которая принимает именованые параметры
    # и **kwargs.
    # Параметры могут быть такие:
    #
    # connection - engine sql алхимии
    # query - строка с запросом
    # parameters - параметры запроса
    # start_time - время начала запроса
    # end_time - время окончания запроса
    # total_time - время выполнения запроса в секундах

    try:
        hooks = getattr(g, 'sql_hooks')
    except Exception:
        # Тут ловим и игнорируем AttrbuteError,
        # в случае, если атрибута нет, и RuntimeError
        # в случае, если запрос выполняется не в контексте
        # flask:
        # https://st.yandex-team.ru/DIR-2624
        hooks = ()

    for hook in hooks:
        hook(**sql_profile_fields)


def print_sql(sql):
    """Печатает запросы, пойманные с помощью catched_sql_queries.

    Полезно при отладке.
    """
    print(('\n\n'.join(item[0] for item in sql)))


@contextmanager
def catched_sql_queries():
    queries = []

    def catcher(query, total_time, connection, **kwargs):
        queries.append((query, total_time, connection.url.database))

    # Установим новый хук
    previous = getattr(g, 'sql_hooks', ())
    g.sql_hooks = previous + (catcher,)

    # Вернём управление в вызывающий код
    yield queries

    # Вернём список хуков в первоначальное состояние
    g.sql_hooks = previous


def after_dbapi_error(conn, cursor, statement, parameters, context, exception):
    app.stats_aggregator.inc('db_api_errors_count_summ')
    app.stats_aggregator.inc('db_api_errors_to_' + conn.engine.db_info['alias'] + '_' +
                             str(conn.engine.db_info['shard']) + '_' + conn.engine.db_info['role'] + '_summ')
    if is_debug_logging_enabled():
        log_fields = get_sql_profile_fields(conn, statement, parameters)
        log_fields['exception'] = str(exception)

        with log.name_and_fields('db.utils.after-error', **log_fields):
            log.warning('some sql error happened')


# from sqlalchemy import exc
# from sqlalchemy.pool import Pool


# @event.listens_for(Pool, "checkout")
# def ping_connection(dbapi_connection, connection_record, connection_proxy):
#     """Решение из
#     https://stackoverflow.com/questions/34828113/flask-sqlalchemy-losing-connection-after-restarting-of-db-server

#     Работает. Но увеличивает latency при получении коннекта.

#     Оставил для истории.
#     """
#     cursor = dbapi_connection.cursor()
#     try:
#         print 'Pinging connection'
#         cursor.execute("SELECT 1")
#     except:
#         # optional - dispose the whole pool
#         # instead of invalidating one at a time
#         # connection_proxy._pool.dispose()

#         # raise DisconnectionError - pool will try
#         # connecting again up to three times before raising.
#         print 'Connection expired'
#         raise exc.DisconnectionError()
#     cursor.close()


def get_shard_numbers():
    """Возвращает список с номерами шардов, отсортированными по возрастанию.

    Сортировка нужна для того, чтобы можно было пейджинироваться
    по сущностям в нескольких шардах.
    """
    return sorted(engines['main'].keys())


def get_shards_by_org_type(organization_type):
    """
    Возращает список с номерами шардов, которые допустимы для данного типа организаций

    """
    if organization_type == 'portal':
        return app.config['SHARDS_FOR_PORTAL']
    shards = list(set(get_shard_numbers()) - set(app.config['SHARDS_FOR_PORTAL']))
    shards.sort()
    return shards


def get_shards_with_weight(shards, coef=100000):
    """
    Возращает список шардов c взвешенными и отнормированными весами для
    случайного выбора шарда при заведении новой организации с учётом
    заполненности каждого из шардов.
    """
    key = make_key(
        prefix='shards_with_weight',
        key_data=(
            tuple(sorted(shards)),
            coef,
        )
    )
    shards_with_weight = app.cache.get(key)
    if not shards_with_weight:
        query = "select shard, count(*) from organizations join users on users.org_id = organizations.id group by shard"
        with get_meta_connection() as meta_connection:
            user_count_for_shards = dict(meta_connection.execute(query).fetchall())

        # Так как некоторые шарды могут быть пустыми, и SQL запрос данных по ним не отдаст,
        # то надо проставить default = 0
        for shard in shards:
            user_count_for_shards.setdefault(shard, 0)

        # не нам могут быть нужны не все шарды, поэтому оставим только те, что есть в shards
        # и применим формулу DIR-5686
        coef = max(user_count_for_shards.values()) + coef
        shards_with_weight = {
            shard: (1 - float(user_count) / coef)
            for shard, user_count in list(user_count_for_shards.items())
            if shard in shards
        }
        norm_const = sum(shards_with_weight.values())
        shards_with_weight = {
            i: j / norm_const
            for i, j in list(shards_with_weight.items())
        }
        shards_with_weight = sorted([(j, i) for i, j in list(shards_with_weight.items())], key=lambda x: x[0])
        for i in range(1, len(shards_with_weight)):
            shards_with_weight[i] = (shards_with_weight[i][0] + shards_with_weight[i - 1][0], shards_with_weight[i][1])
        app.cache.set(key, shards_with_weight, app.config['SHARDS_WITH_WEIGHT_CACHE_TTL'])
    return shards_with_weight


def _select_engine(db_alias, shard, for_write):
    # # возвращает engine через который можно установить коннект
    # # через контекстный менеджер
    # engines = get_db_engines()
    shards = engines[db_alias]

    if shard not in shards:
        raise UnknownShardError('Unknown shard "{0}"'.format(shard))

    shard = shards[shard]

    if not for_write:
        return shard['replica']

    # последующий код выполняется, когда
    # был запрошен коннект на запись,
    # или на чтение, но не нашлось ни одной живой реплики
    return shard['master']


@retry(stop_max_attempt_number=3,
       wait_incrementing_increment=50,
       retry_on_exception=retry_if_cannot_connect_to_server)
def get_meta_connection(for_write=False, no_transaction=False):
    # в автотестах надо убедиться, что внутри
    # читающих view мы не пытаемся писать в базу
    # а поскольку у нас в юниттестах все коннекты
    # к мастеру, тот нужна такая обертка, которая
    # запрещает всё кроме селектов

    if for_write and app.config['READ_ONLY_MODE']:
        raise ReadonlyModeError

    if not for_write and app.config['ENVIRONMENT'] == 'autotests':
        wrapper = ReadOnly
    else:
        wrapper = lambda connection: connection

    if not for_write:
        no_transaction = True

    if _meta_connection is not None:
        return start_transaction(not_closing(wrapper(_meta_connection)), no_transaction)
    else:
        # у метабазы всегда один шард
        shard = 1
        engine = _select_engine('meta', shard, for_write)
        connect = engine.connect()
        connect.shard = 'meta'
        return start_transaction(closing(wrapper(connect)), no_transaction)


def get_or_pass_meta_connection(meta_connection=None, for_write=False, no_transaction=False):
    if meta_connection is not None:
        return not_closing(meta_connection)
    return get_meta_connection(for_write=for_write, no_transaction=no_transaction)


def get_or_pass_main_connection(shard, main_connection=None, for_write=False, no_transaction=False):
    if main_connection is not None:
        return not_closing(main_connection)
    return get_main_connection(shard, for_write=for_write, no_transaction=no_transaction)


@contextmanager
def not_closing(connection):
    """Простой контекстный менеджер, который не закрывает соединение.

    Нужен исключительно для того, чтобы результатом функции get_main_connection
    можно было пользоваться внутри блока with.
    """
    yield connection


@contextmanager
def closing(connection):
    """Простой контекстный менеджер, который возвращает соединение в пул.

    Нужен исключительно для того, чтобы результатом функции get_main_connection
    можно было пользоваться внутри блока with.
    """
    try:
        yield connection
    finally:
        connection.close()


@contextmanager
def start_transaction(context_manager, no_transaction):
    """
    Контекстный менеджер, который начинает транзакцию, если no_transaction == False
    """
    with context_manager as connection:
        if no_transaction:
            yield connection
        else:
            with connection.begin_nested():
                yield connection


class NoNeedToCloseConnection(object):
    """Этот контекстный менеджер нужен для того, чтобы предотвратить
    закрытие коннекта во время выполнения юнит-теста. Ведь юнит-тесты
    у нас выполняются в рамках одного коннекта и потом транзакция откатывается.
    """

    def __init__(self, connection):
        self.__conn = connection

    def __getattr__(self, name):
        if name == '__wrapped__':
            raise AttributeError
        return getattr(self.__conn, name)

    def __eq__(self, other):
        return self.__conn is other

    def __ne__(self, other):
        return self.__conn is not other

    def close(self):
        pass


@retry(stop_max_attempt_number=3,
       wait_incrementing_increment=50,
       retry_on_exception=retry_if_cannot_connect_to_server)
def get_raw_main_connection(shard, for_write=False):
    """Внимание! За закрытие коннекта, возвращаемого данной
    функцией, ответственен вызывающий её код.
    """
    if for_write and app.config['READ_ONLY_MODE']:
        raise ReadonlyModeError

    if _main_connection is not None:
        result = NoNeedToCloseConnection(_main_connection)
    else:
        engine = _select_engine('main', shard, for_write)
        result = engine.connect()

    # в автотестах надо убедиться, что внутри
    # читающих view мы не пытаемся писать в базу
    # а поскольку у нас в юниттестах все коннекты
    # к мастеру, тот нужна такая обертка, которая
    # запрещает всё кроме селектов
    if not for_write and app.config['ENVIRONMENT'] == 'autotests':
        result = ReadOnly(result)

    result.shard = shard
    return result


@retry(stop_max_attempt_number=3,
       wait_incrementing_increment=50,
       retry_on_exception=retry_if_cannot_connect_to_server)
def get_main_connection(shard, for_write=False, no_transaction=False):
    # в автотестах надо убедиться, что внутри
    # читающих view мы не пытаемся писать в базу
    # а поскольку у нас в юниттестах все коннекты
    # к мастеру, тот нужна такая обертка, которая
    # запрещает всё кроме селектов
    if for_write and app.config['READ_ONLY_MODE']:
        raise ReadonlyModeError

    if not for_write and app.config['ENVIRONMENT'] == 'autotests':
        wrapper = ReadOnly
    else:
        wrapper = lambda connection: connection

    if not for_write:
        no_transaction = True

    if _main_connection is not None:
        return start_transaction(not_closing(wrapper(_main_connection)), no_transaction)
    else:
        engine = _select_engine('main', shard, for_write)
        connect = engine.connect()
        # Запомним шард для которого сгенерили коннект
        # нам потом пригодится при логгинге.
        connect.shard = shard
        return start_transaction(closing(wrapper(connect)), no_transaction)


@retry(stop_max_attempt_number=3,
       wait_incrementing_increment=50,
       retry_on_exception=retry_if_cannot_connect_to_server)
def get_main_connection_for_new_organization(for_write=False, no_transaction=False):
    # Получаем шард для новой организации и возвращаем коннект к нему
    from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationMetaModel

    shard = OrganizationMetaModel.get_shard_for_new_organization()
    return get_main_connection(shard, for_write=for_write, no_transaction=no_transaction)


class AlreadyLockedError(RuntimeError):
    pass


@contextmanager
def wait_lock(connection, lock_name, ttl=60):
    while True:
        try:
            lock(connection, lock_name)
            yield
            break

        except AlreadyLockedError:
            time.sleep(0.1)
            ttl -= 0.1
            if ttl < 0:
                raise


@contextmanager
def lock(connection, lock_name):
    """
    Пытаемся получить блокировку по хэшу ключа lock_name.
    Для того, чтобы взять такой лок, не обязательно открывать транзакцию.

    Входной параметр connection должен быть коннектом, открытым на запись.
    """
    lock_key = int(hashlib.sha256(lock_name.encode('utf-8')).hexdigest(), 16) % 10**8
    params = {'key': lock_key}

    # пробуем взять лок
    with log.fields(lock_key=lock_key, lock_name=lock_name):
        if not connection.in_transaction():
            raise RuntimeError('Not in transaction')

        log.debug('Trying to get pg_try_advisory_xact_lock')
        locked = connection.execute(
            'select pg_try_advisory_xact_lock(%(key)s)',
            params,
        ).fetchone()[0]

        if not locked:
            log.debug('Lock aquired by someone else.')
            raise AlreadyLockedError('Lock {0} aquired by someone else.'.format(lock_name))
        log.debug('Lock aquired')
        yield locked
        log.debug('pg_advisory_unlock')


def sorted_hosts(hosts):
    """Получает на вход строку или список строк.
       Если на входе был список, сортирует хосты по удалённости от текущей машинки.
       Каждый хост должен начинаться с названия датацентра.
       Например: sas-k3lpwznrunwkk6zb.db.yandex.net
       Или хост может быть словарём, где явно указан датацент:
       {'host': 'dirdb01h.db.yandex.net', 'dc': 'sas'}
       тогда на выходе, в отсортированном списке, всё равно будет строка:
       'dirdb01h.db.yandex.net'.
    """
    if isinstance(hosts, str):
        return [hosts]
    else:
        preferred_dcs = {
            'iva': ('iva', 'myt', 'vla', 'sas', 'man'),
            'myt': ('myt', 'iva', 'vla', 'sas', 'man'),
            'sas': ('sas', 'vla', 'iva', 'myt', 'man'),
            'vla': ('vla', 'iva', 'myt', 'sas', 'man'),
            'man': ('man', 'myt', 'iva', 'vla', 'sas'),
        }

        current_dc = os.environ.get('DEPLOY_NODE_DC', os.environ.get('QLOUD_DATACENTER', 'myt')).lower()
        if current_dc not in preferred_dcs:
            raise RuntimeError('DC {0} is not supported'.format(current_dc))

        ranks = {
            dc: rank
            for rank, dc in enumerate(preferred_dcs[current_dc])
        }

        def get_dc(host):
            if isinstance(host, str):
                return host.split('-', 1)[0]
            elif isinstance(host, dict):
                return host['dc']
            else:
                raise RuntimeError('Hosts like {} are not supported'.format(host))

        def ensure_string(host):
            if isinstance(host, str):
                return host
            else:
                return host['host']

        return list(map(ensure_string, sorted(hosts, key=lambda host: ranks[get_dc(host)])))


def rebuild(data):
    """Внимание, эта функция модифицирует
    переданный ей словарь inplace.

    Аргументом должен быть словарь типа:
    defaultdict(
        lambda: defaultdict(
            lambda: defaultdict(list)
        )
    )

    В результате создается словарь:
    {
        'meta': {
            1: {
                'master': Engine(...),
                'replica': Engine(...),
            },
        }
        'main': {
            1: {
                'master': [engine],
                'replica': [engine, engine],
            },
            2: {
                'master': [engine],
                'replica': [engine, engine],
            }
        }
    }

    На первом уровне alias базы, на втором шард,
    на третьем тип базы master или реплика, а затем коннект.
    """
    for db_alias, db_shards_config in list(app.config['DATABASES'].items()):
        for shard, db_shard_config in list(db_shards_config.items()):
            for role in ['master', 'replica']:
                config = db_shard_config[role]
                dsn_kwargs = {
                    'host': ','.join(sorted_hosts(config['host'])),
                    'port': config['port'],
                    'database': config['database'],
                    'user': config['user'],
                    # Эти параметры рекомендованы d0uble.
                    # Со стороны pgbouncer они такие же.
                    # Нам настройки keep alive нужны для того, чтобы
                    # уменьшить вероятность того, что во время выполнения
                    # долгого SQL запроса, какой-то промежуточный узел
                    # сети решит порвать соединение из-за отсутствия пакетиков.
                    'keepalives_idle': 15,
                    'keepalives_interval': 5,
                    'keepalives_count': 3,
                }
                if config.get('password'):
                    dsn_kwargs['password'] = config.get('password')

                dsn = build_dsn(**dsn_kwargs)

                engine = create_engine(
                    dsn,
                    connect_args=config['connect_args'],
                    pool_size=config['pool_size'],
                    max_overflow=config['pool_max_overflow'],
                    pool_recycle=config['pool_recycle'],
                )

                engine.db_info = {
                    'connection_info': dsn_kwargs,
                    'dsn': dsn,
                    'secure_dsn': build_dsn(
                        host=dsn_kwargs['host'],
                        port=dsn_kwargs['port'],
                        database=dsn_kwargs['database'],
                    ),
                    'role': role,
                    'shard': shard,
                    'alias': db_alias,
                }

                if db_alias not in data:
                    data[db_alias] = {}
                if shard not in data[db_alias]:
                    data[db_alias][shard] = {}
                data[db_alias][shard][role] = engine

    # убеждаемся, что у нас всего один шард с метабазой
    assert len(data['meta']) == 1, 'We have {0} shards for metabase, but only 1 should exists'.format(
        len(data['meta'])
    )

    return data


def get_connection_info_for(engine):
    return engine.db_info['connection_info']


def get_secure_dsn_for(engine):
    return engine.db_info['secure_dsn']


def mogrify(connection, query, vars=None):
    return connection.connection.cursor().mogrify(query=query, vars=vars).decode('utf-8')


class Json(PgSQLJson):
    def __init__(self, obj):
        if isinstance(obj, (Json, PgSQLJson)):
            raise RuntimeError('Object already wrapped into the Json')
        super(Json, self).__init__(obj)

    def dumps(self, obj):
        return json.dumps(obj)


def retry_if_db_integrity_error(exception):
    """
    Return True if we should retry
    (in this case when it's an IntegrityError), False otherwise
    """
    return isinstance(exception, IntegrityError)


def get_shard(meta_connection, org_id=None):
    """
    Определяем шард организации или возвращаем None, если org_id==None.

    Если организация не найдена в базе, то тоже возвращаем None.
    """
    shard = None
    if org_id is not None:
        with log.fields(org_id=org_id):
            try:
                return get_shards_by_org_ids(org_id)[org_id]
            except OrganizationIsNotInAnyShardError:
                return None
    return shard


class OrganizationIsNotInAnyShardError(Exception):
    pass


def get_shards_by_org_ids(*org_ids):
    """
    Определяем шарды организаций или бросаем исключение OrganizationIsNotInAnyShardError.

    Если организация не найдена в базе, то в лог пишем warning.
    """
    from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationMetaModel

    with get_meta_connection() as meta_connection:
        organization_meta = {
            meta['id']: meta['shard']
            for meta in OrganizationMetaModel(meta_connection).filter(id__in=org_ids).fields('id', 'shard')
        }
    for org_id in org_ids:
        with log.fields(org_id=org_id):
            if org_id not in organization_meta:
                log.warning('Unable to find shard for organization: %d not in %s', org_id, organization_meta)
                raise OrganizationIsNotInAnyShardError()
    return organization_meta


def use_connections(for_write=False):
    """Декоратор, который передает в обертнутую функцию
    новые коннекты к мета и обычной базам.

    При этом, для того, чтобы вычислился шард, в kwargs функции должен
    присутствовать org_id.
    """

    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            if 'org_id' not in kwargs:
                raise RuntimeError('Keyword argument "org_id" should be passed to the function.')

            org_id = kwargs['org_id']

            with get_meta_connection(for_write=for_write) as meta_connection:

                shard = get_shard(
                    meta_connection,
                    org_id=org_id,
                )
                if shard is None:
                    raise RuntimeError('Shard not found')

                with get_main_connection(shard, for_write=for_write) as main_connection:
                    return func(
                        meta_connection,
                        main_connection,
                        *args,
                        **kwargs
                    )

        return wrapper

    return decorator


class ReadOnly(object):
    """Эта обертка проверяет, что через коннект не пытаются выполнить пишущих
    операций. Нужна для того, чтобы в юнит-тестах ограничить пишущие ручки.

    Это всё потому, что у нас все юнит-тесты работают в рамках одного коннекта,
    который открывается к мастеру.

    Без этого ограничения, возможна ситуация, когда тесты не отловят попытку
    записать что-либо в базу через соединение к реплике.
    """

    def __init__(self, connection):
        self.__connection = connection

    def __getattr__(self, name):
        if name == '__wrapped__':
            raise AttributeError
        # чтобы не пропустить никакие важные атрибуты,
        # используем whitelist
        allowed = {'mogrify', 'close', 'begin_nested', 'in_transaction', 'engine'}

        if name in allowed:
            return getattr(self.__connection, name)
        raise AttributeError('Attribute {0} not found'.format(name))

    @property
    def connection(self):
        # у connection из SQLAlchemy, есть внутренний connection
        # из psycopg2. Его то мы тут и обертываем
        return ReadOnly(self.__connection.connection)

    def cursor(self):
        # некоторые берут из connection курсор и дальше работают
        # с ним. Поэтому его тут тоже надо обернуть в ReadOnly
        return ReadOnly(self.__connection.cursor())

    def execute(self, statement, *args, **kwargs):
        # так как в запросе может быть несколько SQL
        # конструкций, то нам надо проверить каждую
        queries = statement.split(';')
        queries = [q.lstrip() for q in queries]
        queries = [_f for _f in queries if _f]

        for query in queries:
            first_word = query.split(' ', 1)[0]

            if first_word.lower().strip() not in ('select', 'with'):
                raise ReadonlyConnectionError(
                    'Attempting to modify database through readonly connection with "{}"'.format(
                        query
                    )
                )

        return self.__connection.execute(statement, *args, **kwargs)


engines = defaultdict(
    lambda: defaultdict(
        lambda: defaultdict(list)
    )
)


def ping_pool_connections(engine):
    """Эта функция проверяет все коннекты которые есть в пуле и не используются,
    и инвалидирует те, которые "отвалились" или у них скоро должно подойти
    к концу время жизни.

    Проверку живости коннекта мы делаем путём простого селекта. Если он не
    прошёл, то мы инвалидируем коннект, это позволяет минимизировать количество
    ошибок, видимых конечным пользователям, так как даже если порестартили базу,
    мы обновим все коннекты в пуле в течении 15 секунд.

    Инвалидация до истечении времени жизни коннекта (recycle) нужна для того,
    чтобы в пуле коннекты всегда оставались свеженькими и не было необходимости
    устанавливать новое соединение в тот момент когда к нам в API пришёл запрос.
    """
    now = time.time()
    # Столько коннектов сейчас лежат свободные в пуле
    # и их все надо пропингать.
    num_free_connects = engine.pool.checkedin()
    connections = [
        engine.connect()
        for i in range(num_free_connects)
    ]
    recycle_time = engine.pool._recycle

    for connection in connections:
        try:
            connection.execute('select 1')
        except Exception:
            log.trace().warning('Connection invalidated')
            connection.invalidate()
            continue

        if recycle_time > 0:
            # Если до конца жизни коннекта осталось меньше 30 секунд,
            # то инвалидирем его.
            created_at = connection.connection._connection_record.starttime
            connection_age = now - created_at
            if connection_age > (recycle_time - 30):
                log.debug('Connection was recycled')
                connection.invalidate()
                continue

        # Если ничего такого не случилось, то закрываем коннект, чтобы
        # вернуть его в пул.
        connection.close()


def ping_all_connections():
    """Пингует неиспользуемые коннекты во всех пулах.
    """
    for name, shards in list(engines.items()):
        for shard, shard_engines in list(shards.items()):
            for role, engine in list(shard_engines.items()):
                ping_pool_connections(engine)


def connection_pinger():
    """Вечно пингует все коннекты к базе в каждом из пулов.
    """
    sleep_between = app.config['PING_DB_CONNECTIONS_INTERVAL']

    with log.name_and_fields('connection-pinger'):
        while True:
            try:
                ping_all_connections()
            except:
                log.trace().error('Unhandled exception')

            time.sleep(sleep_between)


connection_pinger_thread = None


def setup_engines(app):
    rebuild(engines)


def setup_database(app):
    global connection_pinger_thread

    event.listen(Engine, "after_cursor_execute", send_stats_to_golovan)
    event.listen(Engine, "before_cursor_execute", before_cursor_execute)
    event.listen(Engine, "dbapi_error", after_dbapi_error)

    if app.config['PING_DB_CONNECTIONS_INTERVAL']:
        connection_pinger_thread = threading.Thread(target=connection_pinger)
        connection_pinger_thread.daemon = True
        connection_pinger_thread.start()
