# coding: utf-8
import inspect

from abc import ABCMeta, abstractproperty
from werkzeug.exceptions import BadRequest


class ImmediateReturn(Exception):
    """Это исключение нужно, чтобы можно было с помощью него
    преждевременно закончить обработку view.
    Например если мы при валидации поняли, что произошла ошибка.
    """

    def __init__(self, response):
        self.response = response


class ConstraintValidationError(Exception):
    """Исключение, возбуждаемое в случае несоблюдения каких-либо ограничений
    при работе с БД"""

    def __init__(self, code, message, **params):
        self.error_code = 'constraint_validation.' + code
        self.error_message = message
        self.params = params

        # чтобы у исключения было нормальное описание,
        # надо подставить параметры вместо плейсхолдеров
        # в сообщение об ошибке
        super(ConstraintValidationError, self).__init__(
            message.format(**params)
        )


class APIError(RuntimeError, metaclass=ABCMeta):
    code = 'api-error'
    with_trace = True
    status_code = 500
    message = 'Unknown API error.'
    log_level = 'ERROR'
    headers = {}
    # Человеческое описание, на которое мы скоро заменим message
    description = abstractproperty()

    def __init__(self, **params):
        """В **params можно передать переменные, которые
        должны быть подставлены в message и description, перед тем, как
        показывать ошибку пользователю.
        """

        # Чтобы можно было инстанциировать только "листья" дерева из исключений.
        if inspect.isabstract(self.__class__):
            raise RuntimeError(
                'Some abstract methods and properties are not implemented for class {}'.format(self.__class__.__name__))

        self.params = params


class AuthenticationError(APIError):
    code = 'authentication-error'
    status_code = 401
    log_level = 'WARNING'
    description = 'Невозможно аутентифицировать запрос. Проверьте правильность переданных заголовков.'

    def __init__(self, message, headers={}):
        super(AuthenticationError, self).__init__()
        self.message = message
        if headers:
            self.headers = headers


class AuthorizationError(APIError):
    with_trace = False
    code = 'authorization-error'
    status_code = 403
    log_level = 'WARNING'
    description = 'У вас нет прав на выполнение данной операции.'

    def __init__(self, message, error_code=None, headers={}, **params):
        super(AuthorizationError, self).__init__(**params)
        self.message = message
        if error_code:
            self.code = error_code
        if headers:
            self.headers = headers


class NoScopesError(AuthorizationError):
    code = 'no-required-scopes'
    description = 'Для работы с API требуется один или несколько скоупов: {scopes}.'
    message = 'This operation requires one of scopes: {scopes}.'

    def __init__(self, *scopes):
        super(NoScopesError, self).__init__(
            self.message,
            scopes=', '.join(scopes)
        )


class OrganizationNotReadyError(APIError):
    code = 'not-ready'
    status_code = 403
    log_level = 'WARNING'
    message = 'Organization is not ready yet'
    description = 'Организация ещё не готова к работе. Пожалуйста, подождите.'


class ReadonlyConnectionError(RuntimeError):
    """Это исключение генерится при попытке выполнения деструктивной операции
    через коннект, взятый на чтение. Но это только для юниттестов.
    """
    pass


class ReadonlyModeError(RuntimeError):
    """Это исключение генерится если приложение находится в ro режиме.
    """
    pass


class UnknownShardError(RuntimeError):
    """Это исключение генерится если запрашиваемый шард не существует.
    """
    pass


class NotFoundError(APIError):
    status_code = 404
    with_trace = False
    code = 'not_found'
    log_level = 'WARNING'
    description = 'Объект не найден'
    message = 'Not found'

    def __init__(self, message=None, **params):
        super(NotFoundError, self).__init__(**params)
        self.message = message or self.message


class DomainNotFound(NotFoundError):
    code = 'domain_not_found'
    message = 'Domain {domain} not found'
    description = 'Домен {domain} не найден.'


class MasterDomainNotFound(DomainNotFound):
    code = 'master_domain_not_found'
    message = 'Domain for organization {org_id} not found'
    description = 'В организации {org_id} нет главного домена.'


class ServiceNotFound(NotFoundError):
    code = 'service_not_found'
    description = 'Сервис не найден.'
    message = 'Service not found'


class UserNotFoundError(NotFoundError):
    code = 'user_not_found'
    description = 'Сотрудник не найден.'


class DomainNotValidatedError(ConstraintValidationError):
    description = 'Имя домена не допустимо.'


class DomainNameTooLongError(RuntimeError):
    description = 'Слишком длинное имя домена.'


class FeatureNotFound(NotFoundError):
    code = 'feature_not_found'
    message = 'Feature not found'
    description = 'Эксперимент ненайден.'


class DeleteMasterDomainError(APIError):
    """ Это исключение генерится при попытке удалить мастер домен
    """
    status_code = 422
    code = 'master_domain_cant_be_deleted'
    message = 'Domain {domain} can\'t be deleted because it is main domain of organization'
    description = 'Основной домен нельзя удалить. Сделайте основным другой домен и попробуйте снова.'


class AdminUidNotFound(APIError):
    """ Это исключение генерится, если не найден admin_uid
    """
    status_code = 422
    code = 'admin_uid_not_found'
    message = 'Admin UID not found'
    description = 'У организации нет владельца.'


class BlockedDomainError(APIError):
    """
    Это исключение генерится при попытке удалить заблокированный домен
    или при смене мастер домена, если мастер или кандидат залокирован
    """
    status_code = 422
    code = 'blocked_domain_cant_be_modified'
    message = 'Domain {domain} is blocked'
    description = 'Заблокированный домен не может быть изменён или удалён.'


class InvalidGroupTypeError(APIError):
    code = 'invalid_group_type'
    status_code = 422
    message = 'Invalid group type "{type}"'
    description = 'Тип команды "{type}" не поддерживается.'

    def __init__(self, group_type):
        super(InvalidGroupTypeError, self).__init__(type=group_type)


class ServiceNotLicensed(APIError):
    code = 'service_not_licensed'
    message = 'Service is not licensed'
    status_code = 422
    description = 'Данный сервис не поддерживает распределение лицензий.'


class EmptyOrganizationName(APIError):
    status_code = 422
    code = 'empty_organization_name'
    message = "Organization name can't be empty string"
    description = 'Заполните название организации.'


class UserNotFoundInOrganizationError(APIError):
    status_code = 422
    code = 'user_not_found_in_organization'
    message = 'Not found user with this uid in current organization'
    description = 'Пользователь не является сотрудником организации.'


class InvalidInputTypeError(APIError):
    status_code = 422
    code = 'invalid_input_type'
    message = 'Request with invalid data was made'
    description = 'Некорректно указаны id и версия регистратора.'


class SubscriptionIncompatibleContainerType(APIError):
    code = 'subscription.incompatible_container_type'
    status_code = 422
    message = 'Incompatible for this organization license container. Use only license by user.'
    description = 'Для партнерских организаций можно выдавать лицензии только на сотрудников .'


class DuplicateDomain(APIError):
    code = 'duplicate_domain'
    status_code = 409
    message = 'Domain already added into another your organization'
    description = 'Данный домен уже прикреплён к одной из ваших организаций.'


class SubscriptionLimitExceeded(APIError):
    code = 'subscription.limit_exceeded'
    status_code = 422
    message = 'Subscription limit exceeded'
    description = 'Лимит лицензий исчерпан.'


class InvalidValue(APIError):
    code = 'invalid_value'
    status_code = 422
    message = 'Invalid value'
    # TODO: https://st.yandex-team.ru/DIR-7720
    description = 'Значение указано неверно'

    def __init__(self, message=None, error_code=None, status_code=None, **params):
        super(InvalidValue, self).__init__(**params)
        if message:
            self.message = message
        if error_code:
            self.code = error_code
        if status_code:
            self.status_code = status_code


class ResourceAlreadyExists(APIError):
    code = 'resource_already_exists'
    status_code = 409
    message = 'Resource with such id already exists'
    description = 'Ресурс с таким id уже привязан к организации.'


class IsNotMember(APIError):
    code = 'is_not_member'
    status_code = 422
    message = 'The {type} with id {id} is not a member of the organization'
    description = 'Один из аккаунтов не является сотрудником организации.'


class ParameterNotCorrect(APIError):
    code = 'parameters_not_correct'
    status_code = 422
    message = 'Parameter not correct'
    description = 'Не указан обязательный параметр: {parameter}.'


class UnableToCreateMailList(APIError):
    code = 'unable_to_create_maillist'
    status_code = 422
    message = 'Unable to create maillist for organization without domains'
    description = 'Невозможно создать рассылку, так как в организации нет домена.'


class VerificationAlreadyInProgress(APIError):
    log_level = 'WARNING'
    with_trace = False
    code = 'already_in_progress'
    status_code = 409
    message = 'Verification is in progress'
    description = 'Подтверждение домена уже запущено.'


class CantMoveDepartmentToItself(APIError):
    code = 'cant_move_department_to_itself'
    status_code = 409
    message = 'Department could not be used as parent of itself'
    description = 'Отдел не может являться подотделом самого себя.'


class CantMoveDepartmentToDescendant(APIError):
    code = 'cant_move_department_to_descendant'
    status_code = 409
    message = ('Department with id {child_id} is a descendant of department with id {parent_id} and cannot be used as '
               'it\'s parent')
    description = 'Нельзя перемещать отдел в один из его собственных подотделов.'


class NewAdminAlreadyOwner(APIError):
    code = 'new_admin_already_owner'
    status_code = 422
    message = 'User {uid} is already owner'
    description = 'Вы уже являетесь администратором в организации.'


class CannotChangeOrganizationOwner(APIError):
    code = 'cannot_change_organization_owner'
    status_code = 403
    message = 'Change permissions for organization owner is prohibited'
    description = 'Нельзя отобрать права администратора у владельца организации.'

# SSO exceptions


class DomainUserCreationInSsoOrgsNotAllowed(APIError):
    code = 'cannot_create_domain_user_in_sso_organization'
    status_code = 403
    message = 'Domain user creation in sso organizations is prohibited'
    description = 'Нельзя создавать доменных пользователей в организациях с включенным sso.'


class SsoPersonalInfoPatchNotAllowed(APIError):
    code = 'cannot_change_personal_info_for_sso_user'
    status_code = 403
    message = 'SSO user personal info change is prohibited'
    description = 'Нельзя изменять персональные данные SSO пользователя.'


class SsoPasswordPatchNotAllowed(APIError):
    code = 'cannot_change_password_for_sso_user'
    status_code = 403
    message = 'SSO user password change is prohibited'
    description = 'Нельзя менять пароль SSO пользователя.'


class SsoUserUnblockNotAllowed(APIError):
    code = 'cannot_unblock_sso_user_in_not_sso_organization'
    status_code = 403
    message = 'SSO user unblock in not sso organization is prohibited'
    description = 'Нельзя разблокировать SSO пользователя в организации без SSO.'


class SsoUserBlockInProvisionedOrgNotAllowed(APIError):
    code = 'cannot_block_sso_user_in_provisioned_organization'
    status_code = 403
    message = 'SSO user block in provisioned organization is prohibited'
    description = 'Нельзя заблокировать SSO пользователя в организации c включенным провижинингом.'


class SsoUserUnblockInProvisionedOrgNotAllowed(APIError):
    code = 'cannot_unblock_sso_user_in_provisioned_organization'
    status_code = 403
    message = 'SSO user unblock in provisioned organization is prohibited'
    description = 'Нельзя разблокировать SSO пользователя в организации c включенным провижинингом.'


class SsoUserAliasesNotAllowed(APIError):
    code = 'cannot_create_sso_user_alias'
    status_code = 403
    message = 'SSO user alias creation is prohibited'
    description = 'Нельзя создавать алиасы для SSO пользователя.'


class LastOuterAdminDismissNotAllowed(APIError):
    code = 'cannot_dismiss_last_outer_admin_in_sso_org'
    status_code = 403
    message = 'Last outer admin dismiss in sso organization is prohibited'
    description = 'Нельзя удалить последнего портального администратора в sso организации.'


class UserDoesNotExist(APIError):
    code = 'user_does_not_exist'
    status_code = 422
    message = 'User does not exist'
    description = 'Пользователь не существует.'


class DepartmentNotEmpty(APIError):
    code = 'department_not_empty'
    status_code = 422
    message = 'Users or child departments were found'
    description = 'Нельзя удалить отдел в котором есть сотрудники или другие отделы.'


class RootDepartmentCantBeRemoved(APIError):
    code = 'root_department_cant_be_removed'
    status_code = 422
    message = 'Root department can not be removed'
    description = 'Корневой отдел не может быть удалён.'


class UserAlreadyMemberOfOrganization(APIError):
    log_level = 'WARNING'
    with_trace = False
    code = 'has_organization'
    status_code = 409
    message = 'User already a member of some organization'
    description = 'Аккаунт уже является сотрудником другой организации.'


class UserIsNotMemberOfThisOrganization(APIError):
    code = 'user_is_not_member_of_this_organization'
    status_code = 409
    message = 'User is not a member of this organization'
    description = 'Аккаунт не состоит в этой организации.'


class DomainUserCanNotBeAddedByInvite(APIError):
    code = 'domain_user_can_not_be_added_by_invite'
    status_code = 409
    message = 'Domain user can not be added by invite'
    description = 'Доменный пользователь не может вступить в организацию по приглашению.'


class CantChangeMembersForGroupType(APIError):
    code = 'cant_change_members_for_group_type'
    status_code = 422
    message = 'Can not change members for group of type "{type}"'
    description = 'Состав команд такого типа редактировать нельзя.'


class CantChangeMembersForNurKz(APIError):
    code = 'cant_change_members_nur_kz'
    status_code = 422
    message = 'You cannot change groups, please contact support'
    description = 'Редактирование команд невозможно, свяжитесь с поддержкой'


class CantDeleteGroupType(APIError):
    code = 'cant_delete_this_group_type'
    status_code = 422
    message = 'Can not delete group with type "{type}"'
    description = 'Команды такого типа нельзя удалять.'


class CannotDismissOwner(APIError):
    code = 'cannot_dismiss_owner'
    status_code = 422
    message = 'Organization owner can not be dismissed'
    description = 'Нельзя уволить владельца организации'


class TooManyAliases(APIError):
    code = 'too_many_aliases'
    status_code = 422
    message = 'User has to many aliases'
    description = 'Вы достигли ограничения на число алиасов сотрудника.'


class DuplicateLogin(APIError):
    code = 'duplicate_login'
    status_code = 422
    message = 'The following logins are duplicated: {duplicate_logins}'
    description = 'В данных для миграции повторяются следующие логины: {duplicate_logins}.'


class InvalidDomain(APIError):
    code = 'invalid_domain'
    status_code = 422
    message = 'Invalid domain'
    description = 'Некорректное имя домена.'


class TooManyInvites(APIError):
    code = 'too_many_invites'
    status_code = 403
    message = 'Too many invites'
    description = 'Вы достигли лимита на число приглашений. Отмените часть или дождитесь пока их примут.'


class DomainOccupied(APIError):
    code = 'domain_occupied'
    status_code = 409
    message = 'Domain "{domain}" already occupied'
    description = 'Домен занят.'


class DomainAddToSsoOrganization(APIError):
    code = 'domain_add_to_sso_organization'
    status_code = 403
    message = 'Adding domain to organization with sso turned on is forbidden '
    description = 'Нельзя добавлять домены к организации с включенным SSO.'


class NewAdminFromOtherOrganization(APIError):
    code = 'new_admin_from_other_organization'
    status_code = 422
    message = 'User {uid} is from other organization'
    description = 'Новый владелец должен быть внешней учеткой или состоять в той же ораганизации.'


class InvalidMigration(APIError):
    code = 'invalid_migration'
    status_code = 422
    description = 'В данных для миграции есть ошибки: {errors}'

    def __init__(self, message=None, error_code=None, errors=None):
        super(InvalidMigration, self).__init__(errors=errors)
        if message:
            self.message = message
        if error_code:
            self.code = error_code


class UnknownService(APIError):
    code = 'unknown_service'
    status_code = 404
    message = 'Service {service_slug} not found'
    description = 'Такой сервис не интегрирован с Коннектом.'


class OrganizationIsBlocked(APIError):
    code = 'organization_is_blocked'
    status_code = 403
    message = 'Organization is blocked'
    description = 'Организация заблокирована.'


class RegistrarCheckFailed(APIError):
    code = 'cannot_reg_check'
    status_code = 400
    message = 'Registrar check failed'
    description = 'Проверка регистратора завершилась ошибкой.'


class DomainUserError(APIError):
    log_level = 'WARNING'
    with_trace = False
    code = 'forbidden_for_domain_user'
    status_code = 403
    message = 'This action forbidden for domain user'
    description = 'Операция запрещена для сотрудника внутри домена.'


class UnsupportedService(APIError):
    code = 'unsupported_service'
    status_code = 422
    message = 'Unsupported service'
    description = 'Такой сервис не поддерживается.'


class ResourceNotFound(APIError):
    code = 'resource_not_found'
    status_code = 404
    message = 'Resource not found'
    description = 'Объект не найден.'


class OrganizationAlreadyHasDirectResource(APIError):
    code = 'organization_already_has_direct_resource'
    status_code = 422
    message = 'There is another client_id in organization'
    description = 'К организации можно привязать только один аккаунт Директа.'


class UserNotFoundInBlackbox(APIError):
    code = 'user_not_found_in_blackbox'
    status_code = 404
    message = 'User not found in blackbox'
    description = 'Пользователь не найден в Яндекс Паспорте.'


class InternalRouteError(APIError):
    code = 'internal_route'
    status_code = 404
    message = 'This route is internal'
    description = 'Этот URL доступен только внуренним сервисам.'


class CatalogRouteForbiddenError(APIError):
    code = 'catalog_route'
    status_code = 403
    message = 'This route is available only to yandex-team.ru accounts'
    description = 'Этот URL доступен только для сотрудников Яндекса.'


class HeaderIsRequiredError(APIError):
    code = 'header_is_required'
    status_code = 400
    message = 'Header "{header}" is required.'
    description = 'Нужно указать заголовок "{header}".'

    def __init__(self, header):
        super(HeaderIsRequiredError, self).__init__(header=header)


class HeaderShouldBeIntegerError(APIError):
    code = 'header_should_be_integer'
    status_code = 400
    message = 'Header "{header}" should be a positive integer.'
    description = 'Заголовок "{header}" должен содержать целое положительное число.'

    def __init__(self, header):
        super(HeaderShouldBeIntegerError, self).__init__(header=header)


class NotAYambTestOrganizationError(APIError):
    code = 'not_a_yamb_test_organization'
    status_code = 403
    message = 'The organization is not available for test.'
    description = 'Эта организация не доступна для сервиса yamb-test.'


class ExternalServiceInteractionError(APIError):
    code = 'external_service_error'
    status_code = 424
    message = 'External service error'
    description = 'Ошибка взаимодействия с внешним сервисом.'

    def __init__(self, message=None):
        super(ExternalServiceInteractionError, self).__init__()
        if message:
            self.message = message


class DirectServiceInteractionError(ExternalServiceInteractionError):
    code = 'direct_service_error'
    status_code = 424
    message = 'Direct service error'
    description = 'Ошибка взаимодействия с директом.'


class BillingInvoiceServiceInteractionError(ExternalServiceInteractionError):
    code = 'billing_invoice_api_service_error'
    status_code = 424
    message = 'Billing api service error'
    description = 'Произошла ошибка при взаимодействии с апи биллинга'


class UsersLimitError(Exception):
    """Исключение, возбуждаемое в случае превышения установленного лимита
    на пользователей в организации"""


class UsersLimitApiError(APIError):
    code = 'users_limit_error'
    status_code = 422
    message = 'Users limit in organization exceeded'
    description = 'Превышен лимит сотрудников организации'


class ServiceCanNotBeDisabledError(APIError):
    code = 'service_can_not_be_disabled'
    status_code = 422
    message = 'Service can not be disabled'
    description = 'Сервис не может быть отключен'


class UserOrganizationsLimit(APIError):
    code = 'user_organizations_limit'
    status_code = 409
    message = 'Organizations limit exceeded'
    description = 'Превышен лимит организаций у пользователя'


class UserAlreadyExists(APIError):
    status_code = 422
    code = 'user_already_exists'
    message = 'User already exists'
    description = 'Данный пользователь уже существует.'


class InvalidEmail(APIError):
    status_code = 422
    code = 'invalid_email'
    message = 'Invalid email'
    description = 'Некорректный email'
