# encoding: UTF-8

from contextlib2 import ExitStack
from flask import current_app
from werkzeug.local import LocalProxy

from intranet.yandex_directory.src.yandex_directory.common.appcontext import appcontext_cached
from intranet.yandex_directory.src.yandex_directory.common.appcontext import reset_appcontext_cached
from intranet.yandex_directory.src.yandex_directory.common.extensions import AbstractExtension
from intranet.yandex_directory.src.yandex_directory.common.components import component_registry
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import error_log


class ExitStackExtension(AbstractExtension):
    """
    Расширение добавляющее индивидуальный экземпляр класса ``ExitStack`` к
    контексту приложения Flask. Позволяет оборачивать контекстным менеджером
    весь HTTP-запрос или код внутри контексного менеджера ``app.app_context()``
    с момента вызова ``service_registry().exit_stack.enter_context(ctx_mgr)``.

    В такой стэк удобно добавлять сессии и транзации SQLAlchemy.
    Выход из стека происходит при завершении HTTP-запроса, выхода из контекста
    приложения Flask.

    https://contextlib2.readthedocs.io/en/stable/#contextlib2.ExitStack
    """

    def _setup_extension(self, app):
        app.before_request(self._enter_exit_stack)
        app.teardown_request(self._exit_stack)
        app.teardown_appcontext(self._exit_stack)

        component_registry(app).exit_stack = LocalProxy(
            ExitStackExtension.get_exit_stack,
        )

    def _enter_exit_stack(self):
        self.get_exit_stack()

    def _exit_stack(self, exc):
        try:
            exc_info = (type(exc), exc, None) if exc else (None, None, None)
            self.get_exit_stack().__exit__(*exc_info)
        except Exception:
            error_log.exception(
                'Error occurred on exiting appcontext stack.',
            )
        finally:
            reset_appcontext_cached(self.get_exit_stack)

    @classmethod
    @appcontext_cached
    def get_exit_stack(cls):
        # type: () -> ExitStack

        if not cls.is_setup:
            raise RuntimeError('%s is not set up.' % cls.extension_name())

        return ExitStack().__enter__()
