# encoding: UTF-8



from collections import defaultdict
from copy import deepcopy

from marshmallow import ValidationError
from marshmallow.base import FieldABC
from marshmallow.fields import Field
from marshmallow.fields import missing_
from marshmallow.fields import String
from typing import Mapping

from intranet.yandex_directory.src.yandex_directory.common.datatools import flatten
from intranet.yandex_directory.src.yandex_directory.common.web import ErrorResponseBuilder
from intranet.yandex_directory.src.yandex_directory.common.web import ResponseMixin


class PublicValidationError(ValidationError, ResponseMixin):
    """
    Ошибка валидации для передачи на клиент.
    """

    def as_response(self, i18n_localizer):
        builder = ErrorResponseBuilder(422, 'schema_validation_error')
        messages = self.normalized_messages()
        for field, messages in flatten(messages).items():
            if field == '_schema':
                for message in messages:
                    builder.add_common_description(message)
            else:
                for message in messages:
                    builder.add_field_description(field, message)
        return builder.build(i18n_localizer)


class Enum(Field):
    """
    Поле схемы для представления Enum'ов. Умеет сериализовать/десериализовать
    по имени и значению элементов.
    """

    default_error_messages = {
        '__invalid': 'Must be one of: {choice}'
    }

    def __init__(self, enum_cls, *args, **kwargs):
        super(Enum, self).__init__(*args, **kwargs)
        self.enum_cls = enum_cls
        self.__name_choice = ', '.join(map(
            repr,
            self.enum_cls.__members__,
        ))
        self.__value_choice = ', '.join(map(
            repr,
            (v.value for v in list(self.enum_cls.__members__.values()))
        ))
        self.by_name = kwargs.get('by_name', False)

    def _serialize(self, value, attr, obj, **kwargs):
        try:
            value = self.enum_cls(value)
        except ValueError:
            if self.by_name:
                self.fail('__invalid', choice=self.__name_choice)
            else:
                self.fail('__invalid', choice=self.__value_choice)

        return value.name if self.by_name else value.value

    def _deserialize(self, value, attr, data, **kwargs):
        if self.by_name:
            try:
                return getattr(self.enum_cls, value)
            except AttributeError:
                self.fail('__invalid', choice=self.__name_choice)
        else:
            try:
                return self.enum_cls(value)
            except ValueError:
                self.fail('__invalid', choice=self.__value_choice)


class TypedDict(Field):
    """
    Поле схемы для представления словарей с фиксироваными типами ключа и
    элемента.
    """

    default_error_messages = {
        'invalid': 'Not a valid mapping type.'
    }

    def __init__(self, item_field, key_field=String, *args, **kwargs):
        super(TypedDict, self).__init__(*args, **kwargs)
        if isinstance(item_field, type):
            if not issubclass(item_field, FieldABC):
                raise ValueError('The type of the dict values '
                                 'must be a subclass of '
                                 'marshmallow.base.FieldABC')
            self.item_field = item_field()  # type: Field
        else:
            if not isinstance(item_field, FieldABC):
                raise ValueError('The instances of the dict '
                                 'values must be of type '
                                 'marshmallow.base.FieldABC')
            self.item_field = item_field  # type: Field

        self.item_field.attribute = None

        if isinstance(key_field, type):
            if not issubclass(key_field, FieldABC):
                raise ValueError('The type of the dict keys '
                                 'must be a subclass of '
                                 'marshmallow.base.FieldABC')
            self.key_field = key_field()  # type: Field
        else:
            if not isinstance(key_field, FieldABC):
                raise ValueError('The instances of the dict '
                                 'keys must be of type '
                                 'marshmallow.base.FieldABC')
            self.key_field = key_field  # type: Field

        self.key_field.attribute = None

    def get_value(self, attr, obj, accessor=None, default=missing_):
        """Return the value for a given key from an object."""
        value = super(TypedDict, self).get_value(
            attr,
            obj,
            accessor=accessor,
            default=default,
        )
        if isinstance(value, Mapping):
            return {key: item for key, item in value.items()}
        else:
            return value

    def _add_to_schema(self, field_name, schema):
        super(TypedDict, self)._add_to_schema(field_name, schema)

        self.item_field = deepcopy(self.item_field)
        self.item_field.parent = self
        self.item_field.name = field_name

        self.key_field = deepcopy(self.key_field)
        self.key_field.parent = self
        self.key_field.name = field_name

    def _serialize(self, value, attr, obj, **kwargs):
        if value is None:
            return None

        if not isinstance(value, Mapping):
            self.fail('invalid')

        result = {}
        errors = defaultdict(dict)

        for key, item in value.items():
            skey = missing_
            try:
                skey = self.key_field._serialize(key, key, None)
            except ValidationError as e:
                errors[key].update({'_key': e.messages})

            sitem = missing_
            try:
                sitem = self.item_field._serialize(item, key, value)
            except ValidationError as e:
                errors[key].update({'_item': e.messages})

            if skey is not missing_ and sitem is not missing_:
                result[skey] = sitem

        if errors:
            raise ValidationError(dict(errors), data=result)

        return result

    def _deserialize(self, value, attr, data, **kwargs):
        if not isinstance(value, Mapping):
            self.fail('invalid')

        result = {}
        errors = defaultdict(dict)

        for key, item in value.items():
            dkey = missing_
            try:
                dkey = self.key_field.deserialize(key, key)
            except ValidationError as e:
                errors[key].update({'_key': e.messages})

            ditem = missing_
            try:
                ditem = self.item_field.deserialize(item, key, value)
            except ValidationError as e:
                errors[key].update({'_item': e.messages})

            if dkey is not missing_ and ditem is not missing_:
                result[dkey] = ditem

        if errors:
            raise ValidationError(dict(errors), data=result)

        return result
