# -*- coding: utf-8 -*-
import datetime
from retrying import retry
from io import StringIO
from inspect import isclass
from itertools import chain
from collections import namedtuple

from intranet.yandex_directory.src.yandex_directory.common.utils import (
    build_sql_limit_and_offset,
    covert_keys_with_dots_to_items,
    build_sql_where,
    build_order_by,
    split_by_comma,
    utcnow,
    ensure_date,
    pickle,
    unpickle,
    grouper,
)

from intranet.yandex_directory.src.yandex_directory.common.db import (
    mogrify,
    retry_if_db_integrity_error,
    Json,
)
from intranet.yandex_directory.src.yandex_directory.core.db import queries
from intranet.yandex_directory.src.yandex_directory.core.utils import only_fields


class CopyFrom(object):
    """
    Стратегия для BaseModel.bulk_create
    """
    pass


class Values(object):
    """
    Стратегия для BaseModel.bulk_create
    """
    do_nothing = 'do_nothing'  # вариант действий при конфликте в БД

    def __init__(self, on_conflict=None):
        self.on_conflict = on_conflict


# Это значение нужно передавать в Fields, чтобы
# явно указать, что нужны все поля объекта, даже
# те для которых нужен prefetch_related.
ALL_FIELDS = object()

# Это значение нужно передавать в Fields, чтобы
# явно указать, что нужны все простые поля объекта,
# для которых не нужен prefetch_related или select_related
ALL_SIMPLE_FIELDS = object()


# Специальное значение фильтра, чтобы сразу возвращать пустой результат
# и не выполнять операции в базе
FilterForEmptyResult = object()


class ConflictingArgumentsError(RuntimeError):
    pass


class UnknownFieldError(RuntimeError):
    """Это исключение выбрасывается моделями, если одно из полей, переданное
    в метод find, не поддерживается моделью. Или если вложенная сущность
    не поддерживает запрошенного поля.

    При этом, для вложенных сущностей поддерживается запрос только тех полей,
    которые не требуют джойнов или дополнительных запросов.
    """
    def __init__(self, model_class, field, supported_fields):
        self.model_class = model_class
        self.field = field
        self.supported_fields = supported_fields
        if model_class is None:
            message = 'Field "{0}" is not supported.'.format(field)
        else:
            message = 'Field "{0}" is not supported by {1}.'.format(
                field,
                model_class
            )
        super(UnknownFieldError, self).__init__(message)


Dependency = namedtuple('Dependency', ['name'])


def make_dependencies(fields):
    return list(map(Dependency, fields))


def split_name_from_operation(not_splitted_name):
    """Отделяет название поля от операции, и возвращает их как Tuple.
    В нашем "ORM" операция отделяется от имени поля двойным подчёркиванием.
    Операцией по-умолчанию считается equal.

    created_at -> (created_at, equal)
    created_at__lte -> (created_at, lte)
    """
    splitted = not_splitted_name.split('__', 1)
    name = splitted[0]
    if len(splitted) == 2:
        operation = splitted[1]
    else:
        operation = 'equal'
    return name, operation


def preprocess_fields(fields,
                      nested_fields=None,
                      result=None,
                      primary_key='id',
                      all_fields=None,
                      simple_fields=None,
                      field_dependencies=None,
                      model_class=None):
    """Превращает список полей в словарь, годный для использования в only_hierarchical_fields.

    В nested_fields может быть передан set из полей, которые
    представляют собой вложенные объекты.

    Аргумент result при обычном использовании передавать не надо, так как
    он используется только внутри, при рекурсивных вызовах.

    При этом:

    - если в списке отсутствует id, то он туда добавляется.
    - если присутствует поле, которое соответствует связанному объекту, то
      по-умолчанию, для такого объекта выбирается только поле id

    Например:

    ['nickname'] -> {'id': True, 'nickname': True}
    ['department'] -> {'id': True, 'department': {'id': True}}
    ['department.name'] -> {'id': True, 'department': {'id': True, 'name': True}}
    ['nickname', 'department.name', 'department.parent_id'] ->
    {
        'id': True,
        'nickname': True,
        'department': {
            'id: True,
            'name': True,
            'parent_id': True,
        }
    }
    """

    # id (primary_key) мы отдаём всегда, поэтому он в любом случае
    # должен быть в списке полей.
    if result is None:
        if isinstance(primary_key, (list, tuple)):
            result = {
                field: True
                for field in primary_key
            }
        else:
            result = {primary_key: True}

    if nested_fields is None:
        nested_fields = {}

    def split_if_needed(field):
        if isinstance(field, str):
            return tuple(field.split('.'))
        else:
            return field

    # Каждое поле должно быть представлено в виде списка с одним или
    # более элементов. Больше одного элемента будет в том случае, если
    # запрошено раскрытие вложенных объектов.
    fields = list(map(split_if_needed, fields))

    def expand_stars(field):
        """Возвращает список с полями.
        Поле подаваемое на вход - тоже является списком.
        Пример входов/выходов:

        [id] -> [[id]]
        [department, name] -> [[department, name]]
        [department, *] -> [[department, *]]
        [*] -> [[id], [nickname], [name], ...]]
        [**] -> [[id], [nickname], [name], [department, *], ...]
        [*, name] -> raise Exception(такое не поддерживается)
        [**, name] -> raise Exception(такое не поддерживается)
        """
        first = field[0]
        if first[0] == '*' and len(field) > 1:
            raise AssertionError('Stars are supported only at the end of a field specification.')

        if first == '*':
            return ((item,) for item in simple_fields)

        if first == '**':
            return (
                (item, '*')
                for item in all_fields
            )

        return (field,)

    # Дальше, надо развернуть все * и ** в списки поддерживаемых
    # моделью полей

    fields = list(chain(
            *list(map(expand_stars, fields))
        )
    )

    # Сначала проверим, что поле поддерживается
    if all_fields is None:
        raise AssertionError('all_fields parameter is required')

    # Если у полей есть зависимости, то сначала надо их раскрыть
    # Будем помечать зависимые поля, оборачивая в Dependency.

    if field_dependencies:
        for field in fields:
            # Так как поля на данном этапе это таплы, то
            # в качестве имени надо брать первый элемент
            field = field[0]

            if field in field_dependencies:
                dependencies = field_dependencies[field]

                # Нормализуем список зависимостей
                if isinstance(dependencies, str):
                    # если зависимость - одно поле, то можно указать только его,
                    # как обычную строку
                    fields.append(make_dependencies(split_if_needed(dependencies)))
                elif isinstance(dependencies, (list, tuple)):
                    # Если это список полей, то добавим эти поля в список
                    for dependency in dependencies:
                        fields.append(make_dependencies(split_if_needed(dependency)))


    for field in fields:
        rest_field_parts = field[1:]
        field = field[0]

        is_dependent = isinstance(field, Dependency)
        if is_dependent:
            field = field.name

        if field not in all_fields:
            raise UnknownFieldError(model_class, field, all_fields)

        if field in nested_fields:
            model = get_model(nested_fields[field])
            if model is None:
                # Если модель не указана, значит у этого вложенного поля
                # не может быть своих собственных вложенных полей
                # поэтому зарегистрируем его, как обычное поле

                # Если поле зависимое или ещё не добавлено в result,
                # то в result записываем True/False если поле независимое/зависимое
                # Иначе поле уже записано как независимое и мы его не трогаем.
                if not result.get(field, False):
                    result[field] = not is_dependent
            else:
                nested_primary_key = model.primary_key
                nested_simple_fields = model.simple_fields
                nested_all_fields = model.all_fields
                nested_nested_fields = model.nested_fields
                nested_field_dependencies = model.field_dependencies
                nested_model_class = model

                result.setdefault(field, {nested_primary_key: not is_dependent})
                # если указаны вложенные поля, то начинаем
                # наполнять словарь рекурсивно,
                # только для вложенных объектов мы не поддерживаем
                # указание дополнительных полей.
                # То есть, максимум на два уровня можно указывать поля
                if rest_field_parts:
                    preprocess_fields(
                        [rest_field_parts],
                        result=result[field],
                        primary_key=nested_primary_key,
                        nested_fields=nested_nested_fields,
                        all_fields=nested_all_fields,
                        simple_fields=nested_simple_fields,
                        model_class=nested_model_class,
                        field_dependencies=nested_field_dependencies,
                    )
        else:
            if not result.get(field, False):
                result[field] = not is_dependent

    return result


# Этот словарь заполняется с помощью MetaModel и содержит map из имени
# каждой модели в её класс. Это нужно, чтобы зависимости между select_related
# и prefertch_related можно было указывать как простые строки
_model_registry = {}


def get_model(model_or_class_name):
    """Принимает класс модели или её название и возвращает
    класс.
    """
    # В случае, если для related поля нет модели, в эту функцию
    # может быть передан None.
    if model_or_class_name:
        if isclass(model_or_class_name) \
           and issubclass(model_or_class_name, (BaseModel, PseudoModel)):
            return model_or_class_name
        else:
            if model_or_class_name in _model_registry:
                return _model_registry[model_or_class_name]
            else:
                raise RuntimeError('Unknown model "{0}"'.format(model_or_class_name))


class ModelMetaclass(type):
    def __new__(cls, class_name, bases, attrs):
        attrs = attrs.copy()

        required_class_attributes = (
            'db_alias',
            'all_fields',
        )

        for field_name in required_class_attributes:
            if field_name not in attrs:
                raise AssertionError(
                    '"{0}" attribute for model "{1}" is required'.format(
                        field_name,
                        class_name,
                    )
                )

        attrs['default_all_projection'] = '{0}.*'.format(attrs['table'])

        attrs['all_fields'] = set(attrs['all_fields'])
        all_fields = attrs['all_fields']
        prefetch_related_fields = attrs.get('prefetch_related_fields', {})
        select_related_fields = attrs.get('select_related_fields', {})

        assert isinstance(prefetch_related_fields, dict), 'self.prefetch_related_fields should be a dict'
        assert isinstance(select_related_fields, dict), 'self.select_related_fields should be a dict'

        # Составим единый словарь, представляющий собой
        # поля, с вложенными объектами
        if 'nested_fields' not in attrs:
            attrs['nested_fields'] = dict(
                select_related_fields,
                **prefetch_related_fields
            )

        for name in attrs['nested_fields']:
            if name not in all_fields:
                raise AssertionError('Поле "{0}" должно быть в атрибуте all_fields'.format(name))

        # Посчитаем, какие поля не требуют дополнительных запросов,
        # если модель не указывает этого явно. Явное указание используется
        # в фейковых моделях, которые просто перечисляют поля, доступные
        # во вложенном объекте. для примера, смотри класс group.GroupMember
        if 'simple_fields' not in attrs:
            attrs['simple_fields'] = set(
                name
                for name in all_fields
                if name not in prefetch_related_fields
               and name not in select_related_fields
            )

        result = super(ModelMetaclass, cls).__new__(cls, class_name, bases, attrs)
        _model_registry[class_name] = result
        return result


class PseudoModelMetaclass(type):
    def __new__(cls, class_name, bases, attrs):
        result = super(PseudoModelMetaclass, cls).__new__(cls, class_name, bases, attrs)
        _model_registry[class_name] = result
        return result


class PseudoModel(object, metaclass=PseudoModelMetaclass):
    """Класс для объектов, мимикрирующих под модель тем, что содержат
    описания полей вложенных объектов, но не имеют представления в базе.
    """
    primary_key = None
    simple_fields = []
    all_fields = []
    nested_fields = {}
    prefetch_related_fields = {}
    select_related_fields = {}
    field_dependencies = {}


class Query(object):
    def __init__(self, model_class, filter_data=None, fields=None, distinct=False, order_by=None, limit=None):
        self._model_class = model_class
        self._filter_data = filter_data or {}
        self._fields = fields or ()
        self._distinct = distinct
        self._limit = limit
        self._order_by = order_by

    def one(self):
        if not hasattr(self, '_result_one'):
            self._result_one = self._model_class.find(
                filter_data=self._filter_data,
                fields=self._fields,
                distinct=self._distinct,
                order_by=self._order_by,
                one=True,
            )
        return self._result_one

    def all(self):
        if not hasattr(self, '_result_all'):
            self._result_all = self._model_class.find(
                filter_data=self._filter_data,
                fields=self._fields,
                distinct=self._distinct,
                order_by=self._order_by,
                limit=self._limit,
            )
        return self._result_all

    def count(self):
        if not hasattr(self, '_result_count'):
            self._result_count = self._model_class.count(
                self._filter_data,
            )
        return self._result_count

    def filter(self, **kwargs):
        return Query(
            self._model_class,
            filter_data=dict(self._filter_data, **kwargs),
            fields=self._fields,
            distinct=self._distinct,
            order_by=self._order_by,
            limit=self._limit,
        )

    def fields(self, *fields):
        return Query(
            self._model_class,
            filter_data=self._filter_data,
            fields=self._fields + fields,
            distinct=self._distinct,
            order_by=self._order_by,
            limit=self._limit,
        )

    def limit(self, limit=None):
        return Query(
            self._model_class,
            filter_data=self._filter_data,
            fields=self._fields,
            distinct=self._distinct,
            order_by=self._order_by,
            limit=limit,
        )

    def scalar(self, *fields):
        # Чтобы избежать циклического импорта
        from intranet.yandex_directory.src.yandex_directory.core.utils import only_attrs

        fields = fields or self._fields
        if len(fields) != 1:
            raise RuntimeError(
                'Please set one field. You can use .fields() method '
                'or pass one parameter to .scalar() method.'
            )

        if not hasattr(self, '_scalar_all'):
            field = fields[0]
            result = self._model_class.find(
                filter_data=self._filter_data,
                fields=self._fields,
                distinct=self._distinct,
                order_by=self._order_by,
                limit=self._limit,
            )
            self._scalar_all = only_attrs(result, field)
        return self._scalar_all

    def distinct(self):
        return Query(
            self._model_class,
            filter_data=self._filter_data,
            fields=self._fields,
            distinct=True,
            order_by=self._order_by,
            limit=self._limit,
        )

    def order_by(self, order_by=None):
        return Query(
            self._model_class,
            filter_data=self._filter_data,
            fields=self._fields,
            distinct=self._distinct,
            order_by=order_by,
            limit=self._limit,
        )

    def update(self, **kwargs):
        return self._model_class.update(
            kwargs,
            filter_data=self._filter_data,
        )

    def delete(self, force_remove_all=False):
        return self._model_class.delete(
            filter_data=self._filter_data,
            # На всякий случай с помощью is защитимся от передачи
            # в качестве параметра какого-нибудь объкта, например словаря:
            # In [1]: {'foo': 'bar'} is True
            # Out[1]: False
            force_remove_all=force_remove_all is True,
        )

    def __iter__(self):
        return iter(self.all())

    def __len__(self):
        return self.count()

    def __getitem__(self, idx):
        return self.all()[idx]


class BaseModel(object, metaclass=ModelMetaclass):
    """Концепция:
    - Модель - не типичное представление ORM класса, а просто сахар над utils методами определенной сущности
    - Манипулирование сущностями должно осуществляться с помощью простых питонячьих типов. Пример:

    >>> print UserModel(connection=some_connection).create(migration_id = 'gena',status = null)
    {
        'name': 'gena',
        'id': 10
    }
    >>> print UserModel(connection=some_connection).update(id=10, data={'name': 'vova'})
    {
        'name': 'vova',
        'id': 10
    }
    >>> UserModel(connection=some_connection).delete(id=10)

    """

    db_alias = None
    table = None

    # список всех полей модели, который используется в поле fields
    all_fields = []
    # список полей, которые хранятся, как JSON
    # их нужно указать отдельно, чтобы при сериализации/десериализации в базу
    # они были правильно обработаны
    json_fields = []

    # то же самое, что и json поля, но для сериализации используется pickle
    pickle_fields = []
    # список полей, у которых в базе тип DATE, они принудительно конвертируются
    # в datetime.date из datetime.datetime, но перед этим приводятся к UTC.
    # Это нужно для того, чтобы операции сравнения были всегда корректны.
    date_fields = []
    # словарь, name -> SomeModel
    select_related_fields = {}
    # словарь, name -> SomeModel
    prefetch_related_fields = {}
    # Словарь с зависимостями полей.
    # Ключом является название поля, значением - список полей,
    # от которых оно зависит. Если при запросе какое-то из зависимых
    # полей не было указано, оно всё равно будет запрошено из базы.
    # Обычно полезно указывать зависимость от какого-то поля, требуемого
    # для заполнения prefetch_related поля.
    field_dependencies = {}

    order_by = 'id'
    primary_key = 'id'

    def __init__(self, connection):
        self._connection = connection

    def _check_used_filters(self, used_filters, filter_data):
        not_used_filters = set(list(filter_data.keys()) if filter_data else []) - set(used_filters)
        if not_used_filters:
            raise NotImplementedError(
                '%s filters have not been used while filtering' % ','.join(['"%s"' % i for i in not_used_filters])
            )

    def prepare_value_for_db(self, field, value):
        field_name, operation = split_name_from_operation(field)

        if field_name in self.json_fields:
            return Json(value)
        if field_name in self.pickle_fields:
            return pickle(value)
        elif field_name in self.date_fields:
            if value is not None:
                # Некоторые модели позволяют в качестве значения передавать несколько полей
                # в этом случае, трансформацию надо применить к ним всем
                if isinstance(value, (list, tuple)):
                    return list(map(ensure_date, value))
                else:
                    return ensure_date(value)
        else:
            return value

    def insert_into_db(self, **kwargs):
        """Этот хелпер создаёт в табличке, которая используется моделью,
        запись с указанными в kwargs полями.

        У нас почему-то есть другой метод insert, который просто любой запрос
        принимает, и при этом не прогоняет параметры через prepare,
        но зато пытается ретраить запросы в случае IntegrityError.
        Я пока не стал с ними разбираться, но по хорошему, стоит их когда-нибудь
        объединить. (art@)
        """
        # список ключей нужен для того, чтобы порядок полей был один и тот же
        keys = list(kwargs)
        fields = ','.join(keys)
        placeholders = ','.join(map('%({0})s'.format, keys))

        query = """
          INSERT INTO {table} ({fields}) VALUES ({placeholders})
          RETURNING *
        """.format(
            table=self.table,
            fields=fields,
            placeholders=placeholders,
        )

        result = self._connection.execute(
            query,
            self.prepare_dict_for_db(kwargs)
        ).fetchone()
        return self.prepare_dict_from_db(dict(result))

    def mogrify(self, query, vars=None):
        if isinstance(vars, dict):
            for key, value in vars.items():
                if isinstance(value, str):
                    vars[key] = value.strip().replace('%', '%%')

        return mogrify(
            connection=self._connection,
            query=query,
            vars=vars
        )

    def find(self,
             filter_data=None,
             skip=None,
             limit=None,
             order_by=None,
             extra_projections=None,
             one=False,
             distinct=False,
             fields=None,
             for_update=False,
             ):
        """
        Метод выборки данных из БД
        Нужно учесть, что при явном указании only_projections из базы будут выбраны только указанные поля
        Args:
            filter_data: dict - словарь с данными для фильтра
            skip: int - OFFSET для конечного запроса
            limit: int - LIMIT для конечного запроса
            order_by: поля для сортировки
            extra_projections: дополнительные поля выборки (подставляющиеся в SELECT {projections} FROM {table} ...)
            one: bool - При True возвращается не список, а первое значение из него
            distinct: bool - DISTINCT для запроса
            fields: list - список полей, которые должны содержать запрашиваемые объекты
            for_update: bool - FOR UPDATE для запроса
        """
        if not self.table:
            raise NotImplementedError('Table name for model "%s" should be provided' % type(self).__name__)
        if filter_data is FilterForEmptyResult:
            return []

        if one:
            limit = 1

        if not fields:
            # По умолчанию, отдаём только простые поля
            fields = set(self.simple_fields)
        elif fields is ALL_FIELDS:
            fields = set(self.all_fields)
        elif fields is ALL_SIMPLE_FIELDS:
            fields = set(self.simple_fields)

        if fields:
            # Если поля являются словарём, то предполагаем, что
            # они уже были прогнанны через препроцессинг
            if not isinstance(fields, dict):
                fields = preprocess_fields(
                    fields,
                    # Объединим два словаря в один новый
                    nested_fields=self.nested_fields,
                    simple_fields=self.simple_fields,
                    all_fields=self.all_fields,
                    primary_key=self.primary_key,
                    model_class=self.__class__,
                    field_dependencies=self.field_dependencies,
                )

            only_projections, select_related, prefetch_related = self.explain_fields(fields)

        if order_by is None and one is False:
            # Дефолтную сортировку мы не применяем, если пользователь модели
            # указал, что хочет получить только один объект. Поэтому здесь
            # в условии указано one is False.
            order_by = getattr(self, 'order_by', None)

        if extra_projections is None:
            extra_projections = set()

        filter_data = filter_data or {}
        projections, joins, fields_processors = self.get_select_related_data(select_related)
        if isinstance(order_by, str):
            order_by = split_by_comma(order_by)
        if only_projections:
            projections = set(only_projections).union(set(projections))

            # если нужно получить только определенные поля,
            # а в projections уже есть default_all_projection,
            # то нужно удалить её оттуда
            if self.default_all_projection in projections:
                projections.remove(self.default_all_projection)
        else:
            projections = set(projections).union(set(extra_projections))
        # тут model_distinct может выставляться в True самой моделью,
        # в методе get_filters_data
        model_distinct, filters, joins_from_filters = self.get_filters_and_check(
            filter_data=filter_data
        )

        joins += joins_from_filters

        parts = (
            '\n'.join(joins),
            build_sql_where(filters),
            build_order_by(order_by),
            build_sql_limit_and_offset(limit=limit, offset=skip),
        )
        # отфильтровываем пустые части запроса, чтобы SQL получился покрасивей
        parts = [_f for _f in parts if _f]

        distinct = distinct or model_distinct
        if order_by:
            order_by = [x[1:] if x.startswith('-') else x for x in order_by]
            projections = projections.union(set(i for i in map(self._build_model_field, order_by) if i is not None))

        # сортируем поля для выборки чтобы они всегда шли в одном и том же порядке
        # для удобства просмотра в логах и тестирования
        projections = list(projections)
        projections.sort()

        query = """
SELECT {distinct}{projections}
FROM {table}
{parts}{for_update}"""
        query = query.format(
            table=self.table,
            projections= ',\n       '.join(projections),
            parts='       \n'.join(parts),
            distinct='DISTINCT ' if distinct else '',
            for_update=' FOR UPDATE ' if for_update else '',
        )
        query = query.strip()
        conn = self._connection
        rows = conn.execute(query).fetchall()
        response = []

        for row in rows:
            row = dict(row)
            obj = covert_keys_with_dots_to_items(row)

            # Теперь зануллим те nested поля, у которых id == None
            # https://st.yandex-team.ru/DIR-3249
            for key, model in list(self.nested_fields.items()):
                model = get_model(model)
                if key in obj and \
                   isinstance(obj[key], dict) \
                   and getattr(model, 'primary_key', None) \
                   and obj[key][model.primary_key] is None:
                    obj[key] = None

            self.prepare_dict_from_db(obj)

            # Процессоры полей используются тогда, когда какое-то
            # из select_related полей опционально и может отсутствовать.
            # тогда в результате JOIN мы из базы достанем NULL для всех полей
            # вложенного объекта.
            # Обычно процессор должен посмотреть на то, что department_id == None
            # и установить атрибут department в None вместо словаря заполненного нонами.
            # Такие типовые процессоры лучше всего генерить с помощью
            # хелперов типа:
            # set_to_none_if_other_field_is_none или set_to_none_if_no_id.
            for process in fields_processors:
                obj = process(obj)

            response.append(obj)
        self.prefetch_related(response, prefetch_related)

        if fields is not None:
            # Чтобы избежать циклического импорта
            from intranet.yandex_directory.src.yandex_directory.core.utils import (
                only_hierarchical_fields,
            )

            # если были запрошены определённые поля, то надо вернуть только их
            response = only_hierarchical_fields(response, fields)

        # если задан ключ one=True, то вернем не список,
        # а первое значение из него
        if one:
            if response:
                response = response[0]
            else:
                response = None

        return response

    def count(self, filter_data=None, distinct_field=None):
        """
        distinct_field указываем, если хотим посчитать количество уникальных значений по заданной колонке
        можно передать с именем таблицы users.id, если колонка получается в результате join из фильтра, или
        просто user_id, тогда таблица берется текущая
        """
        if not self.table:
            raise NotImplementedError('Table name for model "%s" should be provided' % type(self).__name__)
        if filter_data is FilterForEmptyResult:
            return 0

        distinct, filters, joins = self.get_filters_and_check(filter_data=filter_data)

        if distinct_field:
            distinct = True
            distinct_filter = distinct_field if '.' in distinct_field else '{}.{}'.format(self.table, distinct_field)
        else:
            # так как primary key может быть составным, то это надо тут обработать
            if isinstance(self.primary_key, (tuple, list)):
                distinct_filter = ', '.join(
                    '{}.{}'.format(self.table, field)
                    for field in self.primary_key
                )
            else:
                distinct_filter = '{}.{}'.format(self.table, self.primary_key)


        if distinct:
            count_fields = 'DISTINCT ({})'.format(distinct_filter)
        else:
            count_fields = '*'

        query = """
        SELECT count({fields}) FROM {table}
        %(joins)s
        %(filters)s
        """.format(
            fields=count_fields,
            table=self.table,
        )
        query = query % {
            'filters': build_sql_where(filters),
            'joins': '\n'.join(joins)
        }
        return dict(
            self._connection.execute(query).fetchone()
        ).get('count')

    def get_select_related_data(self, select_related):
        return [self.default_all_projection], [], []

    def explain_fields(self, fields):
        """Принимает словарь с названиями полей и возвращает tuple:

        (projections, select_related, prefetch_related)

        Первый элемент должен содержать список полей, которые нужно
        запросить непосредственно из таблички.
        Второй - список того, что нужно заджойнить, чтобы отдать
        некоторые из указанных полей.
        Третий - список того, что нужно подтянуть дополнительными запросами.

        Как правило, если какое-то из полей представляет собой связанный
        объект, то это поле нужно убрать из списка полей и добавить в
        select_related список. А если это не один объект, а несколько, то
        в prefetch_related.

        Внимание! Не надо тут модифицировать входные данные!
        """
        projections = set()
        select_related = {}
        prefetch_related = {}

        supported_fields = set(self.all_fields)


        # Так как на вход у нас должны поступать поля, отданные функцией
        # preprocess_fields, то в переменной fields мы ожидаем словарь с
        # полями и тут в nested_fields должен попадать словарь с полями
        # вложенного объекта, или просто True
        for field, nested_fields in list(fields.items()):
            if field in self.select_related_fields:
                select_related[field] = nested_fields
            elif field in self.prefetch_related_fields:
                prefetch_related[field] = nested_fields
            else:
                projections.add(field)

        # Приведём все "проекции" к виду "табличка.поле"
        projections = list(map(self._build_model_field, projections))
        return projections, select_related, prefetch_related

    def prefetch_related(self, items, prefetch_related):
        pass

    def get_filters_data(self, filter_data):
        """Этот метод должен возвращать tuple:
        (
            distinct - True/False,
            filters - список с кусочками SQL для WHERE,
            joins -  список с кусочками запроса типа LEFT JOIN blah ON (minor.org_id == blah.org_id),
            used_filters - список строк с обработанными ключами из входного словаря filter_data,
        )
        """
        raise NotImplementedError

    def get_filters_and_check(self, filter_data):
        if filter_data:
            # Если заданы какие-то поля, то преобразуем типы значений
            # к тем, что используются в БД, например Json или DATE
            filter_data = self.prepare_dict_for_db(filter_data)

        distinct, filters, joins, used_filters = self.get_filters_data(filter_data)
        self._check_used_filters(used_filters, filter_data)
        return distinct, filters, joins

    def update(self, update_data, filter_data=None):
        if not self.table:
            raise NotImplementedError('Table name for model "%s" should be provided' % type(self).__name__)

        if not update_data:
            raise ValueError('Update data should be provided')
        if filter_data is FilterForEmptyResult:
            return {}

        distinct, filters, joins = self.get_filters_and_check(filter_data=filter_data)

        query = """
        UPDATE {table}
        SET %(set_data)s
        %(joins)s
        %(filters)s
        RETURNING *
        """.format(table=self.table) % {
            'set_data': self.get_update_set_data_params(update_data),
            'filters': build_sql_where(filters),
            'joins': '\n'.join(joins)
        }
        set_data_kwargs = self.prepare_dict_for_db(update_data)
        connection = self._connection
        result = connection.execute(query, set_data_kwargs).fetchone()
        if result is not None:
            return dict(result)

    def get_update_set_data_params(self, update_data):
        return ', '.join(
            ['{key} = %({key})s'.format(key=key) for key in update_data]
        )

    def prepare_dict_for_db(self, data):
        return dict([
            (
                key,
                self.prepare_value_for_db(key, value)
            )
            for key, value in list(data.items())
        ])

    def prepare_dict_from_db(self, data):
        """Расшифровывает некоторые поля, вынутые из базы.
           Меняет входной словарь, чтобы не делать лишних выделений памяти.
        """
        for key in self.pickle_fields:
            if key in data:
                data[key] = unpickle(data[key])

        return data

    def delete(self, filter_data=None, force_remove_all=False):
        if not self.table:
            raise NotImplementedError('Table name for model "%s" should be provided' % type(self).__name__)
        if filter_data is FilterForEmptyResult:
            return

        if not filter_data and not force_remove_all:
            raise ValueError('filter_data is empty! To remove all data you must set force_remove_all=True')

        distinct, filters, joins = self.get_filters_and_check(filter_data=filter_data)
        query = """
        DELETE FROM {table}
        %(joins)s
        %(filters)s
        """.format(table=self.table) % {
            'filters': build_sql_where(filters),
            'joins': '\n'.join(joins)
        }
        self._connection.execute(query)

    def all(self):
        return self.find()

    def filter(self, **kwargs):
        return Query(self, filter_data=kwargs)

    def fields(self, *fields):
        return Query(self, fields=fields)

    def get(self, id, fields=None, for_update=False):
        return self.find(
            {self.primary_key: id},
            fields=fields,
            one=True,
            for_update=for_update,
        )

    def _build_model_field(self, field):
        if '.' not in field:
            return '%s.%s' % (self.table, field)
        return field

    def create(self, *args, **kwargs):
        raise NotImplementedError()

    def update_one(self, id):
        raise NotImplementedError()

    def delete_one(self, pk):
        self.delete(
            filter_data={self.primary_key: pk}
        )

    @retry(stop_max_attempt_number=queries.INSERT_RETRY_COUNT,
           retry_on_exception=retry_if_db_integrity_error)
    def insert(self, query, params):
        connection = self._connection
        with connection.begin_nested():
            prepared_params = self.prepare_dict_for_db(params)
            row = connection.execute(
                query,
                prepared_params,
            ).fetchone()
            return self.prepare_dict_from_db(dict(row))

    def bulk_create(self, data, strategy=CopyFrom(), batch_size=None):
        """
        Множественное добавление, вставка нескольних записей одним запросом.
        Результаты тестов на вставку серии данных разными способами.
        https://github.yandex-team.ru/gist/art/f7e9e3efdc844e6736b9
        Args:
            data(list of dict) - список словарей с данными,
            набор ключей в словарях должен быть одинаковым,
            strategy способ вставки  CopyFrom, Values,
            batch_size (int) - резмер маленьких пачек, на которые надо поделить большую
        """
        if not data:
            return

        if batch_size:
            batches = grouper(batch_size, data)
        else:
            batches = [data]

        connection = self._connection

        if type(strategy) == CopyFrom:
            for batch in batches:
                lines = ['\t'.join([str(n) for n in list(x.values())])
                              for x in batch]
                output = StringIO('\n'.join(lines))
                cursor = connection.connection.cursor()
                cursor.copy_from(
                    output,
                    self.table,
                    columns=list(data[0].keys())
                )
        elif type(strategy) == Values:
            columns = list(data[0].keys())
            unknown_columns = set(columns) - set(self.all_fields)
            if unknown_columns:
                raise ValueError('Unknown columns {} for table {}'.format(tuple(unknown_columns), self.table))

            # для вставляемых столбцов формируем шаблон для данных
            # (%(column1_name)s, %(column2_name)s)
            placeholder = ', '.join(map('%({})s'.format, columns))
            morgify_str = '({})'.format(placeholder)
            # подставляем реальные значения в шаблон (%(column1_name)s, %(column2_name)s)
            for batch in batches:
                values = ','.join(
                    [mogrify(connection, morgify_str, item) for item in batch]
                )
                if strategy.on_conflict == Values.do_nothing:
                    do_on_conflict = 'ON CONFLICT DO NOTHING'
                else:
                    do_on_conflict = ''

                query = "insert into {table} ({columns}) VALUES {{values}} {{on_conflict}}".format(
                    table=self.table,
                    columns=','.join(columns),
                )
                query = query.format(values=values, on_conflict=do_on_conflict)
                cursor = connection.connection.cursor()
                cursor.execute(query)
        else:
            ValueError('Unknown strategy {}'.format(strategy))

    def replace_email_field(self, org_id, master_domain):
        """
        Меняем email поле у всех сущностей c old_domain на master_domain
        """
        queries_map = {
            'users': queries.BULK_UPDATE_EMAIL_USER,
            'groups': queries.BULK_UPDATE_EMAIL_GROUP,
            'departments': queries.BULK_UPDATE_EMAIL_DEPARTMENT,
        }
        if self.table not in queries_map:
            raise ValueError('You can use this method only with users/groups/departments tables')

        data_patterns = {
            'org_id': org_id,
            'master_domain_pattern': '@'+master_domain,
        }
        self._connection.execute(
            queries_map[self.table]['query'],
            self.prepare_dict_for_db(data_patterns)
        ).fetchall()

    def filter_by(self, filter_data, filter_parts, used_filters):
        """Эта функция возвращает ещё одну, посредством
        которой легко делать фильтрацию по типовым полям модели:

        filter_by(filter_data, filter_parts, used_filters) \
            ('id', can_be_list=True) \
            ('name')

        Используй её в get_filters_data, чтобы не плодить копипасту.
        """

        operation_map = {
            'in': 'IN',
            'equal': '=',
            'notequal': '!=',
            'gt': '>',
            'gte': '>=',
            'lt': '<',
            'lte': '<=',
            'isnull': 'IS',
            'between': 'BETWEEN',
            'contains': '@>',
        }

        # Предварительно подготовим поля так, чтобы ключами были исключительно названия,
        # а в значениях были tuple (operation, value).
        def process_item(item):
            not_splitted_name = item[0]
            value = item[1]
            name, operation = split_name_from_operation(not_splitted_name)
            return (name, (operation, value, not_splitted_name))

        filter_data = dict(list(map(process_item, list(filter_data.items()))))

        def filter_field(field_name, can_be_list=False, array=False, cast=None, encode=None):
            if field_name in filter_data:
                operation, value, not_splitted_name = filter_data[field_name]
                if cast:
                    cast = '::' + cast
                else:
                    cast = ''

                operator = operation_map.get(operation)
                if operator is None:
                    raise RuntimeError('Operation "{0}" is not supported by ORM. Supported operations: {1}'.format(
                        operation,
                        ', '.join(operation_map),
                    ))

                if operator == 'IS':
                    value_is_null = 'NULL' if value else 'NOT NULL'
                    filter_parts.append(
                        self.mogrify(
                            '{table}.{field_name} IS {value_is_null}'.format(
                                field_name=field_name,
                                table=self.table,
                                value_is_null=value_is_null,
                            )
                        )
                    )
                elif operator == 'BETWEEN':
                    if not isinstance(value, tuple):
                        raise ValueError('Tuple is expected in this filter.')
                    if len(value) != 2:
                        raise ValueError('Tuple of two items required.')

                    filter_parts.append(
                        self.mogrify(
                            '{table}.{field_name} BETWEEN %(a)s{cast} AND %(b)s{cast}'.format(
                                field_name=field_name,
                                table=self.table,
                                cast=cast,
                            ),
                            {'a': value[0], 'b': value[1]}
                        )
                    )
                else:
                    # Здесь мы специально не поддерживаем любые iterable
                    # поскольку так можно случайно съесть значения из
                    # какого-нибудь итератора.
                    if can_be_list and isinstance(value, (list, tuple, set)):
                        value = tuple(value)
                        if not value:
                            raise ValueError('Non empty list is expected in this filter.')
                        if operation not in ['notequal', 'equal', 'in']:
                            raise RuntimeError('Unknown operation {} for list filter.'.format(operation))
                        operator = 'NOT IN' if operation == 'notequal' else 'IN'

                    # Если поле помечено, как ARRAY, то позволяем по нему
                    # делать поиск вхождений
                    elif array and operation == 'contains':
                        operator = '@>'
                        value = [value]

                    # Если задана функция для предварительной конвертации, то сделаем её
                    if encode:
                        value = encode(value)

                    filter_parts.append(
                        self.mogrify(
                            '{table}.{field_name}{cast} {operator} %({field_name})s{cast}'.format(
                                field_name=field_name,
                                operator=operator,
                                table=self.table,
                                cast=cast,
                            ),
                            {
                                field_name: value,
                            }
                        )
                    )
                used_filters.append(not_splitted_name)
            return filter_field
        return filter_field

    def _check_fields_come_together(self, fields_together, filter_data):
        """
        Вернуть True, если поля использованы в фильтрации строго вместе.
        :param fields_together: поля которые должны быть вместе
        :param filter_data: условие фильтрации
        :rtype: bool
        """
        return not any(
            field in filter_data for field in fields_together
        ) or all(
            field in filter_data for field in fields_together
        )

    def remove_private_fields(self, data):
        """Эта фунция удаляет из выдачи вспомогательные поля, которые
           есть в схеме, но не описаны в модели.

           Например в табличке организации есть поля для хранения
           последних id выданных для отделов и команд.
        """
        return only_fields(data, *self.all_fields)

    @classmethod
    def raise_on_prohibited_fields(cls, fields, prohibited):
        """В некоторых view может быть запрещено запрашивать определенные поля.
           В таких случаях надо использовать этот метод, чтобы выкинуть правильное
           исключение, как буд-то такого поля и в помине нет.

           * fields - список строк с перечислением полей запрошенных пользователем или None.
           * prohibited - список строк с полями, которые запрещены.
        """

        if fields:
            for field in prohibited:
                if field in fields:
                    allowed = set(cls.all_fields) - set(prohibited)
                    raise UnknownFieldError(cls, field, allowed)



class BaseAnalyticsModel(BaseModel):
    """
    Модель для временного хранения аналитических данных перед отправкой в yt
    """
    db_alias = None
    table = None
    all_fields = []

    def save(self):
        raise NotImplementedError()

    def save_analytics(self):
        self.delete_old_data()
        self.save()

    def get_filters_data(self, filter_data):
        distinct = False

        if not filter_data:
            return distinct, [], [], []

        filter_parts, joins, used_filters = [], [], []

        # дата хранится в строковом виде, чтобы упростить копирование данных в yt
        # поэтому приведение типов требуется в фильтре
        if 'for_date' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'for_date::date = %(for_date)s::date',
                    {
                        'for_date': filter_data.get('for_date')
                    }
                )
            )
            used_filters.append('for_date')

        if 'for_date__lte' in filter_data:
            filter_parts.append(
                self.mogrify(
                    'for_date::date <= %(for_date)s::date',
                    {
                        'for_date': filter_data.get('for_date__lte')
                    }
                )
            )
            used_filters.append('for_date__lte')

        return distinct, filter_parts, joins, used_filters

    def delete_old_data(self, days=5):
        """
        Удаляем все записи старше 5 дней
        """
        self.delete({'for_date__lte': utcnow() - datetime.timedelta(days=days)})


def set_to_none_if_other_field_is_none(field_to_remove, field_to_check):
    """Возвращает функцию, которая проверяет, есть ли в объекте
    значение отличное от None у поля field_to_check. И если нет, то
    устанавливает в None значение поля field_to_remove, если оно в объекте есть.

    Входной объект модифицируется и возвращается в качестве результата.
    """

    def field_nullificator(obj):
        if field_to_remove in obj and field_to_check in obj:
            if obj[field_to_check] is None:
                obj[field_to_remove] = None
        return obj

    return field_nullificator


def set_to_none_if_nested_field_is_none(field_to_remove, field_to_check):
    """Возвращает функцию, которая заменяет атрибут fields_to_remove на None,
    если значение этого атрибута - словарь, у которого поле field_to_check - None.

    Например, если объект:

    {
        department: {id: None, name: None}
    }
    fields_to_remove = 'department'
    fields_to_check = 'id'

    то на выходе объект будет:

    {
        department: None
    }

    Входной объект модифицируется и возвращается в качестве результата.
    """

    def field_nullificator(obj):
        if field_to_remove in obj:
            nested_obj = obj[field_to_remove]
            if isinstance(nested_obj, dict):
                if nested_obj[field_to_check] is None:
                    obj[field_to_remove] = None
        return obj

    return field_nullificator


def set_to_none_if_no_id(key, id_field='id'):
    """Вспомогательный генератор процессора для полей.

    Обычно занулить поля надо в том случае, если у вложенного объекта нет id.
    """
    return set_to_none_if_nested_field_is_none(key, id_field)
