# -*- coding: utf-8 -*-
import math
from operator import itemgetter

from copy import deepcopy

from intranet.yandex_directory.src.yandex_directory import app


class BasePaginator(object):
    def __init__(self,
                 total,
                 page,
                 path,
                 per_page=None,
                 max_per_page=None,
                 query_params=None,
                 multishard=False,
                 ):
        self.path = path
        self.total = total
        self.multishard = multishard
        self.query_params = query_params or {}
        self.per_page = self._build_per_page(per_page, max_per_page)
        self.pages = self._build_pages(self.total, self.per_page)
        self.page = self._build_page(page)
        self.response = {}

    def _build_page(self, page):
        raise NotImplementedError()

    def _build_per_page(self, per_page, max_per_page):
        if per_page is None:
            per_page = app.config['PAGINATION']['per_page']
            self.per_page_is_has_been_set = False
        else:
            self.per_page_is_has_been_set = True

        if max_per_page is None:
            max_per_page = app.config['PAGINATION']['max_per_page']

        # ограничиваем сверху значение, которое может быть передано в per_page
        return min(per_page, max_per_page)

    def _build_pages(self, total, per_page):
        return int(math.ceil(total / float(per_page)))

    def _build_url(self, **kwargs):
        from intranet.yandex_directory.src.yandex_directory.common.utils import url_join
        params = deepcopy(self.query_params)
        params.update(kwargs)

        if self.per_page_is_has_been_set:
            params['per_page'] = self.per_page

        return url_join(
            app.config['SITE_BASE_URI'],
            self.path,
            force_trailing_slash=True,
            query_params=params,
        )


class Paginator(BasePaginator):
    # todo: test me
    def __init__(self,
                 total,
                 page,
                 path,
                 per_page=None,
                 max_per_page=None,
                 query_params=None,
                 multishard=False,
                 is_last_page=False,
                ):
        super(Paginator, self).__init__(
            total,
            page,
            path,
            per_page,
            max_per_page,
            query_params,
            multishard,
        )

        self.links = self._build_links(
            page=self.page,
            pages=self.pages,
            is_last_page=is_last_page,
        )
        self.response['data'] = {
            'page': self.page,
            'per_page': self.per_page,
            'pages': self.pages,
            'total': self.total,
            'links': self.links,
            'multishard': self.multishard
        }
        self.response['headers'] = {
            'Link': get_link_header(self.links)
        }

    def _build_page(self, page):
        return page or 1

    def _build_pages(self, total, per_page):
        if total is None:
            return None
        return int(math.ceil(total / float(per_page)))

    def build_url(self, **kwargs):
        from intranet.yandex_directory.src.yandex_directory.common.utils import url_join
        params = deepcopy(self.query_params)
        params.update(kwargs)

        if self.per_page_is_has_been_set:
            params['per_page'] = self.per_page

        return url_join(
            app.config['SITE_BASE_URI'],
            self.path,
            force_trailing_slash=True,
            query_params=params,
        )

    def _build_links(self, page, pages, is_last_page):
        links = {}

        if pages is None:
            if not is_last_page:
                links['next'] = self.build_url(page=page + 1)
            if page > 1:
                links['first'] = self.build_url(page=1)
                links['prev'] = self.build_url(page=page - 1)

            return links

        if page < pages:
            links['next'] = self.build_url(page=page + 1)
            links['last'] = self.build_url(page=pages)

        if page > 1:
            links['prev'] = self.build_url(page=page - 1)
            links['first'] = self.build_url(page=1)

        return links


class KeySetPaginator(BasePaginator):
    reverse_response_param = 'reverse'

    def __init__(self,
                 total,
                 page,
                 path,
                 per_page,
                 max_per_page,
                 query_params,
                 model,
                 model_filters,
                 model_fields,
                 multishard=False,
                 ):
        from intranet.yandex_directory.src.yandex_directory.common.utils import _get_int_or_none

        self.reverse = _get_int_or_none(query_params.get(self.reverse_response_param))
        super(KeySetPaginator, self).__init__(
            total,
            page,
            path,
            per_page,
            max_per_page,
            query_params,
            multishard,
        )

        self.model = model
        self.model_filters = model_filters
        self.model_fields = model_fields
        self.order_by = model.order_by

        self.gt_key = '{}__gt'.format(self.order_by)
        self.lt_key = '{}__lt'.format(self.order_by)

        self.data = self._build_data()
        self.links = self._build_links()

        self.response['data'] = {
            'page': self.page,
            'per_page': self.per_page,
            'pages': self.pages,
            'total': self.total,
            'links': self.links,
            'multishard': self.multishard
        }
        self.response['data']['result'] = self.data
        self.response['headers'] = {
            'Link': get_link_header(self.links)
        }

    def _build_data(self):
        sql_query_params = dict(
            filter_data=self.model_filters,
            limit=self.per_page + 1,
            fields=self.model_fields,
        )
        if self.reverse:
            sql_query_params['order_by'] = '-{}'.format(self.order_by)
        return self.model.find(**sql_query_params)

    def _get_bounds(self):
        """
        Этот метод возвращает границы полученного среза данных.
        Границы нужны, чтобы получить ссылки на предыдущий и следующий срезы данных.
        Изначально мы запрашиваем срез длинной на 1 бльше, чем надо, чтобы понять, есть ли дальше данные.
        Если данные есть, и это не самый первый и не самый последний срезы - есть обе границы.
        Если данных меньше, чем мы запросили - значит, это первый или последний срез, и у него нет одной из границ

        Метод имеет side-effect.
        Он отрезает последний элемент self.data, если елементов больше, чем per_page
        (то есть есть ещё данные в ту или другую сторону)
        И сортирует в обратном порядке self.data, если reverse = True,
        чтобы вернуть данные в правильном порядке, по возрастанию
        (если мы делали запрос order by desc для получения предыдущей страницы)
        """
        left_bound = None
        right_bound = None
        if len(self.data) > self.per_page:
            self.data = self.data[:-1]
            if self.reverse:
                self.data.sort(key=itemgetter(self.order_by))
            left_bound = self.data[0][self.order_by]
            right_bound = self.data[-1][self.order_by]
        elif self.data:
            if self.reverse:
                self.data.sort(key=itemgetter(self.order_by))
                right_bound = self.data[-1][self.order_by]
            else:
                left_bound = self.data[0][self.order_by]

        first_or_last = True
        for key in [self.gt_key, self.lt_key]:
            if key in self.query_params:
                first_or_last = False
                break

        if first_or_last:
            if self.reverse:
                right_bound = None
            else:
                left_bound = None

        return left_bound, right_bound

    def _build_links(self):
        links = {}

        left_bound, right_bound = self._get_bounds()

        for key in [self.gt_key, self.lt_key, self.reverse_response_param]:
            if key in self.query_params:
                del self.query_params[key]

        if left_bound:
            lt_params = {
                self.lt_key: left_bound,
                self.reverse_response_param: 1,
                'page': self.page - 1,
                'total': self.total,
            }
            links['prev'] = self._build_url(**lt_params)
        if right_bound:
            gt_params = {
                self.gt_key: right_bound,
                'page': self.page + 1,
                'total': self.total
            }
            links['next'] = self._build_url(**gt_params)

        return links

    def _build_page(self, page):
        if not page:
            if self.reverse:
                page = self._build_pages(self.total, self.per_page)
            else:
                page = 1
        return page


def get_link_header(links):
    return ','.join([
        '<{url}>; rel="{rel}"'.format(url=value, rel=key)
        for key, value in list(links.items())
    ])
