# encoding: UTF-8



from abc import ABCMeta

from sqlalchemy.ext.declarative import DeclarativeMeta
from sqlalchemy.ext.mutable import MutableDict as OrigMutableDict
from sqlalchemy.orm import Query
from sqlalchemy.orm import Session
from typing import Any
from typing import Callable
from typing import Generic
from typing import Optional
from typing import Type
from typing import TypeVar

from intranet.yandex_directory.src.yandex_directory.common.web import ErrorResponseBuilder
from intranet.yandex_directory.src.yandex_directory.common.web import ResponseMixin


class ModelMeta(DeclarativeMeta, ABCMeta):
    """
    Базовый мета-класс для моделей с поддержкой ``abc``.
    """


class EntityNotFoundError(Exception, ResponseMixin):
    """
    Ошибка поиска сущности по ID.
    """

    def __init__(self, entity_id):
        super(EntityNotFoundError, self).__init__('Entity not found')
        self.entity_id = entity_id

    def as_response(self, i18n_localizer):
        return ErrorResponseBuilder(404, 'not_found') \
            .set_parameter('entity_id', self.entity_id) \
            .build(i18n_localizer)


T = TypeVar('T', bound=ModelMeta)


class CRUDRepository(Generic[T]):
    """
    Базовый класс для репозиториев. Умеет находить сущность по id,
    сохранять/удалять сущность.

    Предоставляет наследникам интерфейс для доступа к сессии или создании
    запросов.
    """

    def __init__(self, entity_cls, session_provider):
        # type: (Type[T], Callable[[], Session]) -> None
        self._entity_cls = entity_cls
        self._session_provider = session_provider

    @property
    def entity_cls(self):
        # type: () -> Type[T]
        return self._entity_cls

    def _session(self, session=None):
        # type: (Optional[Session]) -> Session
        if session is None:
            return self._session_provider()
        else:
            return session

    def _query(self, session=None):
        # type: (Optional[Session]) -> Query
        return self._session(session).query(self._entity_cls)

    def count(self, session=None):
        return self._query(session).count()

    def get(self, id, session=None):
        # type: (Any, Optional[Session]) -> T
        entity = self.get_or_none(id, session)
        if entity is None:
            raise EntityNotFoundError(id)
        else:
            return entity

    def get_or_none(self, id, session=None):
        # type: (Any, Optional[Session]) -> Optional[T]
        return self._query(session).get(id)

    def save(self, entity, session=None):
        # type: (T, Optional[Session]) -> T
        return self._session(session).merge(entity)

    def delete(self, entity, session=None):
        # type: (T, Optional[Session]) -> None
        self._session(session).delete(entity)


class MutableDict(OrigMutableDict):
    def pop(self, k, *args, **kwargs):
        result = dict.pop(self, k, *args, **kwargs)
        self.changed()
        return result
