# -*- coding: utf-8 -*-
import pickle as pickle_module
import string
import collections.abc
import datetime
import errno
import calendar
import importlib
import hashlib
import os
import re
import socket
import time
import urllib.request
import urllib.parse
import urllib.error
import pkg_resources
import math
import uuid
import typing
from intranet.yandex_directory.src import settings

from contextlib import contextmanager
from copy import copy
from functools import (
    wraps,
    cmp_to_key,
)
from operator import itemgetter
from itertools import zip_longest

import pytz
import requests
from concurrent.futures import (
    ThreadPoolExecutor,
    TimeoutError,
)
from flask import (
    Response,
    g,
    request,
)
from jsonschema import Draft4Validator
from jsonschema import validators
from jsonschema.exceptions import ValidationError
from requests import RequestException

from intranet.yandex_directory.src import blackbox_client
from . import json
import intranet.yandex_directory.src.yandex_directory as yandex_directory
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ImmediateReturn,
    DomainNotFound,
    InvalidDomain,
    OrganizationIsBlocked,
)
from intranet.yandex_directory.src.yandex_directory.common.pagination import (
    Paginator,
    KeySetPaginator,
)
from intranet.yandex_directory.src.yandex_directory.core.features import (
    MULTIORG,
    is_feature_enabled,
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log


class BlackboxWithTimings(blackbox_client.Blackbox):
    def _blackbox_call(self, *args, **kwargs):
        from intranet.yandex_directory.src.yandex_directory.directory_logging.http_requests_logger import \
            HttpRequestLogService

        log_record = HttpRequestLogService.start('GET', self.url, {})
        try:
            response = super(BlackboxWithTimings, self)._blackbox_call(*args, **kwargs)
            log_record.finish(None)
            return response
        except Exception as e:
            log_record.error(e)
            raise


class BlackboxProtocol(typing.Protocol):
    def userinfo(self, *args, **kwargs) -> dict:
        pass


class BlackboxRoutingProxy:
    def __init__(self, yandex_bb: blackbox_client.Blackbox, cloud_bb: BlackboxProtocol):
        self._yandex_bb = yandex_bb
        self._cloud_bb = cloud_bb

    def __getattr__(self, item):
        return getattr(self._yandex_bb, item)

    def userinfo(self, *args, use_cloud_bb=None, **kwargs):
        if use_cloud_bb is None:
            from intranet.yandex_directory.src.yandex_directory.core.utils import is_cloud_uid
            if 'uid' in kwargs and is_cloud_uid(kwargs['uid']):
                use_cloud_bb = True
            elif 'yc_subject' in kwargs:
                use_cloud_bb = True
            else:
                use_cloud_bb = False

        if use_cloud_bb:
            return self._cloud_bb.userinfo(*args, **kwargs)

        return self._yandex_bb.userinfo(*args, **kwargs)

    def batch_userinfo(self, *args, use_cloud_bb=None, **kwargs):
        if use_cloud_bb is None:
            if 'yc_subject' in kwargs:
                use_cloud_bb = True
            else:
                use_cloud_bb = False

        if use_cloud_bb:
            return self._cloud_bb.userinfo(*args, **kwargs)

        return self._yandex_bb.batch_userinfo(*args, **kwargs)


# специальный объект, который нужно использовать в качестве
# дефолта там, где может быть передан None
NotGiven = object()
Ignore = object()
NotChanged = object()


def dict_to_csv_string(dict_object):
    return ', '.join(map('{0[0]}={0[1]}'.format, list(dict_object.items())))


def _custom_required_validator(validator, required, instance, schema):
    """
    Этот валидатор проверяет, что для объектов все поля, которые перечисленны в параметре required,
    должны быть указаны и при этом не могут иметь пустые значения.
    """
    # todo: test me
    if not validator.is_type(instance, "object"):
        return
    for property in required:
        property_value = instance.get(property)  # check not only existence
        if isinstance(property_value, str):
            property_value = property_value.strip()  # вот тут добавить стрип
        if isinstance(property_value, int):
            continue
        if not property_value:
            yield ValidationError("%r is a required property" % property)


CustomDraft4Validator = validators.extend(
    validator=Draft4Validator,
    validators={
        'required': _custom_required_validator
    }
)


def raise_for_status(response):
    try:
        response.raise_for_status()
    except RequestException as e:
        e.args += (response.content, response.headers)
        raise


def get_localhost_ip_address():
    try:
        return socket.getaddrinfo(socket.getfqdn(), 80)[0][4][0]
    except socket.gaierror as exc:
        log.info('An error while getting ip address "%s", this is of ', exc)
        return '127.0.0.1'


def get_host_and_port_from_url(url):
    parsed = urllib.parse.urlparse(url)
    port = parsed.port
    if not port:
        if parsed.scheme == 'https':
            port = 443
        else:
            port = 80
    return parsed.hostname, port


def force_text(value):
    """Превращает значение в юникодную строку."""
    if isinstance(value, bytes):
        return value.decode('utf-8')
    elif isinstance(value, str):
        return value
    else:
        return str(value)


def force_utf8(value):
    """Превращает значение в utf8 строку.
    Удобно использовать для вывода чего-либо в файл или терминал."""
    if isinstance(value, bytes):
        return value
    elif isinstance(value, str):
        return value.encode('utf-8')
    else:
        return str(value)


def url_join(base, uri, force_trailing_slash=False, query_params=None):
    if query_params is None:
        query_params = {}
    # todo: test me
    url = urllib.parse.urljoin(base, uri)
    splitted = urllib.parse.urlparse(url)
    # url = '/'.join(part.strip('/') for part in parts if part)
    if force_trailing_slash and not splitted.path.endswith('/'):
        splitted = splitted._replace(path=splitted.path + '/')

    if query_params:
        # здесь мы намеряно используем parse_qsl и преобразуем его в dict,
        # чтобы отбросить возможные повторения параметров, а потом переопределить
        # некоторые из них с помощью query_params
        query_params = dict(urllib.parse.parse_qsl(splitted.query), **query_params)
        # а затем сортируем параметры по алфавиту, для удобства тестирования
        query_params = sorted(query_params.items())
        splitted = splitted._replace(query=urllib.parse.urlencode(query_params))

    return urllib.parse.urlunparse(splitted)


def build_error(error):
    return {
        'message': str(error)
    }


def build_multishard_list_response(model,
                                   path,
                                   query_params,
                                   model_filters=None,
                                   model_fields=None,
                                   prepare_result_item_func=None,
                                   max_per_page=None,
                                   order_by=None,
                                   distinct=False):
    # отдаем всегда только первую страницу результатов и сообщаем фронту,
    # в total_count записываем если есть еще результаты, чтобы фронт показал сообщения
    from intranet.yandex_directory.src.yandex_directory.common.db import get_shard_numbers, get_main_connection

    result = []
    per_page = _get_int_or_none(
        query_params.get(app.config['PAGINATION']['per_page_query_param'], app.config['PAGINATION']['per_page'])
    )
    prepare_result_item_func = prepare_result_item_func or (lambda x: x)
    for shard in get_shard_numbers():
        with get_main_connection(shard=shard) as main_connection:
            shard_result = model(main_connection).find(
                filter_data=model_filters,
                limit=per_page * 2,
                fields=model_fields,
                order_by=order_by,
                distinct=distinct,
            )
            for res in shard_result:
                res['shard'] = shard
            result.extend(list(map(prepare_result_item_func, shard_result)))

    paginator = Paginator(
        total=len(result),
        page=1,
        per_page=per_page,
        path=path,
        query_params=query_params,
        max_per_page=max_per_page,
        multishard=True,
    )
    if order_by:
        result = multikeysort(result, order_by)
    response = paginator.response
    response['data']['result'] = result[:per_page]
    return response


def build_fast_list_response(model,
                             path,
                             query_params,
                             model_filters=None,
                             model_fields=None,
                             prepare_result_item_func=None,
                             max_per_page=None,
                             ):
    """
    Тут используется keyset pagination вместо limit-offset
    (в случае, если сортировка происходит по умолчанию по primary key)
    """

    total = _get_int_or_none(query_params.get('total'))
    if not total:
        total = model.count(filter_data=model_filters)

    per_page = _get_int_or_none(query_params.get(app.config['PAGINATION']['per_page_query_param']))
    page = _get_int_or_none(query_params.get('page'))

    paginator = KeySetPaginator(
        total,
        page,
        path,
        per_page,
        max_per_page,
        query_params,
        model,
        model_filters,
        model_fields,
    )
    prepare_result_item_func = prepare_result_item_func or (lambda x: x)
    response = paginator.response
    response['data']['result'] = list(map(prepare_result_item_func, response['data']['result']))

    return response


def build_list_response(model,
                        path,
                        query_params,
                        model_filters=None,
                        # Ручки явно задают поля которые им нужны, в
                        # зависимости от версии API. Если поля None, то
                        # модель считает что нужны все simple поля.
                        model_fields=None,
                        prepare_result_item_func=None,
                        max_per_page=None,
                        order_by=None,
                        distinct=False,
                        preprocess_results=None):
    total_count = model.count(filter_data=model_filters)
    prepare_result_item_func = prepare_result_item_func or (lambda x: x)
    preprocess_results = preprocess_results or (lambda x: x)
    paginator = Paginator(
        total=total_count,
        page=_get_int_or_none(query_params.get(app.config['PAGINATION']['page_query_param'])),
        per_page=_get_int_or_none(query_params.get(app.config['PAGINATION']['per_page_query_param'])),
        path=path,
        query_params=query_params,
        max_per_page=max_per_page,
    )
    if paginator.page > 1:
        skip = (paginator.page - 1) * paginator.per_page
    else:
        skip = 0

    limit = paginator.per_page
    response = paginator.response
    objects = model.find(
        filter_data=model_filters,
        skip=skip,
        limit=limit,
        fields=model_fields,
        order_by=order_by,
        distinct=distinct,
    )
    objects = preprocess_results(objects)
    response['data']['result'] = list(map(
        prepare_result_item_func,
        objects,
    ))

    return response


def _get_int_or_none(value):
    try:
        return int(value)
    except (ValueError, TypeError):
        return None


def coerce_to_int_or_none(value, exc_cls, *args, **kwargs):
    """Приводит value к числу.

    Если value=None, то возвращается None.
    Если привести значение к числу невозможно, то бросается
    исключение переданного класса, которому в конструктор
    переданы параметры *args и **kwargs.
    """
    if value is not None:
        try:
            return int(value)
        except ValueError:
            raise exc_cls(*args, **kwargs)


def validate_data_by_schema(data, schema):
    errors = [build_error(i) for i in CustomDraft4Validator(schema).iter_errors(data)]
    if errors:
        with log.fields(schema_errors=', '.join(err['message'] for err in errors)):
            log.warning('Data is invalid for given schema')
    return errors


def split_by_comma(text):
    splitted = (text or '').split(',')
    splitted = (item.strip() for item in splitted)
    splitted = [_f for _f in splitted if _f]
    return splitted


def build_sql_where(parts):
    if parts:
        parts = list(map('({0})'.format, parts))
        parts = ' AND '.join(parts)
        return 'WHERE ' + parts
    else:
        return ''


def build_sql_limit_and_offset(limit, offset):
    parts = []
    if limit:
        parts.append("LIMIT %d" % limit)
    if offset:
        parts.append("OFFSET %d" % offset)
    return ' '.join(parts)


def build_order_by(fields):
    """Принимает контейнер из строк с названиями полей, например,  ['name', '-id']
       И возвращает ORDER BY name ASC, id DESC
    """
    result = ''
    if fields:
        def process_field(field):
            if field.startswith('-'):
                return '{0} DESC'.format(field[1:])
            return '{0} ASC'.format(field)

        fields = list(map(process_field, fields))
        result = 'ORDER BY ' + ', '.join(fields)

    return result


def covert_keys_with_dots_to_items(data_dict):
    result = {}
    # превратим ключи в tuples
    data = (
        (key.split('.'), value)
        for key, value in data_dict.items()
    )

    def get_subtree(dictionary, key):
        """Возвращает взятый по ключу словарь.
        Если в dictionary нет ключа, то он создаётся со значением {}.
        """
        assert dictionary.get(key) != 'not None', \
            'Надо убрать упоминание поля {0} в select_related'.format(key)
        dictionary.setdefault(key, {})
        return dictionary[key]

    for key_parts, value in data:
        subtree = result
        # Пройдём по дереву, создавая части ключа,
        # если это необходимо
        prefix_keys = key_parts[:-1]
        last_key = key_parts[-1]
        for part in prefix_keys:
            subtree = get_subtree(subtree, part)
        # Теперь в subtree - словарь, куда нам нужно положить значение.
        # Поэтому просто положим его туда:
        assert isinstance(subtree, dict), \
            'Надо убрать из запроса поле {0}, оно там лишнее'.format(
                '.'.join(prefix_keys)
            )
        subtree[last_key] = value

    return result


to_escape = ['>', '<', '|', '&', ':', '*', '!', '(', ')', '\'']
regexp_escape = re.compile(
    r'({0})'.format('|'.join(['\\' + character for character in to_escape]))
)


def prepare_for_query(string):
    """
    Подготовка строки к запросу в базу
    """
    # Заменяем % на два %
    string = re.sub(r'%', '%%', string.strip())
    # Экранируем спецсимволы
    string = regexp_escape.sub(r'\\\1', string)
    return string


def prepare_for_tsquery(string):
    """
    Подготовка текста поискового запроса
    """
    string = prepare_for_query(string)
    # Дописываем :* для поиска по префиксу и добавляем соединитель И(&)
    # между слов
    string = string.replace("'", "")
    string = '%s:*' % re.sub(r'\s+', ':*&', string)
    if '.' in string:
        string = '{}|{}'.format(string, string.replace('.', '-'))
    return string


def import_from_string(val, setting_name):
    """
    Attempt to import a class from a string representation.
    """
    try:
        parts = val.split('.')
        module_path, class_name = '.'.join(parts[:-1]), parts[-1]
        module = importlib.import_module(module_path)
        return getattr(module, class_name)
    except ImportError as e:
        msg = "Could not import '%s' for API setting '%s'. %s: %s." % (val, setting_name, e.__class__.__name__, e)
        raise ImportError(msg)


def build_billing_exception_response(exc):
    return Response(
        json.dumps({
            'params': exc.params,
            'message': exc.message,
            'code': exc.code,
        }),
        status=exc.status_code,
        mimetype='application/json; charset=utf-8'
    )


def parse_date(date_string):
    if type(date_string) == datetime.date:
        return date_string

    # Нормально обрабатываем ситуацию, когда передана не строка, а None
    # это иногда нужно, когда параметр со временем может быть не установлен
    # и в таких случаях он None.
    if date_string is None:
        return

    return datetime.datetime.strptime(date_string, "%Y-%m-%d").date()


def parse_birth_date(date_string):
    from intranet.yandex_directory.src.yandex_directory.passport.exceptions import BirthdayInvalid

    if not date_string:
        return

    try:
        date = parse_date(date_string)
    except Exception:
        log.warning('Unable to parse date')
        raise BirthdayInvalid()
    today = utcnow().date()
    # паспортные проверки даты рождения
    if date > today or date.year < today.year - 100:
        raise BirthdayInvalid()
    return date


def format_date(date, allow_none=False, only_date=False):
    if allow_none and not date:
        return None
    if only_date:
        return ensure_date(date).isoformat()
    return date.isoformat()


def try_format_date(date, allow_none=False):
    """
    Преобразуем дату в UTC ISO 8601 формат если date это date
    :param date: дата
    :type date: date|str
    :return: str
    """
    if isinstance(date, datetime.date):
        return format_date(date, allow_none=allow_none)
    return date


def format_datetime(date):
    """
    Преобразуем время в UTC ISO 8601 формат
    """
    # если время в UTC (без смещения), то возвращаем сразу
    if date.utcoffset() is None:
        return date.isoformat() + 'Z'
    # если есть некоторое смещение, то вычисляем это смещение
    utc_offset_sec = date.utcoffset()
    # и переводим в UTC
    utc_date = date - utc_offset_sec
    # и уберем информацию про смещение
    utc_date_without_offset = utc_date.replace(tzinfo=None)
    return utc_date_without_offset.isoformat() + 'Z'


def try_format_datetime(date):
    """
    Преобразуем время в UTC ISO 8601 формат если date это datetime
    :param date: датавремя
    :type date: datetime|str
    :return: str
    """
    if isinstance(date, datetime.datetime):
        return format_datetime(date)
    return date


def _standard_now(*args, **kwargs):
    """Эта вспомогательная функция нужна для того, чтобы в unit-тестах
       можно было замокать одно место и остановить время с помощью
       контекстного менеджера frozen_time.

       Мок напрямую на datetime.now не работает из-за такой ошибки:
       TypeError: can't set attributes of built-in/extension type 'datetime.datetime'
    """
    return datetime.datetime.now(*args, **kwargs)


def utcnow():
    return _standard_now(tz=pytz.utc)


def date_to_datetime(date):
    """Преобразует дату в полноценный datetime с таймзоной UTC."""
    return datetime.datetime(
        date.year,
        date.month,
        date.day,
        tzinfo=pytz.UTC,
    )


def time_in_future(seconds=None):
    """Возвращает время, которое отстоит от utcnow() на указанное количество секунд."""
    return utcnow() + datetime.timedelta(seconds=seconds)


def time_in_past(seconds=None):
    """Возвращает время, которое отстоит в прошлое от utcnow() на указанное количество секунд."""
    return utcnow() - datetime.timedelta(seconds=seconds)


def make_simple_strings(obj):
    """Возвращает новый словарь или список преобразуя
    интернационализированные строки в обычные.

    В API версий 5 и выше, мы работаем с обычными строками.
    """
    if isinstance(obj, dict):
        # Тут могут быть два варианта:
        # либо этот словарь на самом деле строка
        if 'ru' in obj or 'en' in obj or 'tr' in obj:
            ru_value = obj.get('ru')
            en_value = obj.get('en')
            tr_value = obj.get('tr')
            return ru_value or en_value or tr_value or ''
        else:
            # либо это обычный словарь, и тогда
            # надо обработать его значения
            return dict(
                (key, make_simple_strings(value))
                for key, value
                in obj.items()
            )
    # для списка – просто пропускаем каждый элемент через
    # такую же трансформацию
    elif isinstance(obj, (list, tuple)):
        return list(map(make_simple_strings, obj))
    else:
        # все другие объекты возвращаем как есть
        return obj


def json_response(data, status_code=200, headers=[], allowed_sensitive_params=None, json_format=True):
    assert status_code < 400, 'Для возврата ошибки, используйте json_error'

    # json_response должен вызываться только из View, а в рамках
    # обработки запроса, объект request существует
    if request.api_version >= 5:
        data = make_simple_strings(data)
    if app.config['ENVIRONMENT'] in ['qa', 'testing', 'autotests'] and find_sensitive_params(data,
                                                                                             allowed_sensitive_params):
        raise RuntimeError('Sensitive params detected')

    return Response(
        json.dumps(data, indent=4 if json_format else None),
        status=status_code,
        mimetype='application/json; charset=utf-8',
        headers=headers
    )


def not_modified_response():
    return Response('', status=304)


def json_error(status_code, error_code, error_message, headers=None, **kwargs):
    """
    Возвращает в ответ flask.Response с кодом ошибки и её описанием

    {
        "code": "bad_params.some_fields_are_required",
        "message": "These fields are required: {required_fields}",
        "params": {
            "required_fields": "domain, org_id"
        }
    }

    :param status_code: http код
    :param error_code: код ошибки
    :param error_message: человеко-читаемое сообщение об ошибке
    :param headers: заголовки ответа
    :param kwargs: дополнительные параметры для подстановки в тело сообщения об ошибки
    :rtype: flask.Response
    """
    data = {
        'code': error_code,
        'message': error_message,
    }

    if kwargs:
        data['params'] = kwargs

    # проверим, что есть все данные для подстановки в сообщение об ошибке
    try:
        # попробуем подставить в строку значения из kwargs
        error_message.format(**kwargs)
    except KeyError as e:
        # если не получилось, считаем, что в коде ошибка
        # и программист забыл передать параметры для вставки в сообщение
        # об ошибке
        raise AssertionError(
            ('Error message "{0}" contains placeholders, '
             'but no value for placeholder "{1}" was given.').format(
                error_message,
                str(e)
            )
        )

    return ErrorResponse(
        json.dumps(data),
        status=status_code,
        mimetype='application/json; charset=utf-8',
        headers=headers or [],
    )


# в окружении юнит-тестов мы собираем все коды ошибок
# в список, чтобы после прогона тестов сгенерить из них
# документацию в файл rst-docs/sources/errors.rst
if os.environ.get('ENVIRONMENT', '') == 'autotests':
    original_json_error = json_error
    all_json_errors = []


    def json_error(*args, **kwargs):
        all_json_errors.append((args, kwargs))
        return original_json_error(*args, **kwargs)


def json_error_bad_request():
    return json_error(400, 'bad_request', 'Bad request')


def json_error_exception(ex):
    if app.config['ENVIRONMENT'] in app.config['PRODUCTION_ENVIRONMENTS']:
        error = 'error'
    else:
        try:
            error = str(ex)
        except UnicodeDecodeError:
            # Иногда исключения почему-то содежат utf-8
            # к примеру, так происходит со всеми исключениями от SQLAlchemy,
            # так как внутри она делает тупо:
            #
            #   def __unicode__(self):
            #       return self.__str__()
            #
            # Поскольку, падать с пятисоткой, пытаясь вернуть пятисотку,
            # нехорошо, то сделаем ещё попытку преобразовать исключение
            # из строки:
            error = str(ex).decode('utf-8')

    return json_error(
        500,
        'unhandled_exception',
        'Unhandled exception: {error}',
        error=error,
    )


def json_error_not_found(*args):
    return json_error(404, 'not_found', 'Not found')


def json_error_forbidden(code=None):
    return json_error(403, code or 'forbidden', 'Access denied')


def json_error_service_unavailable():
    return json_error(
        503,
        'service_unavailable',
        'Service unavailable'
    )


def json_error_unknown():
    return json_error(500, 'unknown', 'Unknown error')


def json_error_invalid_value(field):
    return json_error(
        422,
        'invalid_value',
        'Field "{field}" has invalid value',
        field=field,
    )


def json_error_required_field(field):
    return json_error(
        422,
        'required_field',
        'Please, provide field "{field}"',
        field=field,
    )


def json_error_user_dismissed(uid):
    return json_error(
        422,
        'user_dismissed',
        'User {uid} is dismissed',
        uid=uid,
    )


def json_error_conflict_fields(fields):
    return json_error(
        422,
        'conflict_fields',
        'Conflicts founded in fields: "{fields}"',
        fields=fields
    )


def get_object_or_404(model_instance, **get_kwargs):
    model_object = model_instance.get(**get_kwargs)

    if model_object:
        return model_object
    else:
        raise ImmediateReturn(json_error_not_found())


def lstring(ru='', en=''):
    """Возвращает локализованную строку.

    Если задан только один параметр, то ru поле будет такое же, как en.
    """
    return {'ru': ru or en, 'en': en or ru}


def build_dsn(host,
              port,
              database,
              user=None,
              password=None,
              keepalives_idle=None,
              keepalives_interval=None,
              keepalives_count=None,
              ):
    if user:
        auth = user
        if password:
            auth += ':{0}@'.format(password)
        else:
            auth += '@'
    else:
        auth = ''

    opts = [
        ('keepalives_idle', keepalives_idle),
        ('keepalives_interval', keepalives_interval),
        ('keepalives_count', keepalives_count),
    ]
    # уберём опции, которые не были заданы
    opts = [item for item in opts if item[1] is None]
    # превратим в строку с разделением амперсандами
    opts = urllib.parse.urlencode(opts)

    return "postgresql://{auth}{host}:{port}/{database}?{opts}".format(
        auth=auth,
        host=host,
        database=database,
        port=port,
        opts=opts,
    )


def makedirs(path):
    # todo: test me
    try:
        os.makedirs(path)
    except OSError as exc:
        if exc.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise


def get_user_data_from_blackbox_by_login(login, attributes=None):
    attributes = attributes if attributes else []
    attributes.append(blackbox_client.IS_MAILLIST_ATTRIBUTE)
    userinfo = app.blackbox_instance.userinfo(
        login=login,
        userip=get_localhost_ip_address(),
        dbfields=app.config['BLACKBOX']['dbfields'],
        attributes=','.join(attributes),
        aliases='all',
        emails='getdefault',
    )

    return parse_user_info(userinfo, attributes=attributes)


def get_user_domain_from_blackbox(uid):
    userinfo = app.blackbox_instance.userinfo(
        uid=uid,
        userip=get_localhost_ip_address(),
    )
    return userinfo.get('domain')


def parse_user_info(userinfo, attributes=None):
    if userinfo.get('uid'):
        aliases = _get_user_aliases(userinfo)

        is_maillist = False
        is_available = False
        org_ids = []
        if attributes and userinfo.get('attributes') is not None:
            if blackbox_client.IS_MAILLIST_ATTRIBUTE in attributes:
                is_maillist = (userinfo['attributes'].get(blackbox_client.IS_MAILLIST_ATTRIBUTE, '0') == '1')
            if blackbox_client.IS_AVAILABLE_ATTRIBUTE in attributes:
                is_available = (userinfo['attributes'].get(blackbox_client.IS_AVAILABLE_ATTRIBUTE, '0') == '1')
            if blackbox_client.ORG_IDS_ATTRIBUTE in attributes:
                org_ids = userinfo['attributes'].get(blackbox_client.ORG_IDS_ATTRIBUTE, '')
                org_ids = org_ids.split(',')
                org_ids = map(int, filter(None, org_ids))

        avatar_id = userinfo.get('display_name', {}).get('avatar', {}).get('default')
        if not avatar_id:
            # фолбечимся на атрибут 98
            avatar_id = userinfo.get('attributes', {}).get(blackbox_client.AVATAR_ATTRIBUTE)

        if avatar_id == '0/0-0':
            avatar_id = None

        return {
            'uid': userinfo['uid'],
            'default_email': userinfo['default_email'],
            'aliases': aliases,
            'birth_date': userinfo['fields']['birth_date'],
            'login': userinfo['fields']['login'],
            'first_name': userinfo['fields']['first_name'],
            'last_name': userinfo['fields']['last_name'],
            'sex': userinfo['fields']['sex'],
            'language': userinfo['fields']['language'],
            'is_maillist': is_maillist,
            'karma': int(userinfo['karma']),
            'avatar_id': avatar_id,
            'is_available': is_available,
            'org_ids': org_ids,
            'attributes': userinfo.get('attributes'),
        }


def get_user_data_from_blackbox_by_uids(ip, uids, attributes=None, ):
    result = {}
    attributes = attributes if attributes else []
    attributes.extend((blackbox_client.IS_MAILLIST_ATTRIBUTE, blackbox_client.AVATAR_ATTRIBUTE))
    for batch in _split_in_batches(uids, 50):
        users_info = app.blackbox_instance.batch_userinfo(
            uids=batch,
            userip=ip,
            dbfields=app.config['BLACKBOX']['dbfields'],
            aliases='all',
            emails='getdefault',
            attributes=','.join(attributes)
        )
        for user_info in users_info:
            parsed_user_info = parse_user_info(user_info, attributes=attributes)
            if parsed_user_info:
                result[parsed_user_info['uid']] = parsed_user_info
    return result


def get_user_data_from_blackbox_by_uid(uid, ip=None, attributes=None, dbfields=None):
    ip = ip or get_localhost_ip_address()
    attributes = attributes if attributes else []
    attributes.extend((blackbox_client.IS_MAILLIST_ATTRIBUTE, blackbox_client.AVATAR_ATTRIBUTE))
    if dbfields is None:
        dbfields = app.config['BLACKBOX']['dbfields']
    userinfo = app.blackbox_instance.userinfo(
        uid=uid,
        userip=ip,
        dbfields=dbfields,
        aliases='all',
        emails='getdefault',
        attributes=','.join(attributes)
    )
    return parse_user_info(userinfo, attributes=attributes)


def get_user_contacts_from_blackbox_by_uid(uid):
    userinfo = app.blackbox_instance.userinfo(
        uid=uid,
        userip=get_localhost_ip_address(),
        emails='getall',
        getphones='bound',
        phone_attributes='1',
    )
    if userinfo and userinfo.get('uid'):
        phones = userinfo['fields'].get('phones')
        return {
            'uid': userinfo['uid'],
            'phone_number': phones[0] if phones else None,
            'email': [mail.get('address') for mail in userinfo.get('emails', []) if _email_is_outer(mail)],
        }
    else:
        return None


def _email_is_outer(email):
    """
    Функция проверят, что email является внешним
    Подробнее про параметры:
        default: Признак того, что e-mail является адресом по умолчанию.
        native: Признак внутреннего e-mail.
        rpop: Признак почтового ящика, письма с которого забирает почтовый сборщик Яндекса.
        silent: Признак адреса, на который не следует отправлять оповещения от Паспорта.
        validated: Признак подтвержденности электронного адреса.
    """
    if email["default"] is False and \
            email["native"] is False and \
            email["rpop"] is False and \
            email["silent"] is False and \
            email["validated"] is True:
        return True
    else:
        return False


def get_suid_from_blackbox_for_uid(uid, sid):
    """
    Выдаёт suid пользователя на сервисе, если у пользователя имеется подписка на указанный сервис,
    иначе None.
    :param uid: id аккаунта в паспорте
    :param sid: id сервиса
    :return: id пользователя в сервисе
    :rtype: int|None
    """
    subscription = app.blackbox_instance.subscription(uid, sid)
    if subscription:
        return subscription['suid']


def _get_user_aliases(userinfo):
    if userinfo['fields']['aliases']:
        aliases = [
            alias_value
            for alias_type, alias_value in userinfo['fields']['aliases']
            # Выбираем только алиасы с типом pddalias, потому что бывают и другие:
            # https://wiki.yandex-team.ru/passport/dbmoving/#tipyaliasov
            #
            # В частности, в свиске алиасов может отдаваться и текущий логин
            # пользователя
            if alias_type == '8'
        ]
    else:
        aliases = []
    return aliases


def get_account_list_from_blackbox(**kwargs):
    account_uids = app.blackbox_instance.account_uids(**kwargs)

    with log.fields(uids=account_uids):
        log.debug('Got account uids from Blackbox')

    account_infos = [get_user_data_from_blackbox_by_uid(uid) for uid in account_uids]
    return account_infos


def is_domain_has_accounts(domain):
    account_uids = app.blackbox_instance.account_uids(domain=domain)
    return bool(account_uids)


def get_domain_info_from_blackbox(domain, with_aliases=False):
    domain_info = app.blackbox_instance.hosted_domains(
        domain=domain,
        aliases=with_aliases,
    )
    # Тут две проверки выглядят странно, но зато оно работает когда
    # blackbox_instance - Mock
    if 'hosted_domains' in domain_info and domain_info['hosted_domains']:
        info = domain_info['hosted_domains'][0]
        result = {
            'admin_id': int(info['admin']),
            'registration_date': datetime.datetime.strptime(
                info['born_date'],
                '%Y-%m-%d %H:%M:%S'
            ),
            'domain_id': info['domid'],
            'master_domain': info['master_domain'] or None,
            'mx': info['mx'],
            'blocked': info.get('ena', '1') == '0',

        }
        if with_aliases:
            result['aliases'] = [_f for _f in info.get('slaves', '').split(',') if _f]
        return result


def get_domain_id_from_blackbox(domain):
    domainsinfo = get_domain_info_from_blackbox(domain)
    if domainsinfo:
        return domainsinfo['domain_id']
    return None


def get_user_id_from_passport_by_login(login):
    user_data_passport = get_user_data_from_blackbox_by_login(login)
    return int(user_data_passport['uid']) if user_data_passport else None


def get_user_login_from_passport_by_uid(uid):
    user_data_passport = get_user_data_from_blackbox_by_uid(uid)
    return user_data_passport['login'] if user_data_passport else None


def check_permissions(meta_connection,
                      main_connection,
                      permissions,
                      object_type=None,
                      object_id=None,
                      any_permission=False):
    """
    Проверка прав доступа. Отдает Nonе, если у пользователя есть все (или одно из них, если any_permission=True) права
    из списка permissions, иначе кидает исключение и ручка вернёт 403
    Args:
        permissions (list): список прав на проверку
        object_type (string): тип объекта {'group', 'user', 'department',
            'resource'}
        any_permission (bool): проверка должна проходить если у пользователя есть хотя бы одно право из списка
    """
    if not hasattr(permissions, '__iter__') or isinstance(permissions, (str, bytes)):
        raise TypeError('First argument must be iterable object')

    # Права проверяем только для токенов выписанных от имени пользователя.
    if not g.user:
        return

    from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationModel
    if main_connection and OrganizationModel(main_connection).is_blocked(g.org_id):
        raise OrganizationIsBlocked()

    has_permissions = g.user.has_permissions(
        meta_connection,
        main_connection,
        permissions,
        object_type,
        object_id,
        any_permission=any_permission,
        org_id=g.org_id,
    )

    if not has_permissions:
        with log.fields(required_permissions=permissions):
            log.debug('User has no permissions')
        raise ImmediateReturn(json_error_forbidden())


def check_label_or_nickname_or_alias_is_uniq_and_correct(
    main_connection,
    label_or_nickname_or_alias,
    org_id,
    user_id=None,
    email=None,
    is_cloud=False,
):
    """
    Проверяем уникальность передаваемого извне имени для группы,
    отдела или сотрудника. На одну организацию пока не даём создавать
    одинаковые email-ы для групп, отделов или сотрудников.
    Если есть конфликт, то возвращаем response с 409 и соотв. ответом
    Далее проверяем email в паспорте на валидность.
    """

    if not label_or_nickname_or_alias:
        return None
    from intranet.yandex_directory.src.yandex_directory.core.utils import (
        check_label_length,
        is_yandex_team_org_id,
    )

    check_label_length(label_or_nickname_or_alias, is_cloud=is_cloud)
    if is_cloud:
        # для Облака не проверяем ничего, надеемся на констреинты в базе
        # в Облаке могут совпадать атрибуты (например email) или даже быть пустыми
        return

    from intranet.yandex_directory.src.yandex_directory.core.models.user import UserModel
    from intranet.yandex_directory.src.yandex_directory.core.models.department import DepartmentModel
    from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel
    from intranet.yandex_directory.src.yandex_directory.core.utils import build_email, is_outer_uid, is_cloud_uid
    from intranet.yandex_directory.src.yandex_directory.passport.exceptions import LoginNotavailable, LoginNotAvailable
    from intranet.yandex_directory.src.yandex_directory.common.db import get_meta_connection

    alias_filter = {
        'alias': label_or_nickname_or_alias,
        'org_id': org_id,
    }
    label_filter = {
        'label': label_or_nickname_or_alias,
        'org_id': org_id,
    }
    nickname_filter = {
        'org_id': org_id,
    }
    with get_meta_connection() as meta_connection:
        has_feature = is_feature_enabled(meta_connection, org_id, MULTIORG)
    if has_feature:
        if email is not None and email != '':
            nickname_filter['email'] = email
        else:
            nickname_filter['email'] = build_email(
                main_connection,
                label_or_nickname_or_alias,
                org_id=org_id,
                user_id=user_id,
            )
    else:
        nickname_filter['nickname'] = label_or_nickname_or_alias

    if UserModel(main_connection).find(nickname_filter) or UserModel(main_connection).find(alias_filter):
        raise ImmediateReturn(
            json_error(
                409,
                'some_user_has_this_login',
                'Some user already exists with login "{login}"',
                login=label_or_nickname_or_alias,
            )
        )

    if DepartmentModel(main_connection).find(label_filter) or DepartmentModel(main_connection).find(alias_filter):
        raise ImmediateReturn(
            json_error(
                409,
                'some_department_has_this_label',
                'Some department already uses "{login}" as label',
                login=label_or_nickname_or_alias,
            )
        )

    if GroupModel(main_connection).find(label_filter) or GroupModel(main_connection).find(alias_filter):
        raise ImmediateReturn(
            json_error(
                409,
                'some_group_has_this_label',
                'Some group already uses "{login}" as label',
                login=label_or_nickname_or_alias,
            )
        )

    # логины из тима, портальные и облачные аккаунты не валидируем через паспорт
    if is_yandex_team_org_id(main_connection, org_id) or is_outer_uid(user_id) or is_cloud_uid(user_id):
        return
    # валидируем email в Паспорте
    domain_login = build_email(main_connection, label_or_nickname_or_alias, org_id=org_id, user_id=user_id)
    try:
        app.passport.validate_login(domain_login)
    except (LoginNotavailable, LoginNotAvailable):
        # не обращаем внимание на эту ошибку сейчас, она обрабатывается дальше
        pass


def update_maillist_label(main_connection, label, org_id, update_dict, old_uid):
    from intranet.yandex_directory.src.yandex_directory.core.utils import create_maillist, build_email
    from intranet.yandex_directory.src.yandex_directory.core.models.service import OrganizationServiceModel, MAILLIST_SERVICE_SLUG

    bigml_enabled = OrganizationServiceModel(main_connection).is_service_enabled(
        org_id,
        MAILLIST_SERVICE_SLUG
    )
    check_label_or_nickname_or_alias_is_uniq_and_correct(main_connection, label, org_id)
    if label:
        if bigml_enabled:
            uid = create_maillist(main_connection, org_id, label, ignore_login_not_available=True)
            update_dict['uid'] = uid
    else:
        update_dict['uid'] = None

    if bigml_enabled and old_uid:
        app.passport.maillist_delete(old_uid)

    update_dict['label'] = label
    update_dict['email'] = build_email(main_connection, label, org_id)

    return update_dict


def log_work_time(func):
    """Выводит в лог записи о времени выполнения
    обёрнутой функции. Время логгируется только если
    установлен X_DEBUG=yes или g.x_debug=1
    """

    @wraps(func)
    def wrapper(*args, **kwargs):
        start_time = time.time()
        message = 'Section success'

        try:
            result = func(*args, **kwargs)
        except Exception:
            # Если произошла ошибка, то логгировать время
            # будем с другим текстом
            message = 'Section failure'
            # Снова бросим исключение, чтобы оно выкинулось
            # дальше.
            raise
            # Но перед тем, как управление будет передано выше
            # по стеку, у нас вызовется finally
            # и залоггирует время с правильным сообщением
            # либо об успехе, либо о провале.
        finally:
            if is_debug_logging_enabled():
                function_name = func.__name__

                # Если у функции есть аргументы, то попробуем вычислить
                # имя класса.
                if args:
                    cls = getattr(args[0], '__class__', None)
                    class_name = getattr(cls, '__name__', None)
                    if class_name:
                        function_name = class_name + '.' + function_name

                end_time = time.time()
                total_time = end_time - start_time

                with log.name_and_fields(
                        'trace',
                        function=function_name,
                        start_time=start_time,
                        end_time=end_time,
                        total_time=total_time):

                    log.debug(message)
        # И если исключения не произошло, то нужно вернуть
        # результат.
        return result

    return wrapper


def is_debug_logging_enabled():
    try:
        g_x_debug = g.get('x_debug')
        app_x_debug = app.config.get('X_DEBUG')
        return g_x_debug or app_x_debug
    except Exception:
        return False


@contextmanager
def app_context():
    # в автотестах уже есть контекст
    if app.config['ENVIRONMENT'] == 'autotests':
        yield
    else:
        with app.app_context():
            yield


def with_app_context(function):
    @wraps(function)
    def wrapper(*args, **kwargs):
        with app_context():
            return function(*args, **kwargs)

    return wrapper


def strip_object_values(obj):
    if isinstance(obj, dict):
        for key, value in obj.items():
            if isinstance(value, str):
                obj[key] = value.strip()
            elif isinstance(value, dict):
                strip_object_values(obj[key])
    return obj


def measure_request_time_hook(response, *args, **kwargs):
    """
    Хук для requests чтобы измерять время походов в другие сервисы и писать его в лог и аггрегатор для голована
    """
    host, port = get_host_and_port_from_url(response.request.url)
    metric_name = 'response_time_%s_%s' % (host.replace('-', '_').replace('.', '_'), port)

    app.stats_aggregator.add_to_bucket(
        metric_name,
        response.elapsed.total_seconds(),
    )

    metric_name = 'response_status_%s_%s_%s_summ' % (host.replace('.', '-'), port, response.status_code)
    app.stats_aggregator.inc(metric_name)


def find_domid(hosted_domains, name):
    """Ищет среди доменов отданных блекбоксом, домен с заданным именем.
enable_service
       Параметр name должен быть в punicode.
    """
    # локально т.к. иначе циклический импорт
    punycode_name = to_punycode(name.lower())
    for domain in hosted_domains['hosted_domains']:
        if domain['domain'].lower() == punycode_name:
            return domain['domid']


def create_requests_session(ca_chain):
    requests_session = requests.Session()
    requests_session.verify = ca_chain
    requests_session.hooks['response'].append(measure_request_time_hook)

    user_agent = 'yandex-directory'

    class DirectoryHTTPAdapter(requests.adapters.HTTPAdapter):
        def add_headers(self, request, **kwargs):
            request.headers['User-Agent'] = user_agent
            return request

    adapter = DirectoryHTTPAdapter(max_retries=3)
    requests_session.mount('https://', adapter)
    requests_session.mount('http://', adapter)
    return requests_session


def ignore_first_args(n):
    """Этот декоратор возвращает функцию, первые n аргументов
    которой просто игнорируются, а остальные - передаются
    в обёрнутую функцию.

    Это работает только для позиционных аргументов.
    """

    def decorator(func):
        @wraps(func)
        def ignore(*args, **kwargs):
            return func(*args[n:], **kwargs)

        return ignore

    return decorator


def first_or_none(iter_obj):
    """
    Функция, которая возвращает либо первый объект, либо None
    """
    if not isinstance(iter_obj, collections.abc.Iterable):
        raise ValueError('Object should be iterable')
    iterator = iter(iter_obj)
    try:
        return next(iterator)
    except StopIteration:
        pass


def ensure_list_or_tuple(obj):
    """Если объект уже является списком, то возвращает его как есть.
       В противном случае - оборачивает в список.
    """
    if isinstance(obj, (list, tuple)):
        return obj
    else:
        return [obj]


def get_days_in_month(for_date):
    date = ensure_date(for_date)
    return calendar.monthrange(date.year, date.month)[1]


def ensure_date(date_or_datetime):
    if isinstance(date_or_datetime, datetime.datetime):
        # Важно сначала привести таймзону к UTC, иначе последующее сравнение
        # этой даты с другими значениями времени будет неверным.
        return date_or_datetime.astimezone(pytz.UTC).date()
    elif isinstance(date_or_datetime, datetime.date):
        return date_or_datetime
    elif isinstance(date_or_datetime, str):
        # Если была передана строка, то она должна являться датой
        return parse_date(date_or_datetime)
    else:
        raise TypeError(
            'Don\'t know how to cast this value "{0}" to the date.'.format(
                date_or_datetime
            )
        )


def get_auth_fields_for_log(obj):
    """
    Берёт из объекта `obj` описания текущего сервиса и пользователя,
    и возвращает словарь, годный для использования в полях логгера:
    obj это объект g или auth_response

    {
       "user": {
           "id": <uid пользоватля>,
           "up": "его IP"
       },
       "service": {
           "ip": <IP с которого было обращение к API Директории>,
           "slug": "слаг сервиса",
           "internal": True|False
       },
       "type": "token|tvm|oauth"
    }
    """
    fields = {}

    if isinstance(obj, dict):
        getter = obj.get
    else:
        getter = lambda name, default: getattr(obj, name, default)

    org_id = getter('org_id', None)
    if org_id is not None:
        fields['org_id'] = org_id

    user = getter('user', None)
    if user is not None:
        fields['user'] = dict(
            id=user.passport_uid,
            cloud_uid=user.cloud_uid,
            ip=user.ip,
            karma=user.karma,
            is_cloud=user.is_cloud,
        )

    service = getter('service', None)
    if service is not None:
        fields['service'] = dict(
            ip=service.ip,
            slug=service.identity,
            internal=service.is_internal,
        )

    auth_type = getter('auth_type', None)
    if auth_type is not None:
        fields['type'] = auth_type

    return fields


def get_boolean_param(params, name, required=False):
    """
    Приводит значение параметра к boolean
      'true' -> True
      'false' -> False
      'если не обязателен' -> None
    :param params: словарь параметров
    :param name: имя параметра
    :param required: обязательность
    """
    if required and name not in params:
        raise ImmediateReturn(response=json_error_required_field(name))

    value = params.get(name)
    if value is not None:
        value = value.lower()
        if value not in ('true', 'false'):
            raise ImmediateReturn(response=json_error_required_field(name))
        return value == 'true'


def join_dicts(*dicts):
    """Принимает любое количество словарей и возвращает новый, содержимое
    которого построено посредством склеивания входных словарей в один.
    """
    result = dict()
    for d in dicts:
        result.update(d)
    return result


def to_lowercase(text):
    """
    Переводит строковое значение в нижний регистр
    """
    if isinstance(text, str):
        return text.lower().strip()
    return text


def make_internationalized_strings(data, schema):
    """Проходится по словарю или списку, заменяя обычные строки на
    интернационализированные, там где того требует JSON схема.
    """
    properties = schema.get('properties')
    if properties:
        if isinstance(data, dict):
            # Если на вход поступил словарь, то возможно его
            # значения должны быть преобразованы к интернационализированным
            # строкам.
            return dict(
                (
                    key,
                    make_internationalized_strings(
                        value,
                        schema=properties.get(key, {})
                    )
                )
                for key, value in data.items()
            )
        else:
            # Если требуется интернационализированная строка, то
            if 'ru' in properties:
                if isinstance(data, str):
                    return {'ru': data}
    else:
        # Если в схеме описано, что данные должны быть массивом объектов,
        # то извлечём схему для этих вложенных сущностей
        item_schema = schema.get('items')
        if item_schema:
            # Но данные будем процессить только если это
            # действительно list или tuple
            if isinstance(data, (list, tuple)):
                return [
                    make_internationalized_strings(
                        item,
                        item_schema
                    )
                    for item in data
                ]

    return data


def get_environment():
    environment = os.environ.get('ENVIRONMENT')
    if isinstance(environment, str):
        return environment.lower()


def stopit(func, timeout=1, raise_timeout=True, default=None):
    """
    Запускаем функцию в треде с таймаутом, если не получилось её выполнить,
    и raise_timeout == True, бросаем исключение TimeoutError,
    иначе возвращаем значение default
    """
    executor = ThreadPoolExecutor(max_workers=1)

    def wrapper(*args, **kwargs):
        future = executor.submit(func, *args, **kwargs)
        try:
            return future.result(timeout=timeout)
        except TimeoutError:
            function_name = '{}.{}'.format(func.__module__, func.__name__)
            with log.fields(func=function_name, timeout=timeout):
                log.warning('Function execution timeout')
            if raise_timeout:
                raise
            else:
                return default

    return wrapper


def remove_sensitive_data(data, secret_params=None):
    """
    Заменяем пароли и т.п. на "****"
    :param data:
    :type data: dict
    :rtype: dict
    """
    if secret_params is None:
        secret_params = app.config['SENSITIVE_KEYS']

    result = copy(data)
    if isinstance(result, tuple):
        result = list(result)

    if isinstance(result, list):
        for index, item in enumerate(result):
            if isinstance(item, (tuple, list, dict)):
                result[index] = remove_sensitive_data(item, secret_params)
    elif isinstance(result, dict):
        for k, v in result.items():
            if isinstance(v, (tuple, list, dict)):
                result[k] = remove_sensitive_data(v, secret_params)
            if k.lower() in secret_params:
                result[k] = app.config['SECRET_PLACEHOLDER']
    return result


def log_service_response(service_name, method, url, data, response, get_logger=None):
    """Логгирует json ответ сервиса.

    Если код ответа не 200, то ответ логгируется как ERROR.

    При этом имя логгера устанавливается в requests.disk, или requests.webmaster

    Можно в параметре get_logger передать функцию, принимающую один
    аргумент - раскодированный из json ответ сервиса. Функция должна вернуть
    log.info, log.error или None. В случае с None, логгироваться ничего не будет.
    """

    # Попробуем раскодировать ответ, если получится, то сложим как отдельные поля.
    try:
        response_data = response.json()
    except:
        response_data = response.text

    status_code = response.status_code

    with log.name_and_fields(
            'requests.' + service_name,
            method=method,
            url=url,
            data=remove_sensitive_data(data),
            response_code=status_code,
            response=remove_sensitive_data(response_data),
    ):
        # По-умолчанию, логгируем как info
        log_func = log.info
        if service_name == 'domenator' and status_code == 404:
            log_func = log.info
        # А если код какой-то ошибочный, то как ERROR
        elif status_code >= 400:
            log_func = log.error
        else:
            # Некоторые API отдают 200, и была ли ошибка надо проверять по телу ответа
            # в таком случае, мы можем использовать проверку, переданную извне
            if get_logger:
                log_func = get_logger(response_data)

        # Дополнительная проверка может вернуть None, чтобы отменить
        # логгирование вовсе.
        if log_func is not None:
            log_func('Service response - {}'.format(url))


class hashabledict(dict):
    """
    Для случая если нам надо передать словарь как элемен set обычный dict не работает
    т.к. он unhashable.
    """

    def __hash__(self):
        """
        Вычислим хэш для словаря.
        Рекурсивно обходим его для значений типа dict.
        """
        h = 0
        for key, value in self.items():
            if isinstance(value, dict):
                val = hashabledict(value)
            else:
                val = value
            h ^= hash((key, val))
        return h


class ErrorResponse(Response):
    pass


def create_domain_in_passport(main_connection,
                              org_id,
                              punycode_name,
                              admin_uid):
    # создаем домен в паспорте
    with log.fields(domain=punycode_name):
        log.info('Creating  domain in passport')

        from intranet.yandex_directory.src.yandex_directory.core.models.domain import DomainModel

        domain_model = DomainModel(main_connection)

        try:
            master_domain = domain_model.get_master(org_id)
        except DomainNotFound:
            master_domain = None

        add_master = False
        # если мастер домена нет или мы пытаемся его подтвердить, то будем добавлять домен как основной
        if not master_domain or master_domain['name'] == punycode_name:
            add_master = True

        if add_master:
            log.info('Add master domain')
            # добавляем домен в паспорт как основной
            from intranet.yandex_directory.src.yandex_directory.passport.exceptions import DomainAlreadyExists
            try:
                app.passport.domain_add(punycode_name, admin_uid)
            except DomainAlreadyExists:
                # пытаемся восстановить консистентное состояние
                is_deleted = safe_delete_domain_in_passport(punycode_name, skip_check_domain_in_connect=True)
                if not is_deleted:
                    raise

                # вторая попытка добавить домен
                app.passport.domain_add(punycode_name, admin_uid)
        else:
            master_domain = master_domain['name']
            master_domain_id = get_domain_id_from_blackbox(master_domain)
            # добавляем домен как алиас в паспорте
            log.info('Adding alias in passport')
            app.passport.domain_alias_add(master_domain_id, punycode_name)
            alias_domain_id = get_domain_id_from_blackbox(punycode_name)
            app.passport.domain_edit(alias_domain_id, {'admin_uid': admin_uid})

        # нужно добавить DKIM подпись
        from intranet.yandex_directory.src.yandex_directory.core.tasks import CheckDomainInGendarmeAndGenDkimTask
        CheckDomainInGendarmeAndGenDkimTask(main_connection).delay(
            org_id=org_id,
            domain=punycode_name,
        )

        log.info('Domain was created in passport')
        return add_master


def delete_domain_in_passport(domain_name, admin_uid, is_alias, master_domain_id=None):
    with log.name_and_fields('delete_domain_from_passport', domain=domain_name):
        log.info('Trying to delete domain from passport')

        passport_domain_info = get_domain_info_from_blackbox(domain_name)
        # удаляем домен из паспорта, если он там есть и принадлежит этому админу
        if passport_domain_info and passport_domain_info['admin_id'] == admin_uid:
            # если алиас - удаляем из паспорта алиас
            # если мастер - удаляем домен
            if is_alias:
                app.passport.domain_alias_delete(master_domain_id, passport_domain_info['domain_id'])
            else:
                app.passport.domain_delete(passport_domain_info['domain_id'])

        else:
            log.info('Domain was not found in passport')


def check_domain_is_correct(meta_connection,
                            main_connection,
                            domain_name,
                            admin_uid,
                            org_id,
                            connect=False,
                            ignore_duplicate_domain_in_same_organization=False,
                            ):
    """
    Проверяем, что доменное имя можно добавить в директорию:
        - доменное имя корректно переводится в пуникод
        - домен не подтвержден этим же админом в другой организации
        - доменное имя корректное с точки зрения Паспорта (проверка на допустимые символы)
        - если домен не коннектный и есть в Паспорте у другого админа -> возвращается ошибка
    если connect=True, значит проверяем технический домен вида .yaconnect.com, дополнительные проверки:
        - имя оканчивается на .yaconnect.com
        - в организации еще нет технического домена
    """
    # проверяем, что имя домена корректное и он никем не занят

    from intranet.yandex_directory.src.yandex_directory.common.exceptions import DomainNameTooLongError
    from intranet.yandex_directory.src.yandex_directory.core.models.domain import to_punycode, DomainModel
    from intranet.yandex_directory.src.yandex_directory.core.utils.domain import assert_can_add_invalidated_alias
    from intranet.yandex_directory.src.yandex_directory.passport.exceptions import DomainAlreadyExists
    from intranet.yandex_directory.src.yandex_directory.core.utils.domain import domain_is_tech

    domain_name = domain_name.lower()
    if '.' not in domain_name:
        # если в доменном имени нет точки, вернем ошибку
        raise InvalidDomain()
    if domain_name in settings.BLACK_DOMAINS_LIST:
        raise InvalidDomain()

    is_tech_domain = domain_is_tech(domain_name)

    if connect:
        # если технический домен уже есть, не даем добавить еще один
        domains = DomainModel(main_connection).find(
            filter_data={
                'org_id': org_id,
                'owned': True,
            },
            fields=['name']
        )
        for d in domains:
            if domain_is_tech(d['name']):
                raise ImmediateReturn(json_error(
                    422,
                    'tech_domain_already_exists',
                    'Can\'t add tech domain',
                ))
        if not is_tech_domain:
            raise InvalidDomain()
    elif is_tech_domain:
        raise ImmediateReturn(json_error(
            422,
            'domain_is_tech',
            'Can\'t add domain with this name',
        ))

    try:
        punycode_name = to_punycode(domain_name)
    except DomainNameTooLongError:
        raise ImmediateReturn(json_error(
            422,
            'domain_too_long',
            'Domain name is too long',
        ))
    except UnicodeError:
        raise ImmediateReturn(json_error(
            422,
            'domain_unicode_error',
            'Domain name have some unicode issues',
        ))

    exists_domains = DomainModel(None).find({'name': punycode_name})
    if exists_domains and any([dom['org_id'] == org_id for dom in exists_domains]):
        # домен не должен быть добавлен в этой же организации
        if not ignore_duplicate_domain_in_same_organization:
            raise ImmediateReturn(json_error(
                409,
                'duplicate_domain',
                'Domain already added',
            ))

    # https://st.yandex-team.ru/DIR-2156
    assert_can_add_invalidated_alias(
        meta_connection,
        main_connection,
        org_id,
        domain_name,
    )

    if all([not dom['owned'] for dom in exists_domains]):
        # в коннекте этот домен никто не подтвердил, проверяемм что в паспорте его нет
        from intranet.yandex_directory.src.yandex_directory.core.exceptions import DomainExistsInPassport
        try:
            assert_domain_not_exists_in_passport(punycode_name, admin_uid)
        except DomainExistsInPassport:
            # пытаемся восстановить консистентное состояние
            is_deleted = safe_delete_domain_in_passport(punycode_name, skip_check_domain_in_connect=True)
            if not is_deleted:
                raise

            # вторая попытка проверки
            assert_domain_not_exists_in_passport(punycode_name, admin_uid)

    # проверяем, можно ли вообще добавить такой домен в Паспорт
    try:
        if connect:
            app.passport.validate_connect_domain(domain_name)
        else:
            app.passport.validate_natural_domain(domain_name)
    except DomainAlreadyExists:
        log.info('Domain %s exists in passport' % domain_name)

    return punycode_name


def assert_domain_not_exists_in_passport(domain_name, admin_uid):
    from intranet.yandex_directory.src.yandex_directory.core.exceptions import (
        DomainExistsInPassport,
        DomainIsAlias,
        DomainHasAccounts,
        DomainHasAliases,
    )

    # если домен не коннектный, то идем в Паспорт и смотрим, есть ли он там
    blackbox_domain_info = get_domain_info_from_blackbox(domain_name, with_aliases=True)
    if blackbox_domain_info:
        # если домен есть в Паспорте и нет в коннекте, значит что-то пошло не так
        if blackbox_domain_info['admin_id'] != admin_uid:
            # если домен подтвержден другим админом, то
            # т.к. домен не коннектный, то автоматически передать мы его не сможем
            raise DomainExistsInPassport

        # проверим что это не алиас
        if blackbox_domain_info['master_domain']:
            raise DomainIsAlias()

        if blackbox_domain_info['aliases']:
            raise DomainHasAliases()

        if is_domain_has_accounts(domain_name):
            raise DomainHasAccounts()


def get_email_from_team_blackbox_by_uid(uid):
    userinfo = app.team_blackbox_instance.userinfo(
        uid=uid,
        userip=get_localhost_ip_address(),
        emails='getdefault',
    )
    return userinfo.get('default_email')


def get_user_login_from_team_blackbox_by_uid(uid):
    userinfo = app.team_blackbox_instance.userinfo(
        uid=uid,
        userip=get_localhost_ip_address(),
        dbfields=[blackbox_client.FIELD_LOGIN],
    )
    if userinfo.get('uid'):
        return userinfo.get('fields', {}).get('login')
    else:
        return None


def get_internal_roles_by_uid(uid):
    login = get_user_login_from_team_blackbox_by_uid(uid)
    if not login:
        return []
    roles = app.idm_internal_client.get_all_active_roles(login)
    return roles.get(login, [])


def multikeysort(items, columns=None, getter=itemgetter):
    """
    Сортировка списка словарей или объектов по нескольким ключам в разном направлении.

    :param items: список словарей или объектов
    :param columns: поля, по которым сортируем. Для сортировки desc использовать -column
    :param getter: как брать поле из объетка.
           operator.itemgetter для словарей
           operator.attrgetter для объектов
    """
    comparers = []
    if not columns:
        return items
    for col in columns:
        column = col[1:] if col.startswith('-') else col
        comparers.append((getter(column), 1 if column == col else -1))

    def cmp_own(a, b):
        return (a > b) - (a < b)

    def comparer(left, right):
        for func, direction in comparers:
            result = cmp_own(func(left), func(right))
            if result:
                return direction * result
        else:
            return 0

    return sorted(items, key=cmp_to_key(comparer))


def remove_keys_which_is(the_dict, checked_value):
    """Возвращает новый словарь, в котором убраны все ключи, для которых значение идентично указанному."""
    return {
        key: value
        for key, value in the_dict.items()
        if value is not checked_value
    }


def remove_not_changed_keys(the_dict):
    """Возвращает новый словарь, в котором убраны все ключи, для которых значение is NotChanged."""
    return remove_keys_which_is(the_dict, NotChanged)


def remove_ignored_keys(the_dict):
    """Возвращает новый словарь, в котором убраны все ключи, для которых значение is Ignore."""
    return remove_keys_which_is(the_dict, Ignore)


def remove_not_given_keys(the_dict):
    """Возвращает новый словарь, в котором убраны все ключи, для которых значение is NotGiven."""
    return remove_keys_which_is(the_dict, NotGiven)


# Пароль должен заменяться на такое значение
SENSITIVE_DATA_PLACEHOLDER = '******'
# При первом использовании is_sensitive_param в эту переменную будет сохранён скомпилированный regex.
SENSITIVE_DATA_PATTERN = None


def is_sensitive_param(key):
    global SENSITIVE_DATA_PATTERN
    if SENSITIVE_DATA_PATTERN is None:
        # если параметр содержит какое-нибудь значение из списка SENSITIVE_KEYS -
        # его надо заменить на SENSITIVE_DATA_PLACEHOLDER в логах, параметрах и метаданных тасков.
        SENSITIVE_DATA_PATTERN = re.compile('.*({}).*'.format('|'.join(app.config['SENSITIVE_KEYS'])),
                                            re.IGNORECASE)
    return SENSITIVE_DATA_PATTERN.match(key)


def hide_sensitive_params(smth):
    res = False
    if isinstance(smth, dict):
        for key in smth:
            if isinstance(key, str) and is_sensitive_param(key):
                smth[key] = SENSITIVE_DATA_PLACEHOLDER
                res = True
            else:
                res = hide_sensitive_params(smth[key]) or res
    elif isinstance(smth, (tuple, list)):
        for item in smth:
            res = hide_sensitive_params(item) or res
    return res


def find_sensitive_params(smth, allowed_sensitive_params=None):
    allowed_sensitive_params = allowed_sensitive_params or set()
    if isinstance(smth, dict):
        for key in smth:
            if isinstance(key, str) \
                    and is_sensitive_param(key) \
                    and key not in allowed_sensitive_params \
                    and smth[key] != SENSITIVE_DATA_PLACEHOLDER:
                return True
            else:
                if find_sensitive_params(smth[key], allowed_sensitive_params=allowed_sensitive_params):
                    return True
    elif isinstance(smth, (tuple, list)):
        for item in smth:
            if find_sensitive_params(item, allowed_sensitive_params=allowed_sensitive_params):
                return True
    return False


def log_exception(exception, msg):
    params = getattr(exception, 'params', {})
    with_trace = getattr(exception, 'with_trace', True)
    with log.fields(**params):
        log_level = getattr(exception, 'log_level', 'ERROR').lower()
        if with_trace:
            getattr(log.trace(), log_level)(msg)
        else:
            getattr(log, log_level)(msg)

def get_attrs(obj):
    """Возвращает все нормальные, немагические атрибуты объекта."""
    return [
        getattr(obj, attr) for attr in dir(obj)
        if not attr.startswith('__')
    ]


def get_domain_from_email(email):
    if '@' not in email:
        return ''
    domain = email.split('@', 1)[1]
    return domain


DEFAULT_DOMAIN = 'yandex.ru'


def mask_email(email, mask_domain=False):
    # паспортный алгоритм маскирования
    # https://github.yandex-team.ru/passport/passport-core/blob/master/passport/python/core/types/email.py#L276
    login = email.split('@', 1)[0]
    domain = get_domain_from_email(email)

    if len(login) < 3:
        masked_login = '***'
    else:
        masked_login = '%s***%s' % (login[0], login[-1])

    if mask_domain:
        if '.' not in domain:
            masked_domain = '***'
        else:
            subdomain, domain_zone = domain.rsplit('.', 1)
            if len(subdomain) < 2:
                masked_domain = '***.%s' % domain_zone
            else:
                masked_domain = '%s***.%s' % (subdomain[0], domain_zone)
    else:
        masked_domain = domain or DEFAULT_DOMAIN

    return '@'.join([masked_login, masked_domain])


def has_caps(word):
    all_chars = set(word)
    lower_chars = set(word.lower())
    caps_chars = all_chars - lower_chars
    if len(caps_chars) > 1:
        return word


# Пока данные получены вручную. Но надо бы сделать так, чтобы вычитывались из файлика,
# так чтобы его легко было обновлять.
#
# data = requests.get('https://github.com/publicsuffix/list/raw/master/public_suffix_list.dat').text
_data = pkg_resources.resource_string('intranet.yandex_directory.src.yandex_directory', "domain-tld.dat").decode('utf-8')
_lines = _data.split('\n')
_tlds = sorted(
    set(
        line.rsplit('.')[-1]
        for line in _lines
        if line and not line.startswith('//')
    )
)

tlds_to_add = set()
for tld in _tlds:
    if isinstance(tld, str):
        idna_tld = tld.encode('idna')
        # Если преобразование привело к тому, что домен заэнкодился,
        # то его надо добавить в общий список, потому что спамеры
        # могут использовать разные варианты:
        # https://st.yandex-team.ru/COMPUTERPROBLEM-161#5ca4b825977527001ff33d58
        if tld != idna_tld:
            tlds_to_add.add(idna_tld.decode('utf-8'))

_tlds.extend(tlds_to_add)
has_domain_regex = r'\S\.({})'.format('|'.join(_tlds))


def maybe_has_domain(name):
    """Ищет, есть ли в имени что-то похожее на что-то.tld"""
    if re.search(has_domain_regex, name, re.I) is not None:
        return True


def is_shortener(name):
    shorteners = [
        # TODO: расширить этот список
        'bit.ly',
        'vk.cc',
        'ya.cc',
    ]
    for shortener in shorteners:
        if shortener in name:
            return True


def has_stop_words(name):
    stop_words = [
        'vk.com',
        'победил',
        'сайт',
        'приз',
        '://',
        'www.',
    ]
    lowercased = name.lower()
    for word in stop_words:
        if word in lowercased:
            return True


def is_spammy(name):
    """Проверяет, похож ли текст в name на спам.
       Используется для того, чтобы не давать заводить спамерские организации.
    """
    splitted = re.split(r'\s|\{0}'.format('|\\'.join(string.punctuation)), name)
    words = [_f for _f in splitted if _f]
    caps_words = [_f for _f in map(has_caps, words) if _f]

    if len(caps_words) >= 3:
        return True

    if has_stop_words(name):
        return True

    if maybe_has_domain(name):
        # Имена типа ВыПобедилиВашПризНаСайтеwww.vk.com
        # должны быть запрещены
        if caps_words:
            return True

        except_dot = [char for char in string.punctuation if char != '.']
        words_except_dot = re.split(r'\s|\{0}'.format('|\\'.join(except_dot)), name)
        # Если в домене слишком много слов, разделённых пунктуацией, но не точкам
        # то это подозрительно.
        if len(words_except_dot) > 4:
            return True
        # Если домен состоит из слишком большого числа частей, то
        # это тоже подозрительно. До свиданья!
        domain_parts = name.split('.')
        if len(domain_parts) > 4:
            return True
        # Если и с разделением по всем знакам пунктуации, включая точки,
        # получается слишком много частей, то возможно в названии есть
        # какое-то послание. Не пропускаем.
        #
        # Тут цифра должна быть больше чем в words_except_dot,
        # чтобы позволить
        if len(words) >= 5:
            return True

        if is_shortener(name):
            return True
    return False


def get_karma(uid, ip=None):
    data = get_user_data_from_blackbox_by_uid(uid, ip=ip)
    return data['karma']


def unpickle(data):
    # в базе строка закодирована для сохранения unicode литералов
    # https://docs.python.org/2/howto/unicode.html#unicode-literals-in-python-source-code
    if data:
        if isinstance(data, str):
            data = data.encode('utf-8')
        return pickle_module.loads(data)


def pickle(data):
    # закодируем для сохранения unicode литералов
    # https://docs.python.org/2/howto/unicode.html#unicode-literals-in-python-source-code
    return pickle_module.dumps(data, protocol=0).decode('utf-8')


def grouper(batch_size, iterable):
    """
    Принимает итерируемый объект и размер пачки, на которые его надо поделить
    Возвращает генератор, с итератороми размера batch_size по исходной последовательности
    """
    iterators = [iter(iterable)] * batch_size
    batches = zip_longest(*iterators)
    return ([_f for _f in batch if _f] for batch in batches)


def get_exponential_step(time_since_start,
                         min_interval,
                         max_interval,
                         const_time):
    """Высчитывает следующий шаг так, что он каждый раз увеличивается по экспоненте
       от min_interval до max_interval, но по прошествии const_time интервал
       всегда будет равным max_interval.

       Все значения задаются в секундах.
    """
    # Формула высчитана с помощью кода на Julia:
    # https://st.yandex-team.ru/DIR-7423#5d35678934894f001f74bf60
    return min(
        min_interval * math.pow(
            math.pow(
                float(max_interval) / float(min_interval),
                (1 / float(const_time))
            ),
            time_since_start
        ),
        max_interval
    )


def to_punycode(text):
    try:
        if isinstance(text, tuple):
            return tuple([to_punycode(i) for i in text])
        return force_text(text).encode('idna').decode()
    except UnicodeError as e:
        from intranet.yandex_directory.src.yandex_directory.common.exceptions import DomainNameTooLongError
        if 'too long' in str(e):
            message = 'Domain name {0} too long'.format(text).encode('utf-8')
            raise DomainNameTooLongError(message)
        raise


def from_punycode(text):
    """
    Декодируем строку из idna в unicode
    :param text: строка в idna
    :return: строка в unicode
    :rtype: unicode
    """
    return force_utf8(text).decode('idna')


def generate_id():
    """
    Случайный идентификатор
    :return: идентификатор
    :rtype: string
    """
    return uuid.uuid4().hex


def login_to_punycode(login):
    """
    Переводим доменную часть логина (если она есть) в punycode
    all@борщ.админкапдд.рф -> all@xn--90a0ag0b.xn--80aalbavookw.xn--p1ai
    :param login: логин
    :type login: str
    :return:
    :rtype: str
    """
    if not isinstance(login, str):
        return login
    login = login.lower()
    if '@' not in login:
        return login
    nickname, domain = tuple(login.rsplit('@', 1))
    return '{}@{}'.format(nickname, to_punycode(domain))


def login_from_punycode(login):
    """
    Перекодируем доменную часть логина (если она есть) из punycode в unicode
    all@xn--90a0ag0b.xn--80aalbavookw.xn--p1ai -> all@борщ.админкапдд.рф
    :param login: логин
    :type login: str|unicode
    :rtype: unicode
    """
    if not isinstance(login, str):
        return login
    if '@' not in login:
        return from_punycode(login)
    nickname, domain = tuple(login.rsplit('@', 1))
    return '{}@{}'.format(nickname, from_punycode(domain))


def generate_secret_key_for_invitation(uid):
    """
    Сгененировать секретный ключ для проверки приглашения на фронтенде.

    Сделано по логике, которая уже реализована на фронтенде:
    валидность ключа подтверждает валидность запроса.
    """
    sk = hashlib.md5()

    # full days since 01.01.1970
    days = math.floor(datetime.datetime.now().timestamp() / 86400)

    sk.update(f'{uid}::{days}'.encode('utf-8'))
    return f'u{sk.hexdigest()}'


def safe_delete_domain_in_passport(punycode_name: str, skip_check_domain_in_connect: bool = False) -> bool:
    """
    Пробуем удалить домен из паспорта, если выполнены следующие условия:
    - в коннекте нет подтвержденного домена с таким именем
    - в паспорте у домена нет алисов
    - в паспорте домен не является мастер-доменом с учетками
    """
    with log.name_and_fields('safe_delete_domain_in_passport', domain=punycode_name):
        if not skip_check_domain_in_connect:
            from intranet.yandex_directory.src.yandex_directory.core.models import DomainModel
            exists_domains = DomainModel(None).find({'name': punycode_name, 'owned': True})
            if exists_domains:
                log.info('Domain exists in connect')
                return False

        domain_info = get_domain_info_from_blackbox(punycode_name, with_aliases=True)
        if not domain_info:
            log.info('Domain not found in blackbox')
            return False

        is_master = domain_info['master_domain'] is None

        if is_master:
            if domain_info['aliases']:
                log.info('Domain has aliases')
                return False

            uids = app.blackbox_instance.account_uids(domain=punycode_name)
            if uids:
                log.info('Domain has users')
                return False

        # удаляем домен в паспорте
        domain_id = domain_info['domain_id']

        if is_master:
            log.info('Delete domain from passport')
            app.passport.domain_delete(domain_id)
        else:
            master_domain_name = domain_info['master_domain']

            master_domain_info = get_domain_info_from_blackbox(master_domain_name)
            if not master_domain_info:
                log.info('Master domain not found in blackbox')
                return False

            master_domain_id = master_domain_info['domain_id']

            log.info('Delete alias from passport')
            app.passport.domain_alias_delete(master_domain_id, domain_id)

        return True


def _split_in_batches(l, n):
    for i in range(0, len(l), n):
        yield l[i:i + n]
