# -*- coding: utf-8 -*-
import json
import logging

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log, error_log

from flask.views import MethodView
from flask import request

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    no_scopes,
    no_auth,
    requires,
    internal,
    no_permission_required,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    json_error,
)
from intranet.yandex_directory.src.yandex_directory.common.smoke import get_smoke_tests_results
from intranet.yandex_directory.src.yandex_directory.core.views.base import View


class IndexView(MethodView):
    @no_scopes
    @no_permission_required
    @requires(org_id=False, user=False)
    def get(self):
        return json_response(
            {
                'read': 'https://tech.yandex.ru/connect/directory/',
            },
        )


class TestNotify(MethodView):
    @no_scopes
    @internal
    @no_auth
    def post(self):
        data = request.get_json()
        print("Event %s for %s" % (data['event'], data['object']['id']))
        logging.getLogger('notification').info(data)
        return json_response(
            {
                'pong': 'OK',
            }
        )


class PingView(MethodView):
    # Эта ручка не помечена как internal, но будет закрыта
    # с помощью правила на роутере, так как она должна
    # существовать во всех инсталляциях Директории,
    # чтобы роутер мог пинговать бэкенды.
    @no_scopes
    @no_auth
    def get(self):
        response = get_smoke_tests_results(only_errors=True)

        if response['has_errors_in_vital_services']:
            with log.name_and_fields('views.ping', smoke_tests=response):
                log.error('Smoke tests detected errors in vital services')
            return json_error(
                500,
                'ping_error',
                'Smoke tests detected errors in vital services',
                smoke_tests=response,
            )
        else:
            return json_response(data=response)


class SimplePingView(MethodView):
    """
    Просто отдаем OK без походов в базу
    """
    # Тоже, не internal, как и просто PingView
    @no_scopes
    @no_auth
    def get(self):
        return json_response({'status': 'ok'}, json_format=False)


class ExceptionView(MethodView):
    """
    Проверка как у нас логируються исключеиня
    """
    @no_scopes
    @internal
    @no_auth
    def get(self):
        with log.fields(error='error', ):
            raise Exception('TestException')


class StatsView(View):
    """
    stats-ручка для голована
    https://wiki.yandex-team.ru/jandekspoisk/sepe/monitoring/stat-handle/
    Для проверки формата можно взять скрипт
    https://git.qe-infra.yandex-team.ru/projects/SEARCH_INFRA/repos/yasm/browse/scripts/sre/verify_unistat.py?raw

    Пример:
    [
        ["m0_hgram", [[0.0, 2], [0.5, 1]]],
        ["m1_summ", 100500],
        ["m2_max", 3.1]
    ]
    """
    # Как и ручка для пинга, эта не помечена как internal намеряно.
    @no_scopes
    @no_auth
    def get(self, meta_connection, main_connection):
        data = app.stats_aggregator.get_data()
        try:
            json.dumps(data)
        except TypeError:
            log.warning(f'Got TypeError while dumping data: {data}')
            data = []
        return json_response(data)
