# encoding: UTF-8

from abc import ABCMeta
from abc import abstractmethod

from flask import current_app
from flask import jsonify
from flask import Response
from typing import Any
from typing import Callable
from typing import Dict
from typing import List
from typing import NoReturn
from typing import Optional
from werkzeug.exceptions import HTTPException
from werkzeug.routing import RequestRedirect

from intranet.yandex_directory.src.yandex_directory.common.datatools import unwrap_single_item
from intranet.yandex_directory.src.yandex_directory.common.exceptions import ImmediateReturn
from intranet.yandex_directory.src.yandex_directory.common.extensions import AbstractExtension
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import error_log


def is_method_modifying(method):
    """
    Функция проверки является ли HTTP-метод модифицирующим.
    """
    return method.upper() in {'POST', 'PUT', 'PATCH', 'DELETE'}


class TextMessage(object):
    """
    Класс для преставления текстовых сообщений. Содержит код и опциональное
    сообщение по умолчанию.
    """

    def __init__(self, code, default_message=None):
        # type: (str, Optional[str]) -> None
        self.code = code
        self.default_message = default_message

    def localize(self, i18n_localizer):
        # type: (Callable[[str, Optional[str]], str]) -> str
        """
        Локализует сообщение с использование локализатора.
        """
        return i18n_localizer(self.code, self.default_message)


class ErrorDataBuilder(object):
    """
    Билдер для формирования словарей с описанием ошибки согласно формату из
    тикета DIR-6394.
    """

    def __init__(self, code, default_messsage=None):
        # type: (str, Optional[str]) -> None
        self._message = TextMessage(code, default_messsage)
        self._desc_common_messages = []  # type: List[TextMessage]
        self._desc_fields_messages = {}  # type: Dict[str, List[TextMessage]]
        self._parameters = {}  # type: Dict[str, Any]

    def add_common_description(self, code, default_message=None):
        # type: (str, Optional[str]) -> ...
        message = TextMessage(code, default_message)
        self._desc_common_messages.append(message)
        return self

    def add_field_description(self, field, code, default_message=None):
        # type: (str, str, Optional[str]) -> ...
        message = TextMessage(code, default_message)
        self._desc_fields_messages.setdefault(field, []).append(message)
        return self

    def set_parameter(self, name, value):
        # type: (str, Any) -> ...
        if name in self._parameters:
            raise ValueError('Parameter %s already set.' % name)
        else:
            self._parameters[name] = value
            return self

    def add_parameter(self, name, value):
        # type: (str, Any) -> ...
        values = self._parameters.setdefault(name, [])
        if not isinstance(values, list):
            self._parameters[name] = values = [values]
        values.append(value)
        return self

    def build(self, i18n_localizer):
        # type: (Callable[[str, Optional[str]], str]) -> dict
        data = {
            'code': self._message.code,
            'message': self._message.localize(i18n_localizer),
            'description': {},
            'params': self._parameters.copy(),
        }

        if self._desc_common_messages:
            data['description']['common'] = unwrap_single_item([
                message.localize(i18n_localizer)
                for message in self._desc_common_messages
            ])

        if self._desc_fields_messages:
            data['description']['fields'] = {
                field: unwrap_single_item([
                    message.localize(i18n_localizer)
                    for message in messages
                ])
                for field, messages in list(self._desc_fields_messages.items())
            }

        return data


class ErrorResponseBuilder(ErrorDataBuilder):
    """
    Билдер для формирования ответов с описанием ошибки согласно формату из
    тикета DIR-6394.
    """

    def __init__(self, status_code, code, default_messsage=None):
        # type: (int, str, Optional[str]) -> None
        super(ErrorResponseBuilder, self).__init__(code, default_messsage)
        self._status_code = status_code
        self._headers = {}  # type: Dict[str, List[str]]

    def add_header(self, name, value):
        # type: (str, str) -> ...
        self._headers.setdefault(name, []).append(value)
        return self

    def build(self, i18n_localizer):
        # type: (Callable[[str, Optional[str]], str]) -> Response
        data = super(ErrorResponseBuilder, self).build(i18n_localizer)
        response = jsonify(data)  # type: Response
        response.status_code = self._status_code
        response.headers.extend(self._headers)
        return response

    def raise_as_error(self, i18n_localizer):
        # type: (Callable[[str, Optional[str]], str]) -> NoReturn
        response = self.build(i18n_localizer)
        raise HTTPException(response=response)


class ResponseMixin(object, metaclass=ABCMeta):
    """
    Примесь сообщающая для DefaultErrorHandlingExtension, что класс исключения
    может быть представлен в виде ответа на HTTP-запрос.
    """

    @abstractmethod
    def as_response(self, i18n_localizer):
        # type: (Callable[[str, Optional[str]], str]) -> Response
        """
        Метод возвращет ответ с описанием ошибки.
        """
        raise NotImplementedError


def handle_exception(exc):
    # TODO replace me with real i18n localizer
    i18n_localizer = lambda code, default_message: default_message or code

    if isinstance(exc, RequestRedirect):
        return exc
    elif isinstance(exc, HTTPException):
        return ErrorResponseBuilder(
            status_code=exc.code,
            code=exc.name.lower().replace(' ', '_'),
            default_messsage=str(exc),
        ).build(i18n_localizer)
    elif isinstance(exc, ImmediateReturn):
        return exc.response
    elif isinstance(exc, ResponseMixin):
        return exc.as_response(i18n_localizer)
    else:
        error_log.exception(
            'Request failed with unhandled exception.',
        )
        return ErrorResponseBuilder(
            status_code=500,
            code='unhandled_exception',
        ).build(i18n_localizer)


class DefaultErrorHandlingExtension(AbstractExtension):
    """
    Расширение устанавливающее обработчик исключений по умолчанию для запросов.

    В случае если происходит необработанное исключение оно попадает в этот
    оработчик, где формируется ответ об ошибке.

    Если исключение наследует класс от ``ResponseMixin``, то отправляется ответ
    сформированный методом ``as_response`` исключения. Иначе ошибка логируется
    и формируется стандартный ответ 500 с кодом ``unknown_error``.
    """

    def _setup_extension(self, app):
        app.register_error_handler(
            Exception,
            handle_exception,
        )
