# -*- coding: utf-8 -*-
import contextlib

from yt.wrapper import YtClient
from yt.wrapper.errors import YtProxyUnavailable

from retrying import retry
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory import app


def _get_yt_client(cluster, token):
    return YtClient(
        cluster,
        config={
            'token': token,
            # https://wiki.yandex-team.ru/yt/userdoc/pythonwrapper/config/
            'proxy': {
                'connect_timeout': app.config['YT_CONNECT_TIMEOUT'],
                'request_timeout': app.config['YT_REQUEST_TIMEOUT'],
            },
        },
    )


yt_client = None
billing_yt_clients = {}


def setup_yt(app):
    global yt_client
    yt_client = _get_yt_client(app.config['YT_CLUSTER'], app.config['YT_TOKEN'])

    for cluster in app.config['BILLING_YT_CLUSTERS']:
        billing_yt_clients[cluster] = _get_yt_client(cluster, app.config['BILLING_YT_TOKEN'])


def _retry_on_yt_error(exc):
    return isinstance(exc, YtProxyUnavailable)


@retry(stop_max_attempt_number=3, wait_incrementing_increment=50, retry_on_exception=_retry_on_yt_error)
def create_table_if_needed(table, schema, client=None):
    """
    Создаёт таблицу в YT если это необходимо.
    Args:
        table (str): название таблицы
        schema (list): схема таблицы (https://wiki.yandex-team.ru/yt/userdoc/staticschema/), например
            [
                {"name" : "a", "type" : "int64"},
                {"name" : "b", "type" : "string"}
            ]
        client (YtClient): опциональный инстанс YT клиента
    """
    if not client:
        client = yt_client

    if not is_table_exists(table, client=client):
        create_table(table=table, schema=schema, client=client)
        return True
    return False


@retry(stop_max_attempt_number=3, wait_incrementing_increment=50, retry_on_exception=_retry_on_yt_error)
def create_table(table, schema, client=None):
    """
    Создаёт таблицу в YT.
    Args:
        table (str): название таблицы
        schema (list): схема таблицы (https://wiki.yandex-team.ru/yt/userdoc/staticschema/), например
            [
                {"name" : "a", "type" : "int64"},
                {"name" : "b", "type" : "string"}
            ]
        client (YtClient): опциональный инстанс YT клиента
    """
    if not client:
        client = yt_client

    attributes = {
        'dynamic': False,
        'schema': schema,
    }
    with log.name_and_fields('yt_utils', table=table):
        log.debug('Creating YT table')

    client.create(
        'table',
        table,
        attributes=attributes,
        recursive=True,  # автоматически создаёт весь путь до таблицы, если необходимо
    )


@retry(stop_max_attempt_number=3, wait_incrementing_increment=50, retry_on_exception=_retry_on_yt_error)
def is_table_exists(table, client=None):
    if not client:
        client = yt_client

    with log.name_and_fields('yt_utils', table=table):
        log.debug('Checking if YT table exists')

    return client.exists(table)


def get_yt_clusters_without_table(table, check_empty_tables, yt_clients):
    """
    Возвращает список кластеров YT, в которых нет указанной таблицы

    Args:
        table (str) - путь к таблице
        check_empty_tables (bool) - проверять, пустые ли таблицы. Если пустые, считается что их нет
        yt_clients (list) - список клиентов yt
    """
    not_exists_in_clusters = []

    for yt in yt_clients:
        table_exists = is_table_exists(table, yt)

        # если нет таблицы или таблица есть, но пустая (при check_empty_tables == True).
        # добавляем её в список not_exisis_in_clusters
        if not table_exists or (table_exists and check_empty_tables and not yt.row_count(table)):
            not_exists_in_clusters.append(yt.config['proxy']['url'])

    return not_exists_in_clusters


@retry(stop_max_attempt_number=3, wait_incrementing_increment=50, retry_on_exception=_retry_on_yt_error)
def remove_table(table, client=None):
    if not client:
        client = yt_client

    with log.name_and_fields('yt_utils', table=table):
        log.debug('Removing YT table')

    return client.remove(table)


@retry(stop_max_attempt_number=3, wait_incrementing_increment=50, retry_on_exception=_retry_on_yt_error)
def append_rows_to_table(table, rows_data, client=None):
    """
    Добавляет строку данных в уже существующую таблицу
    https://wiki.yandex-team.ru/yt/userdoc/api/#writetable

    Args:
        table (str): название таблицы
        rows_data (list): список словарей "ключ-значение" с данными строк таблицы
        client (YtClient): опциональный инстанс YT клиента
    """
    if not rows_data:
        log.debug('Data to append to the YT table is empty, skipping')
        return

    if not client:
        client = yt_client

    with log.name_and_fields('yt_utils', table=table, data_length=len(rows_data)):
        log.debug('Appending data to the YT table')

    client.write_table(
        get_table_name_with_attributes(
            table=table,
            attributes=['append=%true'],  # дописываем данные в таблицу
        ),
        rows_data,
    )


def get_table_name_with_attributes(table, attributes):
    """
    Добавляет к пути таблицы необходимые атрибуты.
    В yt атрибуты добавляются в "< >" непосредственно перед путём и разделяются ";":
        '<param_1=true;param_2=true;>/tmp/my_table'
    """
    return '<%s>%s' % (';'.join(attributes), table)



@contextlib.contextmanager
def with_table(path, batch_size=1000, if_does_not_exists='create', if_exists='use', **schema):
    schema = [{'name': name, 'type': t} for name, t in list(schema.items())]
    exists = is_table_exists(path)
    if exists and if_exists != 'use':
        raise RuntimeError('Table {0} already exists'.format(path))
    if not exists:
        if if_does_not_exists == 'create':
            create_table_if_needed(path, schema)
        else:
            raise RuntimeError('Table {0} already does not exist'.format(path))

    batch = []
    
    def write_row(**kwargs):
        batch.append(kwargs)
        
        if len(batch) == batch_size:
            append_rows_to_table(table=path, rows_data=batch)
            batch[:] = []

    yield write_row

    # Допишем остаток данных в табличку, при выходе из контекстного менеджера
    if batch:
        append_rows_to_table(table=path, rows_data=batch)
