import urllib.parse

from flask import g
from retrying import retry

from intranet.yandex_directory.src import settings
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.connect_services.exceptions import ServiceInteractionException
from intranet.yandex_directory.src.yandex_directory.core.resource.metadata import MetaDataBase
from intranet.yandex_directory.src.yandex_directory.core.utils.retry import retry_http_errors
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common import http_client


class Aliceb2bInteractionException(ServiceInteractionException):
    def __init__(self, message='alice_b2b service interaction exception'):
        super(Aliceb2bInteractionException, self).__init__(message)


class Client(MetaDataBase):
    def __init__(self):
        self.host = settings.ALICE_B2B_HOST
        self.timeout = 0.5

    def get_metadata(self, *resources_ids):
        path = 'b2b/internal/connect/resources/'

        try:
            response = self._make_request(path, headers={
                'X-Org-Id': str(g.org_id),
            })
        except Aliceb2bInteractionException:
            return {}

        if 'metadata' not in response:
            return {}

        return response['metadata']

    @retry(stop_max_attempt_number=3, wait_exponential_multiplier=100, wait_exponential_max=2000,
           retry_on_exception=retry_http_errors('alice_b2b'))
    def _make_request(self, path, method='get', params=None, timeout=None, json=None, headers=None):
        timeout = timeout or self.timeout

        url = urllib.parse.urljoin(self.host, path)

        headers = headers or {}
        headers['X-Ya-Service-Ticket'] = tvm.tickets['alice_b2b']

        response = http_client.request(
            method,
            url,
            headers=headers,
            params=params,
            timeout=timeout,
            json=json,
        )

        if response.status_code != 200:
            with log.fields(url=url, params=params, response_status_code=response.status_code,
                            response_data=response.text):
                log.error('alice_b2b response with error')
            raise Aliceb2bInteractionException()

        return response.json()
