# https://bb.yandex-team.ru/projects/CLOUD/repos/cloud-go/browse/private-api/yandex/cloud/priv/resourcemanager/v1/cloud_service.proto
import ssl
from datetime import datetime
from flask import g

import grpc

from intranet.yandex_directory.src import settings
from yandex.cloud.priv.resourcemanager.v1 import cloud_service_pb2
from yandex.cloud.priv.resourcemanager.v1.cloud_service_pb2_grpc import CloudServiceStub
from yandex.cloud.priv.organizationmanager.v1 import organization_service_pb2
from yandex.cloud.priv.organizationmanager.v1 import organization_service_pb2_grpc
from yandex.cloud.priv.organizationmanager.v1 import user_service_pb2
from yandex.cloud.priv.organizationmanager.v1.user_service_pb2_grpc import UserServiceStub
from yandex.cloud.priv.organizationmanager.v1 import invite_service_pb2
from yandex.cloud.priv.organizationmanager.v1.invite_service_pb2_grpc import InviteServiceStub
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    AuthorizationError,
)
from intranet.yandex_directory.src.yandex_directory.core.exceptions import (
    OrganizationUnknown,
    CloudValidationError,
)
from intranet.yandex_directory.src.yandex_directory.connect_services.cloud.service_account import (
    get_service_token,
    get_service_token_by_uid,
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.http_requests_logger import grpc_logging
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.connect_services.cloud.service_account import get_cert


class GrpcCloudClient:
    def __init__(self):
        self._host = settings.CLOUD_ORG_HOST
        self._creds = None
        self._token = None
        self._last_token_update = None

    def _get_iam_token(self, uid=None):
        if uid:
            token = get_service_token_by_uid(uid)

        else:
            if not g.user:
                raise AuthorizationError(
                    'User is required for this operation'
                )
            token = g.user.iam_token
            if not token:
                token = get_service_token_by_uid(g.user.passport_uid)

        return 'Bearer ' + token

    def _get_creds(self):
        if not self._creds:
            host, port = self._host.split(':')
            self._creds = grpc.ssl_channel_credentials(get_cert(host=host, port=port))

        return self._creds

    def _get_token(self):
        now = datetime.now()
        if not self._token or (now - self._last_token_update).total_seconds() > 60 * 60:
            self._token = 'Bearer ' + get_service_token()
            self._last_token_update = now

        return self._token

    def update_organization(self, org_id, data, uid=None):
        token = self._get_iam_token(uid=uid)
        data['organization_id'] = str(org_id)

        with grpc.secure_channel(self._host, self._get_creds()) as channel,\
             grpc_logging(self._host, 'OrganizationServiceStub_Update'):
            stub = OrganizationServiceStub(channel)
            try:
                return stub.Update(
                    organization_service_pb2.UpdateOrganizationRequest(
                        **data
                    ),
                    metadata=(
                        ('authorization', token),
                    ),
                    timeout=app.config['CLOUD_ORG_TIMEOUT']
                )
            except ValueError as exc:
                message = (exc.args and exc.args[0]) or 'schema_validation_error'
                raise CloudValidationError(message=message)

    def get_organization(self, org_id, uid=None):
        token = self._get_iam_token(uid=uid)
        with grpc.secure_channel(self._host, self._get_creds()) as channel,\
             grpc_logging(self._host, 'OrganizationServiceStub_Get'):
            stub = organization_service_pb2_grpc.OrganizationServiceStub(channel)
            try:
                return stub.Get(
                    organization_service_pb2.GetOrganizationRequest(
                        organization_id=str(org_id),
                    ),
                    metadata=(
                        ('authorization', token),
                    ),
                    timeout=app.config['CLOUD_ORG_TIMEOUT']
                )
            except grpc.RpcError as e:
                status_code = e.code()
                if grpc.StatusCode.NOT_FOUND == status_code:
                    raise OrganizationUnknown()
                raise

    def list_organizations(self, uid=None, page_size=1000, page_token=None, filter=None,
                           cloud_subject=None, authorize_as=None):
        assert authorize_as in ('user', 'service')
        if authorize_as == 'user':
            token = self._get_iam_token(uid=uid)
        else:
            token = self._get_token()

        with grpc.secure_channel(self._host, self._get_creds()) as channel,\
             grpc_logging(self._host, 'OrganizationServiceStub_List'):
            stub = organization_service_pb2_grpc.OrganizationServiceStub(channel)
            return stub.List(
                organization_service_pb2.ListOrganizationsRequest(
                    page_size=page_size,
                    page_token=page_token,
                    filter=filter,
                    subject_id=cloud_subject,
                ),
                metadata=(
                    ('authorization', token),
                ),
                timeout=app.config['CLOUD_ORG_TIMEOUT']
            )

    def list_users(self, cloud_ids, page_size=1000, page_token=None):
        self._host = settings.CLOUD_HOST
        page_token = page_token or ''
        with grpc.secure_channel(self._host, self._get_creds()) as channel,\
             grpc_logging(self._host, 'CloudServiceStub_ListUsers'):
            stub = CloudServiceStub(channel)
            return stub.ListUsers(
                cloud_service_pb2.ListUsersRequest(
                    cloud_ids=cloud_ids,
                    page_size=page_size,
                    page_token=page_token
                ),
                metadata=(
                    ('authorization', self._get_token()),
                ),
                timeout=app.config['CLOUD_ORG_TIMEOUT']
            )

    def list_users_by_organization_id(self, cloud_org_id: str, page_size=1000, page_token=None):
        page_token = page_token or ''
        with grpc.secure_channel(self._host, self._get_creds()) as channel,\
             grpc_logging(self._host, 'UserServiceStub_ListMembers'):
            stub = UserServiceStub(channel)
            return stub.ListMembers(
                user_service_pb2.ListMembersRequest(
                    organization_id=cloud_org_id,
                    page_size=page_size,
                    page_token=page_token
                ),
                metadata=(
                    ('authorization', self._get_token()),
                ),
                timeout=app.config['CLOUD_ORG_TIMEOUT']
            )

    def add_ya_user(self, cloud_org_id, ya_login):
        # решили пока не использовать
        with grpc.secure_channel(self._host, self._get_creds()) as channel,\
             grpc_logging(self._host, 'InviteServiceStub_Create'):
            stub = InviteServiceStub(channel)
            return stub.Create(
                invite_service_pb2.InviteUsersRequest(
                    organization_id=cloud_org_id,
                    invites=[
                        invite_service_pb2.Invite(
                            yandex_passport_user_account=invite_service_pb2.Invite.YandexPassportUserAccount(
                                login=ya_login,
                            ),
                        ),
                    ],
                ),
                metadata=(
                    ('authorization', self._get_token()),
                ),
                timeout=app.config['CLOUD_ORG_TIMEOUT']
            )
