# coding: utf-8
import urllib.parse

from typing import List, Optional

from intranet.yandex_directory.src import settings
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.common.utils import log_service_response
from intranet.yandex_directory.src.yandex_directory.connect_services.direct.client.exceptions import DirectInteractionException
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common import http_client


class DirectClient(object):
    def __init__(self):
        self.host = settings.DIRECT_HOST

    def get_roles(self, resource_id):
        # type: (str or int) -> List[DirectRoleDto]

        params = {
            'resource_id': resource_id,
        }

        path = '/connect/idm/get-roles'

        # Пример ответа (https://st.yandex-team.ru/DIRECT-103770#5d8b8d1a29af1d001e2ce022)
        response = self._get(path=path, params=params)

        data = response.json()
        try:
            if data['code'] != 0:
                error = data['error']
                self._raise_exception(path, params, response, error)

            raw_roles = data['roles']

            return [
                DirectRoleDto(
                    org_id=int(raw_role['org_id']) if raw_role['org_id'] is not None else None,
                    resource_id=raw_role['fields']['resource_id'],
                    path=raw_role['path'],
                    uid=int(raw_role['id']),
                )
                for raw_role in raw_roles
                if raw_role['subject_type'] == 'user'
            ]

        except KeyError:
            self._raise_exception(path, params, response, 'Direct response parsing error')

    def _get(self, path, params=None):
        url = urllib.parse.urljoin(self.host, path)

        if not url.startswith('http'):
            url = urllib.parse.urljoin(self.host, url)

        headers = {
            'X-Ya-Service-Ticket': tvm.tickets['direct']
        }

        response = http_client.request(
            'get',
            url=url,
            headers=headers,
            params=params,
        )

        log_service_response('direct', 'GET', url, params, response)

        if response.status_code != 200:
            self._raise_exception(path, params, response)

        return response

    @staticmethod
    def _raise_exception(path, params, response, error_message=None):
        with log.fields(path=path, params=params, response_status_code=response.status_code,
                        response_data=response.text):
            log.error('Direct interaction error')

        raise DirectInteractionException(error_message)


class DirectRoleDto(object):
    def __init__(self, org_id, resource_id, path, uid):
        # type: (Optional[int], str, str, int) -> None
        self.org_id = org_id
        self.resource_id = resource_id
        self.path = path
        self.uid = uid

    def __eq__(self, other):
        return self.__dict__ == other.__dict__
