import urllib.parse
from typing import List, Any, Union, Optional

from intranet.yandex_directory.src import settings
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.utils import log_service_response
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common.exceptions import APIError, InvalidInputTypeError
from intranet.yandex_directory.src.yandex_directory.common.exceptions import NotFoundError
from intranet.yandex_directory.src.yandex_directory.common import http_client


class DomenatorException(APIError):
    status_code = 424
    code = 'domenator_error'
    message = 'Domenator api error acquired'
    description = 'Произошла ошибка при работе с API Domenator'


class Client:
    def __init__(self):
        self.host = settings.DOMENATOR_HOST
        self.timeout = 5

    @staticmethod
    def headers():
        return {
            'X-Ya-Service-Ticket': tvm.tickets['domenator'],
        }

    def create_token(self, admin_id, domain, pdd_version):
        response = self.make_request(
            path=f'/api/domain-token/{admin_id}/{domain}/{pdd_version}/',
            method='post',
            expected_statuses={201},
        )
        return response.json()['token']

    def sync_connect(self, org_id, admin_id):
        self.make_request(
            path=f'/api/domains/sync-connect',
            method='post',
            data={'org_id': org_id, 'admin_id': admin_id},
            expected_statuses={200},
        )

    def who_is(self, domain):
        response = self.make_request(
            path=f'api/domains/who-is/?domain={domain}',
            expected_statuses={200, 404},
        )
        if response.status_code == 200:
            response = response.json()
            return response['org_id'], response['object_id']
        if response.status_code == 404:
            raise NotFoundError()

    def delete_token(self, admin_id, domain, pdd_version):
        response = self.make_request(
            path=f'/api/domain-token/{admin_id}/{domain}/{pdd_version}/',
            method='delete',
            expected_statuses={204, 404},
        )
        if response.status_code == 404:
            raise NotFoundError()

    def get_token_info(self, token, pdd_version):
        response = self.make_request(
            path=f'api/domain-token/{token}/{pdd_version}/',
            expected_statuses={200, 404},
        )
        if response.status_code == 200:
            response = response.json()
            return response['uid'], response['domain']
        if response.status_code == 404:
            raise NotFoundError()

    def get_registrar(self, registrar_id):
        response = self.make_request(
            path=f'api/registrar/{registrar_id}/',
            expected_statuses={200, 404, 422},
        )
        if response.status_code == 404:
            raise NotFoundError()
        elif response.status_code == 422:
            raise InvalidInputTypeError()

        return response.json()

    def patch_registrar(self, registrar_id, data):
        response = self.make_request(
            method='patch',
            path=f'api/registrar/{registrar_id}/',
            expected_statuses={200, 404},
            data=data,
        )
        if response.status_code == 404:
            raise NotFoundError()

    def get_registrar_by_admin(self, admin_uid):
        response = self.make_request(
            path=f'api/registrar/uid/v2/{admin_uid}/',
            expected_statuses={200, 404},
        )
        if response.status_code == 404:
            raise NotFoundError()

        return response.json()

    def get_registrar_by_token(self, token):
        response = self.make_request(
            path=f'api/registrar/token/v2/{token}/',
            expected_statuses={200, 404},
        )
        if response.status_code == 404:
            raise NotFoundError()

        return response.json()

    def get_or_create_registrar_token(self, registrar_id):
        response = self.make_request(
            method='post',
            path=f'api/registrar/{registrar_id}/token/',
            expected_statuses={200, 404},
        )
        if response.status_code == 404:
            raise NotFoundError()

        return response.json()['token']

    def delete_registrar_token(self, registrar_id):
        response = self.make_request(
            method='delete',
            path=f'api/registrar/{registrar_id}/token/',
            expected_statuses={200, 404},
        )
        if response.status_code == 404:
            raise NotFoundError()

    def add_domain(self, org_id, domain, admin_uid):
        return self.make_request(
            method='post',
            path=f'api/domains/',
            data={
                'org_id': str(org_id),
                'domain': domain,
                'admin_uid': str(admin_uid),
            },
            expected_statuses={200},
        )

    def delete_domain(self, domain_name, org_id, admin_uid, author_id=None):
        self.make_request(
            method='delete',
            path=f'api/domains/{domain_name}',
            params={
                'org_id': str(org_id),
                'admin_uid': str(admin_uid),
                'author_id': str(author_id),
            },
            expected_statuses={200},
        )

    def get_domains(self, org_ids: List[Any], domain: str = None, fields: List[str] = None):
        params = {
            'org_ids': ','.join(map(str, org_ids)),
        }
        if domain:
            params['domain'] = domain
        if fields:
            params['fields'] = ','.join(fields)

        response = self.make_request(
            method='get',
            path='api/domains/',
            params=params,
            expected_statuses={200},
        )
        return response.json()

    def check_ownership(self, org_id: int, domain_name: str, verification_type: str):
        response = self.make_request(
            method='post',
            path=f'api/domains/{org_id}/{domain_name}/check-ownership/',
            data={
                'verification_type': verification_type.upper(),
            }
        )
        return response.json()

    def ownership_info(self, org_id: str, domain_name: str):
        response = self.make_request(
            method='get',
            path=f'api/domains/{org_id}/{domain_name}/ownership-info/',
        )
        return response.json()

    def add_owned_domain(self, org_id: int, admin_id: int, domain:str):
        response = self.make_request(
            method='post',
            path=f'api/domains/add-owned/',
            data={
                'org_id': str(org_id),
                'admin_id': str(admin_id),
                'domain': domain,
            }
        )
        return response.json()

    def delete_all_domains(self, org_id: int):
        response = self.make_request(
            method='delete',
            path=f'api/private/domains/delete-all/{org_id}',
        )
        return response.json()

    def private_get_domains(
        self,
        org_id: Optional[Union[int, List[int]]] = None,
        name: Optional[str] = None,
        master: Optional[bool] = None,
        owned: Optional[bool] = None
    ):
        params = {}
        if org_id:
            if isinstance(org_id, list):
                org_id = ','.join(map(str, org_id))
            params['org_id'] = org_id
        if name:
            params['name'] = name
        if master:
            params['master'] = master
        if owned:
            params['owned']= owned

        response = self.make_request(
            method='get',
            path='api/private/domains/',
            params=params,
            expected_statuses={200},
        )
        return response.json()

    def private_patch_domains(
        self,
        org_id: int,
        data: dict,
        name: Optional[str] = None,
        owned: Optional[bool] = None,
        master: Optional[bool] = None,
    ):
        params = {'org_id': org_id}
        if name is not None:
            params['name'] = name
        if owned is not None:
            params['owned'] = owned
        if master is not None:
            params['master'] = master

        return self.make_request(
            method='patch',
            path='api/private/domains/',
            params=params,
            data=data,
            expected_statuses={200},
        )

    def update_admin(self, org_id: int, new_admin_id: int):
        return self.make_request(
            method='post',
            path=f'api/domains/{org_id}/update-admin/',
            data={
                'admin_id': str(new_admin_id)
            },
            expected_statuses={200},
        )

    def make_request(self, path, method='get', params=None, headers=None, data=None, timeout=None, expected_statuses=None):
        expected_statuses = expected_statuses or {200, }
        url = urllib.parse.urljoin(self.host, path)
        default_headers = self.headers()
        if headers is not None:
            default_headers.update(headers)

        response = http_client.request(
            method,
            url=url,
            headers=default_headers,
            params=params,
            json=data,
            timeout=timeout or self.timeout,
        )

        log_service_response('domenator', method, url, params, response)

        if response.status_code not in expected_statuses:
            self._raise_exception(path, params, response)

        return response

    @staticmethod
    def _raise_exception(path, params, response):
        with log.fields(path=path, params=params, response_status_code=response.status_code,
                        response_data=response.text):
            log.error('Domenator interaction error')

        raise DomenatorException()
