# coding: utf-8
from abc import ABCMeta

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.core.idm import IDMB2BApiClient
from intranet.yandex_directory.src.yandex_directory.core.models.resource import (
    ResourceModel,
    ResourceRelationModel,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import only_attrs
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationServiceModel
from intranet.yandex_directory.src.yandex_directory.core.actions import action_resource_modify
from .exceptions import (
    IDMError,
    OperationNotInIdmError,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    APIError,
)

WORKING_WITH_IDM_SYSTEMS = {
    'direct',
}


class InvalidRoleIdError(APIError):
    code = 'forbidden_by_idm_for_role_id'
    status_code = 403
    message = 'This action forbidden for this role_id'
    description = 'Отзыв роли невозможен. Указанной роли нет или на это нет прав.'


class IdmRoleDeleteError(APIError):
    status_code = 424
    message = 'Error from idm api during delete relation'
    error_code = 'idm_delete_relation_error'
    description = 'При отзыве роли произошла ошибка.'


class ConnectService(object, metaclass=ABCMeta):
    """
    Базовый класс для работы с IDM, методы могут возвращать
    IDMError, наследники могут переопределять класс через
    external_error_class

    Так же наследники должны определить service_slug

    """
    _change_relations_roles = []
    external_error_class = IDMError
    responsible_relation = None
    replace_responsible_relation = None

    @property
    def service_slug(self):
        raise NotImplemented

    def remove_role(self):
        raise NotImplemented

    def info(self):
        raise NotImplemented

    def add_responsible_role(self, main_connection, relations, org_id, service_id):
        """
        Выдаем роль владельца текущему ответственному, так же убеждаемся, что
        роль соответствующая ответственному есть только у одного
        человека в переданном списке ролей
        """
        if self.service_slug in WORKING_WITH_IDM_SYSTEMS:
            return

        if not (self.responsible_relation and self.replace_responsible_relation):
            log.warning(
                'responsible_relation or replace_responsible_relation for {} are not set'.format(
                    self.service_slug
                )
            )
            return

        responsible_id = OrganizationServiceModel(main_connection).filter(
            org_id=org_id, service_id=service_id,
        ).one()['responsible_id']

        add_responsible_role = False
        for relation in relations:
            if relation['object_type'] == 'user' and relation['object_id'] == responsible_id:
                relation['name'] = self.responsible_relation
                add_responsible_role = True
            elif relation['name'] == self.responsible_relation:
                relation['name'] = self.replace_responsible_relation

        if not add_responsible_role:
            relations.append(
                {
                    'name': self.responsible_relation,
                    'object_type': 'user',
                    'object_id': responsible_id,
                }
            )

    @property
    def idm_system_provider(self):
        return IdmSystemProvider(system=self.service_slug)

    def request_roles(self, org_id, author_id, *roles):
        return self.idm_system_provider.request_roles(org_id, author_id, *roles)

    def unbind_resource(self, org_id, author_id, resource_id):
        organization_roles = self.idm_system_provider.get_roles(
            org_id=org_id,
            author_id=author_id,
            link_type='organization',
            resource_id=resource_id,
            role_type='active',
        )

        roles_ids = [
            role['id'] for role in
            organization_roles
        ]

        self.revoke_roles(org_id, author_id, None, *roles_ids)

    def revoke_roles(self, org_id, uid, lang, *roles):
        """Пример возвращаемого значения:
        {"errors": 0, "successes": 8}
        """
        return self.idm_system_provider.revoke_roles(
            org_id, uid, lang, *roles
        )

    def roles_by_resource_id(self, org_id, author_id, resource_id):
        return self.idm_system_provider.get_active_user_roles(org_id, author_id, resource_id)

    def has_change_permission_for_resource(self, main_connection, uid, resource_id, org_id):
        resource = ResourceModel(main_connection).filter(
            org_id=org_id,
            service=self.service_slug,
            external_id=str(resource_id),
        ).one()
        if resource:

            return ResourceRelationModel(main_connection).filter(
                org_id=org_id,
                user_id=uid,
                resource_id=resource['id'],
                name__in=self._change_relations_roles
            ).one()

    def get_resources(self, main_connection, org_id):
        return ResourceModel(main_connection).filter(
            org_id=org_id,
            service=self.service_slug,
        ).fields('external_id', 'id', 'service',)

    def change_responsible(self, main_connection, org_id, author_id, responsible_id):
        if not self.responsible_relation:
            log.warning('responsible_relation for {} is not set'.format(self.service_slug))
            return

        resources = self.get_resources(main_connection, org_id)
        resource_relations = ResourceRelationModel(main_connection).filter(
            org_id=org_id,
            resource_id__in=(resource['id'] for resource in resources),
            user_id__isnull=False,
            name=self.responsible_relation,
        ).fields('id').scalar()

        if resource_relations:
            ResourceRelationModel(main_connection).update(
                filter_data={'id__in': resource_relations},
                update_data={'user_id': responsible_id},
            )
            self.change_responsible_notify(main_connection, org_id, resources)

    def change_responsible_notify(self, main_connection, org_id, resources):
        for resource in resources:
            with log.fields(resource={
                'id': resource['external_id'],
                'service': {
                    'slug': self.service_slug,
                },
            }):
                action_resource_modify(
                    main_connection,
                    org_id=org_id,
                    author_id=None,
                    object_value=resource,
                    old_object=resource,
                )


class IdmSystemProvider(object):
    def __init__(self, system):
        if system not in WORKING_WITH_IDM_SYSTEMS:
            raise OperationNotInIdmError(system)
        self.system = system
        self.idm_client = app.idm_b2b_client  # type: IDMB2BApiClient

    def request_roles(self, org_id, author_id, *roles):
        with log.fields(org_id=org_id,
                        author_id=author_id,
                        system=self.system,
                        roles=roles):
            log.info('Request roles')

            if len(roles) == 1:
                role = roles[0]
                result = [
                    self.idm_client.request_role(
                        org_id=org_id,
                        author_id=author_id,
                        system=self.system,
                        path=role['path'],
                        resource_id=role['resource_id'],
                        uid=role['uid'],
                        user_type=role.get('user_type'),
                    )
                ]
            else:
                result = self.idm_client.batch_request_roles(
                    org_id=org_id,
                    author_id=author_id,
                    system=self.system,
                    roles=roles,
                )

            return [
                role['id']
                for role in result
            ]

    def revoke_roles(self, org_id, author_id, lang, *roles):
        """Пример возвращаемого значения:
        {"errors": 0, "successes": 8}
        """
        with log.fields(roles=roles, org_id=org_id, author_id=author_id):
            log.info('Revoke roles')
            current_roles = self.idm_client.get_roles(org_id, author_id, role_id=roles, lang=lang)
            organizations = set(only_attrs(current_roles, 'organization'))
            if len(organizations) != 1:
                raise InvalidRoleIdError
            if org_id not in organizations:
                raise InvalidRoleIdError
            info = self.idm_client.revoke_roles(org_id, author_id, roles=roles, lang=lang)
            if info['errors']:
                raise IdmRoleDeleteError
            return info

    def get_roles(self, org_id, author_id, path=None, resource_id=None, link_type=None, role_type=None,
                  uid=None, state=None):
        return self.idm_client.get_roles(
            org_id=org_id,
            author_id=author_id,
            system=self.system,
            path=path,
            resource_id=resource_id,
            link_type=link_type,
            role_type=role_type,
            uid=uid,
            state=state,
        )

    def get_active_user_roles(self, org_id, author_id, resource_id=None, uid=None):
        all_roles = self.get_roles(
            org_id=org_id,
            author_id=author_id,
            resource_id=resource_id,
            link_type='user',
            state=['awaiting', 'depriving', 'failed', 'granted'],
            uid=uid,
        )

        return [
            role
            for role in all_roles
            if not (role['is_public'] is False or role['node'] is False)
        ]
