# coding: utf-8
import urllib.parse
from intranet.yandex_directory.src import settings

from retrying import retry
from requests import exceptions
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.core.utils.retry import retry_http_errors
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.core.resource.metadata import MetaDataBase
from intranet.yandex_directory.src.yandex_directory.common import http_client
from .exceptions import MetrikaInteractionException


class Client(MetaDataBase):
    max_counters_per_request = 500
    metadata_fields = {'name', 'site', }
    timeout = 0.3

    def is_counters_has_monetization(self, *counters_ids):
        counters_info = self.get_counters_info(*counters_ids)

        return {
            counter_id: counters_info.get(counter_id, {}).get('has_monetization', False)
            for counter_id in counters_ids
        }

    def get_metadata(self, *counters_ids):
        try:
            counters_info = self.get_counters_info(*counters_ids)
        except (MetrikaInteractionException, exceptions.RequestException):
            return {}

        return {
            counter_id: self._filter_metadata(metadata)
            for counter_id, metadata in list(counters_info.items())
        }

    def _filter_metadata(self, metadata):
        metadata = super(Client, self)._filter_metadata(metadata)
        site = metadata.get('site')
        if site:
            metadata['description'] = site
        return metadata

    @retry(stop_max_attempt_number=3, wait_exponential_multiplier=100, wait_exponential_max=2000,
           retry_on_exception=retry_http_errors('metrika'))
    def get_counters_info(self, *counters_ids):
        return self._get_response(counters_ids, path='/connect/get_counters_data')

    def _get_response(self, counters_ids, path, parse_result_func=None):
        result = {}
        for chunk in self._split_resources_in_batches(counters_ids, self.max_counters_per_request):
            counters_ids_param = ','.join(chunk)

            response = self._get(path, {
                'ids': counters_ids_param,
            })

            data = response.json()
            if parse_result_func:
                data = parse_result_func(data, chunk)
            result.update(data)
        return result

    def get_existing_counters(self, *counters_ids):
        result = self._get_response(
            counters_ids, path='/connect/check_counters_active',
            parse_result_func=self._parse_check_counters_active
        )
        return {
            counter for counter, exists
            in result.items() if exists
        }

    def check_transfer_possible(self, counter_id, user_id):
        path = 'connect/check_transfer_possible'
        params = {
            'uid': user_id,
            'counter_id': counter_id,
        }
        response = self._get(path, params=params)

        response_json = response.json()
        has_permission = response_json.get('has_permission')
        if not has_permission:
            with log.fields(
                    response_text=response.text,
                    response_status_code=response.status_code, ):
                log.warning('User doesn\'t have access to resource')
        return has_permission

    def _parse_check_counters_active(self, data, counters):
        return dict(list(zip(counters, data['result'])))

    def _get(self, relative_path, params=None, timeout=None):
        timeout = timeout or self.timeout

        url = urllib.parse.urljoin(settings.METRIKA_HOST, relative_path)

        headers = {
            'X-Ya-Service-Ticket': tvm.tickets[settings.METRIKA_SERVICE_SLUG]
        }

        response = http_client.request(
            'get',
            url=url,
            headers=headers,
            params=params,
            timeout=timeout,
        )

        if response.status_code != 200:
            with log.fields(url=url, params=params, response_status_code=response.status_code,
                            response_data=response.text):
                log.error('Metrika response with error')

            raise MetrikaInteractionException()

        return response
