# coding: utf-8
from retrying import retry

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.core.utils.retry import retry_http_errors
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common.utils import url_join
from intranet.yandex_directory.src.yandex_directory.common.exceptions import APIError
from intranet.yandex_directory.src.yandex_directory.common import http_client


class PartnerDiskError(APIError):
    status_code = 424
    code = 'unknown_partner_disk_error'
    message = 'Error in partner Disk API'
    description = 'Неизвестная ошибка АПи партнерского диска.'


class PartnerDisksNotFoundError(PartnerDiskError):
    status_code = 404
    code = 'partner_disk_uid_not_found'
    message = 'UID not found in partner disk'
    description = 'UID не найден в базе партнерского диска'


class Client(object):

    def __init__(self):
        self.host = app.config['PARTNER_DISK_HOST']
        self.timeout = 0.5
        self.partner_id = 'yandex_directory'
        self.product_id = 'yandex_directory_1tb'
        self.common_prefix = '/v1/disk/partners/{}/services'.format(self.partner_id)

    @retry(stop_max_attempt_number=3, wait_exponential_multiplier=100, wait_exponential_max=2000,
           retry_on_exception=retry_http_errors('partner_disk'))
    def _make_request(self, uri, uid, method, timeout=None):
        timeout = timeout or self.timeout
        get_params = {'product_id': self.product_id} if method != 'delete' else None
        uri = self.common_prefix + uri if uri else self.common_prefix

        url = url_join(self.host, uri, query_params=get_params)

        headers = {
            'X-Ya-Service-Ticket': tvm.tickets['partner_disk'],
            'X-Uid': str(uid),
        }

        response = http_client.request(
            method,
            url=url,
            headers=headers,
            timeout=timeout,
        )

        with log.name_and_fields(
                'partner_disk_requests',
                url=url,
                uid=uid,
                response_status_code=response.status_code,
                response_data=response.text
        ):
            if response.status_code >= 400:
                log.error('Partner disk response with error')
                if response.status_code in (404, 401):
                    # 401 диск отвечает, если пользователя нет в паспорте
                    raise PartnerDisksNotFoundError()

                raise PartnerDiskError()
            else:
                log.info('Partner disk response')

        try:
            return response.json()
        except:
            return response.text

    def delete_disk_space(self, uid, service_id):
        uri = '/{}'.format(service_id)
        try:
            self._make_request(uri, uid, 'delete')
        except PartnerDisksNotFoundError:
            pass

    def add_disk_space(self, uid):
        """
        Ответ диска
        {u'product_id': u'intranet.yandex_directory.src.yandex_directory_1tb',
        u'service_id': u'1cae5083aff94e65456f3101280cb96b'}
        """
        # проверяем, вдруг место уже выдано
        user_space = self.get_disk_space(uid)
        if user_space:
            return user_space

        # ручка очень медленная
        return self._make_request(None, uid, 'post', timeout=60)

    def get_disk_space(self, uid):
        # проверяем, выдано ли место на диске
        response = self._make_request(None, uid, 'get')
        if response.get('partner') == self.partner_id and response.get('items'):
            return response['items'][0]
