# coding: utf-8
import urllib.parse
from intranet.yandex_directory.src import settings
from retrying import retry
from flask import g
from requests import exceptions

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.core.utils.retry import retry_http_errors
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.connect_services.exceptions import ServiceInteractionException
from intranet.yandex_directory.src.yandex_directory.core.resource.metadata import MetaDataBase
from intranet.yandex_directory.src.yandex_directory.common import http_client


class YandexSpravInteractionException(ServiceInteractionException):
    def __init__(self, message='YandexSprav service interaction exception'):
        super(YandexSpravInteractionException, self).__init__(message)


class Client(MetaDataBase):
    metadata_fields = {
        'names': {
            'path': ['value', 'value'],
            'field_name': 'name',
        },
        'urls': {
            'path': ['value'],
            'field_name': 'description',
        },
    }
    max_orgs_per_request = 50

    def __init__(self):
        self.host = settings.YANDEXSPRAV_HOST
        self.timeout = 0.5

    def check_permission(self, resource_id, responsible_id):
        path = '/v1/companies/{}'.format(resource_id)
        params = {'expand': 'grants'}
        response = self._make_request(path=path, params=params)
        return responsible_id in response.get('uids_with_modify_permission', [])

    def get_metadata(self, *resources_ids):
        try:
            orgs_info = self.get_orgs_info(*resources_ids)
        except (YandexSpravInteractionException, exceptions.RequestException):
            return {}
        return {
            str(org_data['id']): self._filter_metadata(org_data)
            for org_data in orgs_info
        }

    def _filter_metadata(self, metadata):
        metadata_result = {}

        for field_name,  field_param in list(self.metadata_fields.items()):
            field_metadata = metadata.get(field_name, [])

            for field_values in field_metadata:
                if field_values.get('type') == 'main':
                    result_value = None
                    for path in field_param['path']:
                        result_value = (result_value or field_values).get(path, {})
                    metadata_result[field_param['field_name']] = result_value or None
        return metadata_result

    def get_orgs_info(self, *resources_ids):
        result = []
        for resources_ids_batch in self._split_resources_in_batches(resources_ids, self.max_orgs_per_request):

            data = self._make_request(
                path='/v1/companies/search?limit={}'.format(len(resources_ids_batch)),
                method='post',
                json={
                    'expand': ['urls', 'names'],
                    'filter': {
                        'user_access_level': 'any',
                        'permalinks': resources_ids_batch,
                    },
                },
            )
            result.extend(data.get('companies', []))
        return result

    @retry(stop_max_attempt_number=3, wait_exponential_multiplier=100, wait_exponential_max=2000,
           retry_on_exception=retry_http_errors('yandexsprav'))
    def _make_request(self, path, method='get', params=None, timeout=None, json=None):
        timeout = timeout or self.timeout

        url = urllib.parse.urljoin(self.host, path)
        user_ticket = g.user_ticket
        if not user_ticket:
            raise YandexSpravInteractionException(message='User ticket is required')

        headers = {
            'X-Ya-Service-Ticket': tvm.tickets['yandexsprav'],
            'X-Ya-User-Ticket': user_ticket,
        }
        response = http_client.request(
            method,
            url=url,
            headers=headers,
            params=params,
            timeout=timeout,
            json=json,
        )

        if response.status_code != 200:
            with log.fields(url=url, params=params, response_status_code=response.status_code,
                            response_data=response.text):
                log.error('YandexSprav response with error')

            raise YandexSpravInteractionException()

        return response.json()
