# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory.core.events import (
    event_department_added,
    event_department_moved,
    event_department_property_changed,
    event_department_deleted,
    event_group_added,
    event_group_deleted,
    event_department_alias_added,
    event_department_alias_deleted,
)
from intranet.yandex_directory.src.yandex_directory.core.models import GroupModel
from intranet.yandex_directory.src.yandex_directory.core.events.utils import get_content_object
from intranet.yandex_directory.src.yandex_directory.common.models.types import (
    TYPE_DEPARTMENT,
    TYPE_GROUP,
)


def on_department_modify(connection,
                         org_id,
                         revision,
                         object_value,
                         object_type,
                         content,
                         **kwargs):
    """
    Обработчик действия: редактирование департамента
    Args:
        org_id (int): id организации
        revision (int): номер ревизии
        object_type (object): объект департамента после изменений
        content (object): {
            'before': объект до изменений,
            'subject': субъект, который вызвал изменения,
            'diff': список измененных свойств объекта ({
                'prop1': [prop1_before, prop1_after, ...],
                'prop2': [pop2_before, prop2_after, ...],
            })
    """
    content = {} if not content else content
    old_department = content.get('before', {})
    both = set(object_value.keys()) & set(old_department.keys())
    ignore_keys = set((
        # это поле синтетичесое и подтягивается как
        # prefetch related
        'parents',
        # поле path используется для внутренних нужд,
        # и не надо показывать его наружу
        'path',
    ))
    keys_to_compare = both - ignore_keys

    diff = {
        k: (old_department[k], object_value[k])
        for k in keys_to_compare
        if old_department[k] != object_value[k]
    }

    # смену родителя отрабатываем генерацией специального события
    # department_moved, в diff события property_changed,
    # ни 'parent', ни 'parent_id', попадать не должны

    if 'parent_id' in diff \
       or 'parent' in diff:

        if 'parent_id' in diff:
            moved_diff = {
                'parent_id': diff['parent_id']
            }
        else:
            moved_diff = {
                'parent_id': [
                    diff['parent'][0]['id'],
                    diff['parent'][1]['id'],
                ]
            }

        # делаем тут pop сразу на оба ключа, так как иногда
        # они могут дновременно в диффе оказаться
        diff.pop('parent_id', None)
        diff.pop('parent', None)

        moved_content = get_content_object(
            diff=moved_diff
        )
        event_department_moved(
            connection,
            org_id=org_id,
            revision=revision,
            object_value=object_value,
            object_type=TYPE_DEPARTMENT,
            content=moved_content,
        )

    if diff:
        content = get_content_object(diff=diff)
        event_department_property_changed(
            connection,
            org_id=org_id,
            revision=revision,
            object_value=object_value,
            object_type=TYPE_DEPARTMENT,
            content=content,
        )


def on_department_add(connection,
                      org_id,
                      revision,
                      department,
                      **kwargs):
    """
    Обработчик действия добавления департамента
    """
    # событие о создании технической группы с руководителями отдела
    head_group_info = GroupModel(connection).get_extend_info(department['heads_group_id'], org_id)
    event_group_added(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=head_group_info,
        object_type=TYPE_GROUP,
    )

    event_department_added(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=department,
        object_type=TYPE_DEPARTMENT,
    )


def on_department_delete(connection,
                         org_id,
                         revision,
                         department,
                         **kwargs):
    """
    Обработчик действия удаления департамента
    """
    if not department['heads_group_id']:
        raise RuntimeError('Field heads_group_id must exist')

    event_department_deleted(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=department,
        object_type=TYPE_DEPARTMENT,
        save_after_notification=True,
    )

    head_group_info = GroupModel(connection).get_extend_info(department['heads_group_id'], org_id)
    GroupModel(connection).delete(
        filter_data={
            'id': department['heads_group_id'],
            'org_id': department['org_id'],
        },
        generate_action=False,
    )
    event_group_deleted(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=head_group_info,
        object_type=TYPE_GROUP,
    )


def on_department_alias_add(connection,
                            org_id,
                            revision,
                            object_value,
                            object_type,
                            content,
                            **kwargs):
    event_department_alias_added(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=object_value,
        object_type=object_type,
        content=content
    )


def on_department_alias_delete(connection,
                               org_id,
                               revision,
                               object_value,
                               object_type,
                               content,
                               **kwargs):
    event_department_alias_deleted(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=object_value,
        object_type=object_type,
        content=content
    )
