# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory.core.models.department import DepartmentModel
from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel
from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationModel

from intranet.yandex_directory.src.yandex_directory.core.utils import only_ids
from intranet.yandex_directory.src.yandex_directory.core.events.utils import (
    get_content_object,
)
from intranet.yandex_directory.src.yandex_directory.common.models.types import (
    TYPE_USER,
    TYPE_GROUP,
)
from intranet.yandex_directory.src.yandex_directory.core.events import (
    event_user_added,
    event_user_moved,
    event_user_dismissed,
    event_user_property_changed,
    event_user_alias_added,
    event_user_alias_deleted,
    event_department_user_added,
    event_group_membership_changed,
)
from intranet.yandex_directory.src.yandex_directory.core.events.tasks import UpdateMembersCountTask
from intranet.yandex_directory.src.yandex_directory.core.events.resource import (
    event_resource_changed_for_each_resource,
)
from intranet.yandex_directory.src.yandex_directory.core.events.department import (
    _save_event_for_parents_departments,
)


def _get_add_remove_groups(groups_before, groups_after):
    old_set = set(only_ids(groups_before))
    new_set = set(only_ids(groups_after))
    to_add = new_set - old_set
    to_remove = old_set - new_set
    return {
        'add': list(to_add),
        'remove': list(to_remove),
    }


def on_user_add(connection, org_id, revision, user,  **kwargs):
    """
    Обработчик действия: добавление пользователя в департамент
    Args:
        org_id (int): id организации
        revision (int): номер ревизии
        user (object): объект пользователя
    """
    # 1. Генерируем событие user_added - пользователь добавлен в департамент (

    diff = {k: (None, user[k]) for k in set(user.keys())}
    content = get_content_object(subject=user, diff=diff)
    event_user_added(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=user,
        object_type=TYPE_USER,
        content=content,
    )
    # 2. Генерируем событие про изменение департамента - department_user_added
    # если пользователь не привязан к департаменту - то не генирируем события связанные с департаментом
    if not user.get('department'):
        return

    dep_id = user.get('department_id') or user.get('department', {}).get('id', 0)
    department = DepartmentModel(connection).get(dep_id, org_id, fields=['*', 'email'])
    diff = {k: (None, department[k]) for k in set(department.keys())}
    diff['members'] = {
        'remove': {},  # никого не удаляем
        'add': {
            'users': [user['id']],  # добавляем одного пользователя
        }
    }

    new_content = get_content_object(subject=user, diff=diff)
    _save_event_for_parents_departments(
        connection,
        department,
        revision,
        event_department_user_added,
        new_content
    )

    # 3. События об изменении всех связанных ресурсов
    event_resource_changed_for_each_resource(
        connection,
        org_id,
        revision,
        user,
        TYPE_USER,
        path=department['path']
    )

    # ВНИМАНИЕ! Этот код вызывает проблемы
    #           с заведением организации через landing на стрессе
    #           потому что в там обработчик запускается синхронно
    #           и не видит изменений в базе, сделанных в текущей транзакции
    if dep_id:
        # 4. Событие об изменении количества сотрудников в отделе
        UpdateMembersCountTask(connection).delay(
            department_ids=[dep_id],
            org_id=org_id,
            revision=revision,
        )
        # пересчитываем количество пользователей в организации
        OrganizationModel(connection).update_user_count(org_id=org_id)


def change_all_group_membership(connection,
                                user,
                                org_id,
                                revision,
                                groups_before,
                                groups_after):

    """Создаёт события group_membership_changed для всех
    команд, перечисленных в groups_to_add и groups_to_remove.
    """
    diff_groups = _get_add_remove_groups(groups_before, groups_after)

    # генерируем событие про изменение членов групп,
    # только если id-шники были разными (а не сами объекты groups)
    group_model = GroupModel(connection)

    for op_method, group_ids in list(diff_groups.items()):
        for group_id in group_ids:
            group = group_model.get(org_id=org_id, group_id=group_id)
            diff_members = {
                op_method: {
                    'users': [user['id']]
                }
            }
            group_members_content = get_content_object(
                diff=dict(members=diff_members)
            )
            event_group_membership_changed(
                connection,
                org_id=org_id,
                revision=revision,
                object_value=group,
                object_type=TYPE_GROUP,
                content=group_members_content,
            )


def on_user_modify(connection, org_id, revision, object_value, object_type, content, **kwargs):
    """
    Обработчик действия редактирования пользователя
    Args:
        org_id (int): id организации
        revision (int): номер ревизии
        user (object): объект после изменений
        content (object): по ключу before объект до изменений
    """

    user = object_value
    old_user = content.get('before', {})
    both = set(user.keys()) & set(old_user.keys())
    # Это не совсем корректный способ построения диффа,
    # потому что в случае, когда значение - список,
    # даже при наличии одинаковых элементов, они могут
    # посчитаться различными из-за разного порядка
    diff = {k: (old_user[k], user[k]) for k in both if old_user[k] != user[k]}

    if 'department_id' in diff:
        department_diff = {
            'department_id': diff['department_id']
        }
        content = get_content_object(
            diff=department_diff
        )
        event_user_moved(
            connection,
            org_id=org_id,
            revision=revision,
            object_value=user,
            object_type=object_type,
            content=content,
        )
        del diff['department_id']  # оставляем необработанные изменения
        if 'department' in list(diff.keys()):
            del diff['department']

    if 'groups' in diff:
        change_all_group_membership(
            connection,
            user,
            org_id,
            revision,
            diff['groups'][0],
            diff['groups'][1]
        )

        del diff['groups']

    if diff:
        content = get_content_object(diff=diff)
        event_user_property_changed(
            connection,
            org_id=org_id,
            revision=revision,
            object_value=user,
            object_type=object_type,
            content=content,
        )


def on_user_dismiss(connection,
                    org_id,
                    revision,
                    object_value,
                    object_type,
                    content,
                    **kwargs):
    """
    Обработчик действия увольнения пользователя
    Args:
        org_id (int): id организации
        revision (int): номер ревизии
        user (object): объект после изменений
        content (object): по ключу before объект до изменений
    """
    user = object_value
    old_user = content.get('before', {})

    # изменение самого пользователя и события для департаментов
    event_user_dismissed(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=user,
        object_type=object_type,
        content=content,
    )
    change_all_group_membership(
        connection,
        user,
        org_id,
        revision,
        old_user['groups'],
        [],
    )
    # пересчитываем количество пользователей в организации
    OrganizationModel(connection).update_user_count(org_id=org_id)


def on_user_alias_add(connection,
                       org_id,
                       revision,
                       object_value,
                       object_type,
                       content,
                       **kwargs):
    event_user_alias_added(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=object_value,
        object_type=object_type,
        content=content
    )


def on_user_alias_delete(connection,
                          org_id,
                          revision,
                          object_value,
                          object_type,
                          content,
                          **kwargs):
    event_user_alias_deleted(
        connection,
        org_id=org_id,
        revision=revision,
        object_value=object_value,
        object_type=object_type,
        content=content
    )
