import ssl

from datetime import datetime
import grpc
from intranet.yandex_directory.src import settings
from yandex.cloud.priv.oauth.v1 import claim_service_pb2_grpc, claim_service_pb2
from google.protobuf.json_format import MessageToDict
from typing import Iterable

from intranet.yandex_directory.src.yandex_directory.connect_services.cloud.service_account import get_service_token
from intranet.yandex_directory.src.yandex_directory.directory_logging.http_requests_logger import grpc_logging
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.connect_services.cloud.service_account import get_cert


class CloudClaimServiceClient:
    def __init__(self):
        self._host = settings.CLOUD_SESSION_SERVICE_HOST
        self._creds = None
        self._token = None
        self._last_token_update = None

    def _get_token(self):
        now = datetime.now()
        if not self._token or (now - self._last_token_update).total_seconds() > 60 * 60:
            self._token = 'Bearer ' + get_service_token()
            self._last_token_update = now

        return self._token

    def _get_creds(self):
        if not self._creds:
            host, port = self._host.split(':')
            self._creds = grpc.ssl_channel_credentials(get_cert(host=host, port=port))

        return self._creds

    def userinfo(self, *, yc_subject, userip=None):
        yc_subjects = yc_subject.split(',')
        claims = self.get_claims(yc_subjects)
        details = claims['subject_details']
        result = []
        for subject_dict in details:
            subject = subject_dict['subject_claims']
            uid = subject['yandex_claims']['passport_uid']
            name = subject.get('name', '')
            first = subject.get('given_name', '')
            last = subject.get('family_name', '')
            login = subject.get('preferred_username', '')
            email = subject.get('email', '')
            info = {
                'uid': uid,
                'name': name,
                'attributes': {
                    '193': subject['sub'],
                    '33': 'Europe/Moscow',
                    '34': 'ru',
                    '27': first,
                    '28': last,
                    '1008': login,
                },
                'claims': {
                    'given_name': first,
                    'family_name': last,
                    'name': name,
                    'preferred_username': login,
                    'email': email,
                },
                'raw_claims': subject,
                'dbfields': {
                    'accounts.login.uid': login,
                    'accounts.country.uid': 'ru',
                    'accounts.lang.uid': 'ru',
                }
            }
            result.append(info)
        return result

    def get_claims(self, subject_ids: Iterable):
        if not isinstance(subject_ids, list):
            subject_ids = [subject_ids]

        with grpc.secure_channel(self._host, self._get_creds()) as channel,\
             grpc_logging(self._host, 'ClaimServiceStub_Get'):
            stub = claim_service_pb2_grpc.ClaimServiceStub(channel)
            response = stub.Get(
                claim_service_pb2.GetClaimsRequest(
                    subject_ids=subject_ids,
                ),
                metadata=(
                    ('authorization', self._get_token()),
                ),
                timeout=app.config['CLOUD_SESSION_SERVICE_TIMEOUT']
            )
            result = MessageToDict(response, preserving_proto_field_name=True)
        return result
