from intranet.yandex_directory.src.yandex_directory.core.models import ServiceModel
from intranet.yandex_directory.src.yandex_directory.core.task_queue import Task
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
from intranet.yandex_directory.src.settings import SYNC_CLOUD_ORGS_TASK_CHUNK_SIZE
from intranet.yandex_directory.src.yandex_directory.core.utils import only_attrs
from intranet.yandex_directory.src.yandex_directory.core.cloud.utils import chunks
from intranet.yandex_directory.src.yandex_directory.core.models.service import (
    TRACKER_SERVICE_SLUG,
    WIKI_SERVICE_SLUG,
)
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_meta_connection,
)

from grpc._channel import _InactiveRpcError

SELECT_CLOUD_ORGS_FOR_SYNC_SQL = '''
SELECT distinct o.id from organizations o
inner join organization_services s on s.org_id = o.id
WHERE o.organization_type IN %(organization_types)s
and s.enabled is TRUE
and s.ready is TRUE
and s.service_id in %(services_ids)s
{filter_id_query}
order by o.id
'''


class SyncCloudOrgTask(Task):
    singleton = True
    default_priority = 25

    def do(self, org_id):
        with log.fields(org_id=org_id):
            log.info('Begin syncing cloud org')
            from intranet.yandex_directory.src.yandex_directory.core.cloud.utils import sync_cloud_org
            try:
                sync_cloud_org(org_id)
            except _InactiveRpcError:
                metadata = self.get_metadata() or {}
                if metadata.get('count_tries', 0) > 10:
                    raise
                else:
                    self.update_metadata(count_tries=metadata.get('count_tries', 0) + 1)
                    self.defer(countdown=60)
            log.info('End syncing cloud org')


class SendSyncCloudOrgTasksForBatch(Task):
    default_priority = 25
    org_id_is_required = False

    def do(self, from_id, to_id, services_ids):
        from intranet.yandex_directory.src.yandex_directory.core.models.organization import organization_type
        log.info(f'Begin sending sync cloud org tasks for batch from {from_id}, to {to_id}')
        org_ids = only_attrs(
            self.main_connection.execute(
                SELECT_CLOUD_ORGS_FOR_SYNC_SQL.format(
                    filter_id_query=f' and o.id >= {from_id}::bigint and o.id <= {to_id}::bigint'
                ),
                services_ids=tuple(services_ids),
                organization_types=tuple(organization_type.cloud_types),
            ).fetchall(),
            'id'
        )
        for org_id in org_ids:
            try:
                SyncCloudOrgTask(self.main_connection).delay(org_id=org_id)
            except DuplicatedTask:
                log.info(f'DuplicatedTask: SyncCloudOrgTask already in queue for {org_id}')

        log.info(f'End sending sync cloud org tasks for batch from {from_id}, to {to_id}')


class SyncCloudOrgsTask(Task):
    singleton = True
    org_id_is_required = False
    default_priority = 15
    lock_ttl = 1800

    def do(self):
        from intranet.yandex_directory.src.yandex_directory.core.models.organization import organization_type
        log.info('Begin syncing cloud orgs')

        with get_meta_connection() as meta_connection:
            services_ids = tuple(only_attrs(
                ServiceModel(meta_connection).find(
                    filter_data={'slug': [WIKI_SERVICE_SLUG, TRACKER_SERVICE_SLUG]},
                    fields=('id', )
                ),
                'id',
            ))

        org_ids = only_attrs(
            self.main_connection.execute(
                SELECT_CLOUD_ORGS_FOR_SYNC_SQL.format(filter_id_query=''),
                services_ids=services_ids,
                organization_types=tuple(organization_type.cloud_types),
            ).fetchall(),
            'id'
        )

        for chunk in chunks(org_ids, chunk_size=SYNC_CLOUD_ORGS_TASK_CHUNK_SIZE):
            SendSyncCloudOrgTasksForBatch(self.main_connection).delay(
                from_id=chunk[0],
                to_id=chunk[-1],
                services_ids=services_ids,
            )
