import collections
from typing import List, Tuple, Iterable

from google.protobuf.json_format import MessageToDict

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_meta_connection, get_main_connection,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import to_lowercase
from intranet.yandex_directory.src.yandex_directory.connect_services.cloud.grpc.client import GrpcCloudClient
from intranet.yandex_directory.src.yandex_directory.core.models import (
    OrganizationMetaModel,
    UserModel,
    UserMetaModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationRevisionCounterModel
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    get_user_info_from_blackbox,
    is_cloud_uid,
    is_domain_uid,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.users.base import (
    create_portal_user,
    UserNotFoundInBlackbox,
    UserAlreadyInThisOrganization
)
from intranet.yandex_directory.src.yandex_directory.core.utils.users.dismiss import dismiss_user
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log, default_log
from intranet.yandex_directory.src.blackbox_client import CLOUD_UID_ATTRIBUTE


def chunks(lst, chunk_size):
    for i in range(0, len(lst), chunk_size):
        yield lst[i:i + chunk_size]


def get_cloud_user_data(cloud_userinfo: dict) -> dict:
    first = cloud_userinfo['given_name']
    last = cloud_userinfo['family_name']
    if not first:
        first = cloud_userinfo['name']
        last = ''

    result = {
        'login': cloud_userinfo['preferred_username'],
        'email': cloud_userinfo['email'],
        'first': first,
        'last': last,
    }
    return result


def get_user_cloud_orgs(cloud_subject: str) -> List[str]:
    cloud_org_ids = []
    page_token = None
    while True:
        list_orgs_response = GrpcCloudClient().list_organizations(
            page_size=1000,
            page_token=page_token,
            cloud_subject=cloud_subject,
            authorize_as='service',
        )
        for org_dict in MessageToDict(list_orgs_response).get('organizations', []):
            cloud_org_ids.append(org_dict['id'])
        page_token = list_orgs_response.next_page_token
        if not page_token:
            break
    return cloud_org_ids


def create_or_dismiss_cloud_user(yc_subject: str, cloud_org_id: str) -> None:
    from intranet.yandex_directory.src.yandex_directory.auth.middlewares import User
    with get_meta_connection() as meta_connection:
        org_meta = OrganizationMetaModel(meta_connection).get(cloud_org_id=cloud_org_id)
        if not org_meta:
            with log.fields(cloud_org_id=cloud_org_id):
                log.warning('Nothing found by cloud_org_id')
                return
        shard = org_meta['shard']
        org_id = org_meta['id']
    with log.fields(org_id=org_id, cloud_org_id=cloud_org_id, yc_subject=yc_subject):
        with get_meta_connection(for_write=True) as meta_connection:
            with get_main_connection(for_write=True, shard=shard) as main_connection:
                log.info('Syncing state for user %s', yc_subject)
                cloud_org_ids = get_user_cloud_orgs(yc_subject)
                log.info('cloud orgs for the user were: %s', ','.join(cloud_org_ids))
                if cloud_org_id in cloud_org_ids:
                    log.info('Creating user via event')
                    user = User(cloud_uid=yc_subject, is_cloud=True)
                    create_cloud_user(meta_connection, user, cloud_org_id)
                else:
                    log.info('Dismissing user via event')
                    userinfo = app.cloud_blackbox_instance.userinfo(yc_subject=yc_subject)
                    if not isinstance(userinfo, list):
                        userinfo = [userinfo]
                    uid = userinfo[0]['uid']
                    should_dismiss = filter_uids_for_dismission([uid])
                    if should_dismiss:
                        log.info('Dismissing users %s', ','.join([str(uid) for uid in should_dismiss]))
                        dismiss_users(meta_connection, main_connection, org_id, should_dismiss)
                    else:
                        log.info('Skip dismissing user due to domain')


def filter_uids_for_dismission(uids_to_maybe_dismiss: Iterable[int]) -> List[int]:
    uids_to_keep = []
    uids_to_dismiss = []
    for uid in list(uids_to_maybe_dismiss):
        if is_domain_uid(uid):
            uids_to_keep.append(uid)
        else:
            uids_to_dismiss.append(uid)
    log.info('Keeping uids: %s', ','.join([str(uid) for uid in uids_to_keep]))
    return uids_to_dismiss


def sync_cloud_org(org_id: int) -> None:
    with log.fields(org_id=org_id):
        log.info('Start sync cloud org')
        with get_meta_connection() as meta_connection:
            org_meta = OrganizationMetaModel(meta_connection).get(org_id)
            shard = org_meta['shard']
            cloud_org_id = org_meta['cloud_org_id']

        if not cloud_org_id:
            log.info('no cloud_org_id, skip sync')
            return

        passport_uids_dict, yc_subjects_data = get_users_by_cloud_org_id(cloud_org_id)
        yc_subjects = yc_subjects_data.keys()

        for yc_subjects_chunk in chunks(list(yc_subjects), 50):
            user_info_list = app.cloud_blackbox_instance.userinfo(yc_subject=','.join(yc_subjects_chunk))
            if not isinstance(user_info_list, list):
                user_info_list = [user_info_list]

            for user_info in user_info_list:
                uid = int(user_info['uid'])
                cloud_uid = user_info['attributes'][CLOUD_UID_ATTRIBUTE]
                passport_uids_dict[uid] = cloud_uid
                if cloud_uid in yc_subjects_data:
                    yc_subjects_data[cloud_uid]['uid'] = uid

        if not passport_uids_dict:
            log.info('No cloud users found.')
            return

        with get_main_connection(shard) as main_connection:
            uids_in_org = set(UserModel(main_connection).filter(
                org_id=org_id,
                is_dismissed=False,
                is_robot=False,
            ).scalar('id'))

        uids_to_add = passport_uids_dict.keys() - uids_in_org
        uids_to_maybe_dismiss = uids_in_org - passport_uids_dict.keys()

        uids_to_dismiss = filter_uids_for_dismission(uids_to_maybe_dismiss)

        with get_meta_connection(for_write=True) as meta_connection:
            with get_main_connection(for_write=True, shard=shard) as main_connection:
                add_users(meta_connection, main_connection,
                          org_id, uids_to_add, passport_uids_dict, yc_subjects_data)
                dismiss_users(meta_connection, main_connection, org_id, uids_to_dismiss)
                org_users = list(UserModel(main_connection).filter(
                    org_id=org_id,
                    is_dismissed=False,
                    is_robot=False,
                ))
                org_meta_users = list(UserMetaModel(meta_connection).filter(
                    org_id=org_id,
                    is_dismissed=False,
                ))
                sync_user_data(meta_connection, main_connection,
                               org_id, org_users, org_meta_users, passport_uids_dict, yc_subjects_data)


def get_users_by_cloud_org_id(cloud_org_id: str) -> Tuple[dict, dict]:
    passport_uids = {}
    yc_subjects_data = {}

    page_token = None
    while True:
        list_users_response = GrpcCloudClient().list_users_by_organization_id(
            cloud_org_id=cloud_org_id,
            page_size=1000,
            page_token=page_token,
        )
        for cloud_user in list_users_response.users:
            passport_uid = cloud_user.subject_claims.yandex_claims.passport_uid
            if passport_uid:
                passport_uids[passport_uid] = cloud_user.subject_claims.sub
            elif cloud_user.subject_claims.sub and cloud_user.subject_claims.federation.id:
                first = cloud_user.subject_claims.given_name
                last = cloud_user.subject_claims.family_name
                if not first:
                    first = cloud_user.subject_claims.name
                    last = ''

                yc_subjects_data[cloud_user.subject_claims.sub] = {
                    'login': cloud_user.subject_claims.preferred_username,
                    'email': cloud_user.subject_claims.email,
                    'first': first,
                    'last': last,
                }

        page_token = list_users_response.next_page_token
        if not page_token:
            break

    return passport_uids, yc_subjects_data


def add_users(meta_connection, main_connection, org_id, uids, passport_uids_dict, yc_subjects_data):
    for uid in uids:
        with log.name_and_fields('Adding user', uid_to_add=uid):
            cloud_uid = passport_uids_dict[uid]
            cloud_user_data = yc_subjects_data.get(cloud_uid)
            try:
                create_portal_user(
                    meta_connection,
                    main_connection,
                    uid,
                    org_id,
                    cloud_uid=cloud_uid,
                    cloud_user_data=cloud_user_data,
                )
            except UserNotFoundInBlackbox:
                log.info('User not found in blackbox, skipping')
            except Exception:
                log.trace().error('Adding user failed')


def dismiss_users(meta_connection, main_connection, org_id, uids):
    for uid in uids:
        with log.name_and_fields('Dismissing user', uid_to_dismiss=uid):
            try:
                dismiss_user(
                    meta_connection,
                    main_connection,
                    org_id=org_id,
                    user_id=uid,
                    author_id=uid,
                    skip_disk=True,
                    skip_passport=True,
                )
            except Exception:
                log.trace().error('Dismissing user failed')


def sync_user_data(meta_connection,
                   main_connection,
                   org_id,
                   org_users,
                   org_meta_users,
                   passport_uids_dict,
                   yc_subjects_data):
    changed = False
    id_to_meta = {meta['id']: meta for meta in org_meta_users}

    for user in org_users:
        user_id = user['id']
        cloud_uid = passport_uids_dict.get(user_id)
        if cloud_uid is None:
            continue

        if yc_subjects_data.get(cloud_uid):
            cloud_user_data = yc_subjects_data[cloud_uid]
            login = cloud_user_data['login']
            email = cloud_user_data['email']
            first = cloud_user_data['first']
            last = cloud_user_data['last']
            gender = None
            birthday = None
        else:
            (login, first, last, gender, birthday, email, _) = get_user_info_from_blackbox(user_id)
            if not login:
                default_log.info('User %s deleted in blackbox' % user_id)
                continue

        changed_data = {}
        changed_meta_data = {}

        if is_cloud_uid(user_id):
            bb_info = {
                'nickname': to_lowercase(login),
                'name': {'first': first or login, 'last': last or ''},
                'email': to_lowercase(email),
                'gender': gender,
                'birthday': birthday,
            }
            for key, value in bb_info.items():
                if user[key] != value:
                    changed = True
                    changed_data[key] = value
        if cloud_uid != user['cloud_uid']:
            changed_data['cloud_uid'] = cloud_uid
        if cloud_uid != id_to_meta[user_id]['cloud_uid']:
            changed_meta_data = {'cloud_uid': cloud_uid}

        if changed_data:
            log.info(f'Updating info for {user_id}')
            UserModel(main_connection).update_one(
                filter_data={'id': user_id},
                update_data=changed_data,
            )
            if 'cloud_uid' in changed_meta_data:
                UserMetaModel(meta_connection).update(
                    filter_data={'id': user_id},
                    update_data={'cloud_uid': changed_meta_data['cloud_uid']}
                )

            from intranet.yandex_directory.src.yandex_directory.core.actions import action_user_modify
            action_user_modify(
                main_connection,
                org_id=org_id,
                author_id=user_id,
                object_value=user,
                old_object=user,
            )

    if changed:
        OrganizationRevisionCounterModel(main_connection).increment_revision(org_id)


def create_cloud_user(meta_connection, user, cloud_org_id=None) -> List[int]:
    cloud_subject = user.cloud_uid
    userip = user.ip

    fake_or_real_uid = None
    cloud_user_data = None

    if user.passport_uid is None:
        user_info_list = app.cloud_blackbox_instance.userinfo(
            yc_subject=cloud_subject,
            userip=userip,
        )
        if not isinstance(user_info_list, list):
            user_info_list = [user_info_list]
        fake_or_real_uid = int(user_info_list[0]['uid'])
        cloud_user_data = get_cloud_user_data(user_info_list[0]['claims'])
    else:
        fake_or_real_uid = user.passport_uid

    with log.fields(cloud_subject=cloud_subject, bbemu_uid=fake_or_real_uid, request_id=user.request_id):
        log.info('Start creating cloud user on the fly')

    # транслируем cloud_org_id в org_id
    # cоздаём пользователя в каждой из необходимых org_id

    if cloud_org_id:
        cloud_org_ids = [cloud_org_id]
    else:
        cloud_org_ids = get_user_cloud_orgs(cloud_subject)

    with log.fields(cloud_subject=cloud_subject, bbemu_uid=fake_or_real_uid, request_id=user.request_id):
        log.info('We got the list of cloud orgs from yorg: %s', ','.join(cloud_org_ids))
    if not cloud_org_ids:
        return []

    ids_and_shards = list(
        OrganizationMetaModel(meta_connection).filter(cloud_org_id__in=cloud_org_ids).fields('id', 'shard').all()
    )
    shard_to_ids = collections.defaultdict(list)
    for id_and_shard in ids_and_shards:
        shard_to_ids[id_and_shard['shard']].append(id_and_shard['id'])

    all_org_ids = []
    with get_meta_connection(for_write=True) as meta_for_write:
        for shard, org_ids in shard_to_ids.items():
            with get_main_connection(for_write=True, shard=shard) as main_connection:
                for org_id in org_ids:
                    with log.fields(org_id=org_id,
                                    uid=fake_or_real_uid,
                                    shard=shard,
                                    cloud_subject=cloud_subject,
                                    request_id=user.request_id):
                        log.info('Creating cloud user on the fly for organization %s', org_id)
                        try:
                            create_portal_user(
                                meta_connection=meta_for_write,
                                main_connection=main_connection,
                                uid=fake_or_real_uid,
                                org_id=org_id,
                                cloud_uid=cloud_subject,
                                cloud_user_data=cloud_user_data,
                            )
                        except UserAlreadyInThisOrganization:
                            all_org_ids.append(org_id)
                            log.info('User is already in the org for organization %s', org_id)
                        except Exception:
                            log.exception('Exception while creating user in the org %s', org_id)
                        else:
                            all_org_ids.append(org_id)
    return all_org_ids
