# -*- coding: utf-8 -*-
from intranet.yandex_directory.src.yandex_directory.core.models import (
    DepartmentModel,
    UserModel,
    OrganizationModel,
    OrganizationServiceModel,
    GroupModel,
)
from intranet.yandex_directory.src.yandex_directory.common.commands.base import (
    BaseCommand,
    Option,
)
from intranet.yandex_directory.src.yandex_directory.core.views.users import (
    UserListView,
    UserDetailView,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    only_ids,
    get_random_password,
    create_maillist,
)

from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_meta_connection,
)
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_department_add,
    action_group_modify,
)
from intranet.yandex_directory.src.yandex_directory.core.models.service import MAILLIST_SERVICE_SLUG

USER_MAP = {}
GROUP_MAP = {}
DEP_MAP = {1: 1}


class Command(BaseCommand):
    name = 'copy-org-schema'
    option_list = (
        Option('--from_org', dest='from_org', type=int),
        Option('--to_org', dest='to_org', type=int),
        Option('--shard', '-s', dest='shard', type=int, required=False)
    )

    def run(self, from_org, to_org, shard=1):
        with get_meta_connection(for_write=True) as meta_connection, \
                get_main_connection(for_write=True, shard=shard) as main_connection:
            from_organization = OrganizationModel(main_connection).find({'id': from_org})
            if not from_organization:
                return 'Не существует исходная организация: {}'.format(from_org)

            to_organization = OrganizationModel(main_connection).find({'id': to_org}, one=True)
            if not to_organization:
                return 'Не существует принимающая организация: {}'.format(to_org)
            admin_uid = to_organization['admin_uid']

            users_to_copy = UserModel(main_connection).find(
                filter_data={
                    'org_id': from_org,
                    'is_robot': False,
                },
                fields=['name', 'nickname', 'position', 'gender', 'about'],
            )

            print('Создаем пользователей')
            for u in users_to_copy:
                user_data = u.copy()
                del user_data['id']
                user_data['password'] = get_random_password()
                user_data['department_id'] = 1
                user_exists = UserModel(main_connection).find(
                    {
                        'org_id': to_org,
                        'nickname': u['nickname']
                    },
                    one=True
                )
                if u['nickname'] == 'admin':
                    USER_MAP[u['id']] = user_exists['id']
                    continue
                if user_exists:
                    new_user = user_exists
                else:
                    new_user = UserListView._post_user(
                        meta_connection,
                        main_connection,
                        to_org,
                        user_data,
                        None,
                        ignore_login_not_available=True,
                        user_type='user',
                        api_version=1,
                    )

                USER_MAP[u['id']] = new_user['id']

            dep_ids = [1]
            print('Создаем отделы')
            while dep_ids:
                dep_id = dep_ids.pop()
                departments_to_copy = DepartmentModel(main_connection).find(
                    filter_data={
                        'org_id': from_org,
                        'parent_id': dep_id
                    },
                    fields=['name', 'description', 'label', 'head'],
                )

                dep_ids.extend(only_ids(departments_to_copy))
                for d in departments_to_copy:
                    data = d.copy()
                    data['parent_id'] = DEP_MAP[dep_id]
                    if d.get('head'):
                        data['head_id'] = USER_MAP[d['head'].get('id')]
                    del data['head']
                    del data['id']
                    dep_exists = DepartmentModel(main_connection).find(
                        {
                            'org_id': to_org,
                            'label': d['label'],
                        },
                        one=True
                    )
                    if dep_exists:
                        new_dep = dep_exists
                    else:
                        new_dep = create_department(main_connection, to_org, data)
                    users_to_move = only_ids(UserModel(main_connection).find(
                        filter_data={
                            'org_id': from_org,
                            'department_id': d['id'],
                        },
                        fields=['id'],
                    ))
                    DEP_MAP[d['id']] = new_dep['id']

                    for u in users_to_move:
                        UserDetailView._patch_user(
                            meta_connection,
                            main_connection,
                            to_org,
                            {'department_id': new_dep['id']},
                            USER_MAP[u],
                            admin_uid,
                            None,
                            1,
                        )
            DepartmentModel(main_connection).update_members_count(list(DEP_MAP.values()), to_org)

            groups_to_copy = GroupModel(main_connection).find(
                filter_data={
                    'org_id': from_org,
                    'type': 'generic',
                },
                fields=['name', 'label', 'description', 'aliases', 'admins', 'members', 'members.type'],
            )

            print('Создаем группы')
            for g in groups_to_copy:
                group_data = g.copy()
                group_data['admins'] = [USER_MAP[admin['id']] for admin in g['admins']]
                del group_data['id']
                del group_data['members']
                group_exists = GroupModel(main_connection).find(
                    {
                        'org_id': to_org,
                        'label': g['label']
                    },
                    one=True
                )
                if group_exists:
                    new_group = group_exists
                else:
                    new_group = create_group(main_connection, to_org, group_data, admin_uid)
                GROUP_MAP[g['id']] = new_group['id']

            print('Заполняем группы')
            for g in groups_to_copy:
                data = {'members': get_members(g['members'])}
                GroupModel(main_connection).update_one(to_org, GROUP_MAP[g['id']], data)
                action_group_modify(
                    main_connection,
                    org_id=to_org,
                    author_id=admin_uid,
                    object_value={'id': GROUP_MAP[g['id']]},
                    old_object=g,
                )

            print('OK')


def create_department(main_connection, org_id, data):
    name = data.get('name')
    description = data.get('description')
    parent_id = data.get('parent_id')
    head_id = data.get('head_id')
    label = data.get('label')
    external_id = data.get('external_id')
    maillist_type = data.get('maillist_type')

    label = label.strip().lower()

    department_model = DepartmentModel(main_connection)

    uid = None
    if OrganizationServiceModel(main_connection).is_service_enabled(org_id, MAILLIST_SERVICE_SLUG):
        uid = create_maillist(main_connection, org_id, label, ignore_login_not_available=True)
    department = department_model.create(
        uid=uid,
        name=name,
        label=label,
        description=description,
        parent_id=parent_id,
        org_id=org_id,
        head_id=head_id,
        external_id=external_id,
        maillist_type=maillist_type,
    )

    action_department_add(
        main_connection,
        org_id=org_id,
        author_id=None,
        object_value=department,
    )
    return department


def create_group(main_connection, org_id, data, admin_uid):
    label = data.get('label')
    members = data.get('members', [])
    members.append(
        dict(
            type='user',
            id=admin_uid,
        )
    )

    uid = None
    if OrganizationServiceModel(main_connection).is_service_enabled(org_id, MAILLIST_SERVICE_SLUG):
        uid = create_maillist(main_connection, org_id, label, ignore_login_not_available=True)

    group_model = GroupModel(main_connection)
    new_group = group_model.create(
        name=data.get('name'),
        org_id=org_id,
        description=data.get('description'),
        members=members,
        label=label,
        author_id=admin_uid,
        admins=data.get('admins', []),
        aliases=data.get('aliases'),
        type=data.get('type', 'generic'),
        external_id=data.get('external_id'),
        action_author_id=admin_uid,
        maillist_type=data.get('maillist_type'),
        uid=uid,
    )
    return new_group


def get_obj_map(obj_type):
    if obj_type == 'user':
        return USER_MAP
    if obj_type == 'department':
        return DEP_MAP
    if obj_type == 'group':
        return GROUP_MAP


def get_members(old_members):
    members = []
    for obj in old_members:
        obj_type = obj['type']
        members.append({
            'type': obj_type,
            'id': get_obj_map(obj_type)[obj['object']['id']]
        })
    return members
